<?php

use App\Models\Nf;
use App\Models\User;
use App\Scopes\TenantScope;
use Carbon\Carbon;
use Detection\MobileDetect;

if (!function_exists('verificarFaturaVencida')) {
    function verificarFaturaVencida(User $user): bool
    {
        if (isset($user)) {
            $pessoa = $user->tenant->idpessoa;

            if (isset($pessoa)) {
                $nf = Nf::withoutGlobalScope(TenantScope::class)
                    ->where('idpessoa', $pessoa)
                    ->where('status', 'ATIVO')
                    ->orderBy('datavencimento')
                    ->first();

                if (isset($nf)) {
                    $diff = date_diff(new DateTime(), new DateTime($nf->datavencimento));
                    $daysToExpire = $diff->days;

                    if ($daysToExpire <= 5) {
                        return true;
                    }
                }
            }
        }

        return false;
    }
}

if (!function_exists('isMobile')) {
    function isMobile()
    {
        $detect = new MobileDetect();
        $isMobile = $detect->isMobile();
        return $isMobile;
    }
}

if (!function_exists('filter')) {
    function filter($request, $query, $qualificacao = null)
    {
        $filters = json_decode($request->input('organizedFilters'), true);

        if ($filters && is_array($filters)) {
            $query->where(function ($mainQuery) use ($filters, $qualificacao) {
                foreach ($filters as $groupKey => $groupDetails) {
                    $groupPrefix = $groupDetails['prefix'] ?? 'ONDE';

                    $mainQuery->{$groupPrefix === 'OU' ? 'orWhere' : 'where'}(function ($groupQuery) use ($groupDetails, $qualificacao) {
                        $filters = $groupDetails['filters'] ?? [];
                        $internalPrefix = 'ONDE';

                        foreach ($filters as $index => $filter) {
                            $campo = $filter['campo'];
                            $operador = $filter['operador'];
                            $valores = $filter['valores'];
                            $prefix = $filter['prefix'] ?? $internalPrefix;

                            $campoQualificado = $qualificacao != null ? $qualificacao . '.' . $campo : $campo;

                            $method = ($index === 0 || $prefix === 'ONDE') ? 'where' : ($prefix === 'E' ? 'where' : 'orWhere');

                            if (is_array($valores) && count($valores) > 1) {
                                switch ($operador) {
                                    case '=':
                                        $groupQuery->{$method . 'In'}($campoQualificado ?? $campo, $valores);
                                        break;
                                    case 'like':
                                        $groupQuery->where(function ($subQuery) use ($campoQualificado, $campo, $valores, $method) {
                                            foreach ($valores as $valor) {
                                                $subQuery->{$method}($campoQualificado ?? $campo, 'like', '%' . $valor . '%');
                                            }
                                        });
                                        break;
                                    case 'between':
                                        if (count($valores) === 2) {
                                            $groupQuery->{$method . 'Between'}($campoQualificado ?? $campo, $valores);
                                        }
                                        break;
                                    case 'IS NOT NULL OR != ""':
                                        $groupQuery->{$method . 'NotNull'}($campoQualificado ?? $campo);
                                        break;
                                    case 'IS NULL OR = ""':
                                        $groupQuery->{$method . 'Null'}($campoQualificado ?? $campo);
                                        break;
                                    default:
                                        foreach ($valores as $valor) {
                                            $groupQuery->{$method}($campoQualificado ?? $campo, $operador, $valor);
                                        }
                                }
                            } else {
                                $valor = $valores[0] ?? null;
                                switch ($operador) {
                                    case 'like':
                                        $groupQuery->$method($campoQualificado ?? $campo, 'like', '%' . $valor . '%');
                                        break;
                                    case 'between':
                                        if (strpos($valor, ',') !== false) {
                                            $dateRange = explode(',', $valor);
                                            if (count($dateRange) === 2) {
                                                $groupQuery->{$method . 'Between'}($campoQualificado ?? $campo, $dateRange);
                                            }
                                        }
                                        break;
                                    case '=':
                                    case '>=':
                                    case '<=':
                                    case '<':
                                    case '>':
                                        $groupQuery->$method($campoQualificado ?? $campo, $operador, $valor);
                                        break;
                                    case 'IS NOT NULL OR != ""':
                                        $groupQuery->{$method . 'NotNull'}($campoQualificado ?? $campo)
                                            ->orWhere($campoQualificado ?? $campo, '<>', '');
                                        break;
                                    case 'IS NULL OR = ""':
                                        $groupQuery->{$method . 'Null'}($campoQualificado ?? $campo)
                                            ->orWhere($campoQualificado ?? $campo, '=', '');
                                        break;
                                    default:
                                        if (strpos($valor, ',') !== false) {
                                            $dateRange = explode(',', $valor);
                                            if (count($dateRange) === 2) {
                                                $groupQuery->{$method . 'Between'}($campoQualificado ?? $campo, $dateRange);
                                            } else {
                                                $groupQuery->$method($campoQualificado ?? $campo, $operador, $valor);
                                            }
                                        } else {
                                            $groupQuery->$method($campoQualificado ?? $campo, $operador, $valor);
                                        }
                                        break;
                                }
                            }

                            if ($index === 1) {
                                $internalPrefix = $prefix;
                            }
                        }
                    });
                }
            });
        }
    }
}

if (!function_exists('getDatesFromFilter')) {
    function getDatesFromFilter($request)
    {
        $dataInicio = Carbon::now();
        $dataFim = Carbon::now();

        foreach (json_decode($request->organizedFilters) as $req) {
            foreach ($req->filters as $filter) {
                if ($filter->campo === 'dataentrada') {
                    foreach ($filter->valores as $value) {
                        $values = explode(',', $value);
                        $dataInicio = Carbon::createFromFormat('Y-m-d', trim($values[0]));
                        $dataFim = Carbon::createFromFormat('Y-m-d', trim($values[1]));
                    }
                }
            }
        }

        return [
            'dataInicio' => $dataInicio,
            'dataFim' => $dataFim,
        ];
    }
}

if (!function_exists('getUltimoKmAntesDe')) {
    function getUltimoKmAntesDe($idVeiculo, $dataInicio) {
        $ultimoRegistro = Nf::join('nfitem', 'nfitem.idnf', '=', 'nf.idnf')
            ->where('nf.idveiculo', $idVeiculo)
            ->where('nf.dataentrada', '<', $dataInicio)
            ->whereIn('nfitem.idprodserv', [20, 31])  // Correção aqui, usando whereIn corretamente
            ->orderByDesc('nf.dataentrada')
            ->first();

        return $ultimoRegistro ? $ultimoRegistro->km : 0;
    }
}

if (!function_exists('calcularMediaConsumoPorMes')) {
    function calcularMediaConsumoPorMes($request)
    {
        // Inicializa a coleção de meses
        $dadosMeses = collect();

        $dates = getDatesFromFilter($request);
        $dataInicio = $dates['dataInicio'];
        $dataFim = $dates['dataFim'];

        // Preenche os meses dentro do intervalo
        $dataInicioLoop = clone $dataInicio;
        while ($dataInicioLoop->lte($dataFim)) {
            $dadosMeses->push([
                'mes' => $dataInicioLoop->month,
                'ano' => $dataInicioLoop->year,
                'total' => 0
            ]);
            $dataInicioLoop->addMonth();
        }

        // Inicializa a consulta
        $query = Nf::pedido()
            ->selectRaw('MONTH(dataentrada) as mes, YEAR(dataentrada) as ano, COALESCE(SUM(valoritem), 0) as total')
            ->join('nfitem', 'nfitem.idnf', '=', 'nf.idnf')
            ->whereIn('idprodserv', [20, 31]);

        // Aplicar filtros
        filter($request, $query, 'nf');

        $dadosConsulta = $query->groupBy('mes', 'ano')->get();

        // Preenche os dados nos meses
        $dadosMeses->transform(function ($mes) use ($dadosConsulta) {
            foreach ($dadosConsulta as $dado) {
                if ($mes['mes'] == $dado->mes && $mes['ano'] == $dado->ano) {
                    $mes['total'] = $dado->total;
                    break;
                }
            }
            return $mes;
        });

        // Formatar os dados para o gráfico
        $graficoDados = [
            'labels' => $dadosMeses->map(fn($mes) => "{$mes['mes']}/{$mes['ano']}")->toArray(),
            'data' => $dadosMeses->map(fn($mes) => round($mes['total'], 2))->toArray() // Arredondar valores
        ];

        return $graficoDados;
    }
}

if (!function_exists('calcularMediaConsumo')) {
    function calcularMediaConsumo($request) {
        $query = Nf::join('nfitem', 'nfitem.idnf', '=', 'nf.idnf')
            ->join('veiculo', 'veiculo.idveiculo', '=', 'nf.idveiculo')
            ->whereIn('nfitem.idprodserv', [20, 31])
            ->where('nf.saida', 'P')
            ->where('nfitem.qtd', '>', 0);

        // Aplicar filtros
        filter($request, $query, 'nf');

        // Obter dados agrupados diretamente do banco
        $dadosAgrupados = $query->selectRaw('
            veiculo.idveiculo,
            MAX(nf.km) - MIN(nf.km) AS distanciaPercorrida,
            SUM(nfitem.qtd) AS combustivelConsumido
        ')
        ->groupBy('veiculo.idveiculo')
        ->get();

        // Calcular a média geral de consumo
        $mediasConsumo = array_filter($dadosAgrupados->map(function ($registro) {
            return $registro->combustivelConsumido > 0
                ? $registro->distanciaPercorrida / $registro->combustivelConsumido
                : 0;
        })->toArray());

        return !empty($mediasConsumo) ? array_sum($mediasConsumo) / count($mediasConsumo) : 0;
    }
}
