<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Agencia as Model;
use App\Models\Empresa;
use App\Models\GrupoItem;
use Carbon\Carbon;
use Exception;


class AgenciaController extends Controller
{
    public function index()
    {
        $agencias = Model::with('empresa')->orderBy('nome')
            ->paginate(100);

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
        ];

        $centroCustos = GrupoItem::all();

        return view('admin.agencias.index')->with(compact('agencias', 'statusOptions', 'centroCustos'));
    }

    public function fetch(Request $request)
    {
        $perPage = $request->input('per_page', 100);

        // Definir os campos e suas classes CSS
        $fields = [
            'ID' => [
                'key' => 'idagencia',
                'class' => 'text-center text-xs',
            ],
            'AGÊNCIA' => [
                'key' => 'nome',
                'class' => 'text-center text-xs',
            ],
            'TELEFONE' => [
                'key' => 'telefone',
                'class' => 'text-center text-xs',
            ],
            'CRIADO EM' => [
                'key' => fn($agencia) => Carbon::parse($agencia->criadoem)->format('d/m/y'),
                'class' => 'text-center text-xs fw-bold',
            ],
            'STATUS' => [
                'key' => fn($agenda) => view('partials.status-switch', [
                    'status' => $agenda->status,
                    'id' => $agenda->idagenda,
                ])->render(),
                'class' => 'no-clicked d-flex align-items-center justify-content-center',
            ]
        ];

        // Buscar os dados paginados
        $agencias = Model::with('empresa')->orderBy('nome')
            ->paginate($perPage, ['*'], 'page', $request->input('page', 2));

        // Montar os dados processados para o frontend
        $processedData = $agencias->map(function ($agencia) use ($fields) {
            $row = [
                'url' => route('admin.agencias.edit', $agencia->idagencia),
            ];
            foreach ($fields as $label => $field) {
                $value = is_callable($field['key'])
                    ? $field['key']($agencia) // Executar a closure
                    : data_get($agencia, $field['key']); // Obter o valor diretamente
                $row[$label] = [
                    'value' => $value,
                    'class' => $field['class'],
                ];
            }
            return $row;
        });

        // Retornar os dados processados como JSON
        return response()->json([
            'columns' => array_keys($fields),
            'data' => $processedData,
            'next_page_url' => $agencias->nextPageUrl(),
        ]);
    }

    public function create(Empresa $empresaModel)
    {
        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.agencias.create')->with(compact('empresas', 'statusOptions'));
    }

    public function store(Request $request)
    {
        $agenciaCreate = $request->all();

        // Campos monetários
        $agenciaCreate['saldoinicial'] = str_replace(['.', ','], ['', '.'], $agenciaCreate['saldoinicial']);

        // Agencia padrao
        $agenciaCreate['padrao'] = isset($agenciaCreate['padrao']) && $agenciaCreate['padrao'] == 'on' ? true : false;

        try {
            $agencia = Model::create($agenciaCreate);

            $agencia->incrementarSaldo($agencia->saldoinicial ?? 0);

            return redirect()
                ->route('admin.agencias.edit', $agencia->idagencia)
                ->with('success', 'Agência criada com sucesso!');
        } catch (Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->with('error', $e->getMessage());
        }
    }

    public function edit($idAgencia, Empresa $empresaModel)
    {
        $agencia = Model::find($idAgencia);
        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
        ];

        if ($agencia->status == 'ATIVO'){
            $bg_status = "bg-success";
        }elseif($agencia->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        return view('admin.agencias.edit')->with(compact('agencia', 'empresas', 'statusOptions', 'bg_status'));
    }

    public function update($idAgencia, Request $request)
    {
        // Convertendo para booleano
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;

        $agencia = Model::find($idAgencia);
        $agenciaUpdate = $request->all();

        if ($redirect == true) {
            // Campos monetários
            $agenciaUpdate['saldoinicial'] = str_replace(['.', ','], ['', '.'], $agenciaUpdate['saldoinicial']);

            // Agencia padrao
            $agenciaCreate['padrao'] = isset($agenciaCreate['padrao']) && $agenciaCreate['padrao'] == 'on' ? true : false;
        }

        try {
            $agencia->update($agenciaUpdate);

            if ($redirect == false) {
                return response()->json($agencia->status);
            }

            return redirect()
                ->route('admin.agencias.edit', $agencia->idagencia)
                ->with('success', 'Agência atualizada com sucesso!');
        } catch (Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->with('error', $e->getMessage());
        }
    }


    public function destroy($idAgencia)
    {
        $agencia = Model::find($idAgencia);

        try {
            $agencia->delete();

            return redirect()
                ->route('admin.agencias.index')
                ->with('success', 'Agência removida!');
        } catch (Exception $e) {
            return redirect()
                ->back()
                ->with('error', $e->getMessage())
                ->withInput();
        }
    }

    public function search(Request $request)
    {
        $agencias =  Model::with('empresa')->where('agencia', 'like', '%'.$request->search.'%')
                        ->where('status', 'like', '%'.$request->status.'%')
                        ->orderBy('nome')
                        ->paginate(15);

        $size = $agencias->count();

        // Colocando mensagens na session
        session()->forget('error');
        session()->forget('success');
        if ($size && $size > 0 )
        {
            session()->flash('success', $size.' valor(es) encontrado(s).');

        } else
        {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.agencias.index')
                ->with(compact('agencias', 'statusOptions'));
    }
}
