<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Agencia;
use App\Models\Empresa;
use App\Models\Nf;
use App\Models\NfParcela;
use Illuminate\Http\Request;

class DashboardMovimentacaoController extends Controller
{
    public function index(Empresa $empresaModel, Agencia $agenciaModel)
    {
        // Paginação da lista principal
        $listaParcelas = [];

        $valorTotalGrupoItem = [];

        // Calcular saldo e totais
        $saldoAtual = 0;
        $totalCredito = 0;
        $totalDebito = 0;
        $totalLucro = 0;
        $saldoInicial = 0;

        // Carregar empresas e agências para os selects
        $empresas = $empresaModel->orderBy('razaosocial')->get();
        $agencias = $agenciaModel->orderBy('nome')->get();

        // Retornar para a view com os dados necessários
        return view('dashboard.movimentacao',
            compact(
                'listaParcelas',
                'saldoAtual',
                'totalCredito',
                'totalDebito',
                'totalLucro',
                'saldoInicial',
                'valorTotalGrupoItem',
                'empresas',
                'agencias'
            ));
    }

    public function search(Request $request, Empresa $empresaModel, Agencia $agenciaModel)
    {
        $query = NfParcela::with(['nf.pessoa', 'nf.fornecedor', 'nf.cliente'])
            ->join('nf', 'nfparcela.idnf', '=', 'nf.idnf')
            ->join('nfitem', 'nf.idnf', '=', 'nfitem.idnf')
            ->leftJoin('grupoitem', 'nfitem.idgrupoitem', '=', 'grupoitem.idgrupoitem')
            ->where('nf.saida', '<>', 'P')
            ->where('nfparcela.status', 'CONCLUÍDO');

        filter($request, $query, 'nfparcela');

        $listaParcelas = $query->orderBy('nfparcela.datatransacao', 'asc')->get();

        // Calcular valor total por grupo de item
        $queryGrupoItem = Nf::join('nfparcela', 'nf.idnf', '=', 'nfparcela.idnf')
            ->join('nfitem', 'nf.idnf', '=', 'nfitem.idnf')
            ->leftJoin('grupoitem', 'nfitem.idgrupoitem', '=', 'grupoitem.idgrupoitem')
            ->where('nf.saida', '<>', 'P')
            ->where('nfparcela.status', 'CONCLUIDO');

        filter($request, $queryGrupoItem, 'nfparcela');

        $resultados = $queryGrupoItem->groupBy('grupoitem.idgrupoitem')
            ->selectRaw('grupoitem.idgrupoitem, COALESCE(grupoitem.grupoitem, "Indefinido") AS rotulo, SUM((nfitem.valoritem / nf.totalnf) * nfparcela.valor) AS valoritem')
            ->orderBy('valoritem', 'DESC')
            ->get();

        $custosTotaisPorGrupoItem = [];
        foreach ($resultados as $registro) {
            $custosTotaisPorGrupoItem[] = [
                'id' => $registro->idgrupoitem,
                'rotulo' => $registro->rotulo,
                'valor' => $registro->valoritem
            ];
        }

        $valorTotalGrupoItem = $custosTotaisPorGrupoItem;

        // Calcular saldos e totais
        $saldoAtual = $this->calcularSaldoAtual($listaParcelas);
        $totalCredito = $this->calculateValorTotal($listaParcelas, 'C');
        $totalDebito = $this->calculateValorTotal($listaParcelas, 'D');
        $totalLucro = $this->calculateLucro($listaParcelas);
        $saldoInicial = $listaParcelas->isNotEmpty() ? $listaParcelas->first()->saldoinicial : 0;

        // Carregar empresas e agências para os selects
        $empresas = $empresaModel->orderBy('razaosocial')->get();
        $agencias = $agenciaModel->orderBy('nome')->get();

        // Retornar para a view com os dados necessários
        return view('dashboard.movimentacao',
            compact(
                'listaParcelas',
                'saldoAtual',
                'totalCredito',
                'totalDebito',
                'totalLucro',
                'saldoInicial',
                'valorTotalGrupoItem',
                'empresas',
                'agencias'
            ));
    }

    private function calcularSaldoAtual($parcelas)
    {
        $saldo = 0;
        foreach ($parcelas as $parcela) {
            if ($parcela->tipo == 'C') {
                $saldo += $parcela->valor;
            } elseif ($parcela->tipo == 'D') {
                $saldo -= $parcela->valor;
            }
        }
        return $saldo;
    }

    private function calculateValorTotal($parcelas, $tipo)
    {
        $total = 0;
        foreach ($parcelas as $parcela) {
            if ($parcela->tipo == $tipo) {
                $total += $parcela->valor;
            }
        }
        return $total;
    }

    private function calculateLucro($parcelas)
    {
        return $this->calculateValorTotal($parcelas, 'C') - $this->calculateValorTotal($parcelas, 'D');
    }
}
