<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ContratoVeiculo;
use App\Models\Empresa;
use App\Models\Nf;
use App\Models\Veiculo;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardVeiculoController extends Controller
{
    public function index(Empresa $empresaModel, ContratoVeiculo $contratoVeiculoModel, Veiculo $veiculoModel)
    {
        $pedidosDieselMesAtual = [];

        // Números
        $totalPedidosDieselMesAtual = 0;
        $percentualPedidosDieselMudanca = 0;
        $totalPedidosDieselMesPassado = 0;
        $totalPedidosQtdDieselMesAtual = 0;
        $percentualPedidosQtdDieselMudanca = 0;
        $totalPedidosQtdDieselMesPassado = 0;
        $mediaConsumoMesAtual = 0;
        $graficoPedidosDieselPorMes = [];
        $mediasConsumoPorVeiculo = [];
        $listaAbastecimento = [];

        $empresas = $empresaModel->orderBy('razaosocial')->get();
        $veiculos = $veiculoModel->orderBy('placa')->get();
        $contratos = $contratoVeiculoModel->select(['contratoveiculo.*', 'empresa.sigla'])
            ->with(['contrato', 'empresa'])
            ->leftJoin('empresa', 'empresa.idempresa', '=', 'contratoveiculo.idempresa')
            ->orderBy('criadoem', 'desc')
            ->get();

        return view('dashboard.veiculo',
            compact(
                'pedidosDieselMesAtual',
                'totalPedidosDieselMesAtual',
                'percentualPedidosDieselMudanca',
                'totalPedidosDieselMesPassado',
                'totalPedidosQtdDieselMesAtual',
                'percentualPedidosQtdDieselMudanca',
                'totalPedidosQtdDieselMesPassado',
                'mediaConsumoMesAtual',
                'graficoPedidosDieselPorMes',
                'mediasConsumoPorVeiculo',
                'listaAbastecimento',
                'empresas',
                'contratos',
                'veiculos',
            ));
    }

    public function search(Request $request,Empresa $empresaModel, ContratoVeiculo $contratoVeiculoModel, Veiculo $veiculoModel)
    {
        $query = Nf::pedido()
            ->join('nfitem', 'nfitem.idnf', '=', 'nf.idnf')
            ->whereIn('idprodserv', [20, 31]);

        filter($request, $query, 'nf');

        $pedidosDieselMesAtual = $query->get();

        // Números
        $totalPedidosDieselMesAtual = $pedidosDieselMesAtual->sum('valoritem');
        $percentualPedidosDieselMudanca = 0;
        $totalPedidosDieselMesPassado = 0;
        $totalPedidosQtdDieselMesAtual = $pedidosDieselMesAtual->sum('qtd');
        $percentualPedidosQtdDieselMudanca = 0;
        $totalPedidosQtdDieselMesPassado = 0;
        $mediaConsumoMesAtual = calcularMediaConsumo($request);
        $graficoPedidosDieselPorMes = calcularMediaConsumoPorMes($request) ?? [];
        $mediasConsumoPorVeiculo = $this->calcularMediaConsumoPorVeiculo($request) ?? [];
        $listaAbastecimento = $this->getListaAbastecimento($request) ?? [];

        $empresas = $empresaModel->orderBy('razaosocial')->get();
        $veiculos = $veiculoModel->orderBy('placa')->get();
        $contratos = $contratoVeiculoModel->with('contrato', 'empresa')
            ->orderBy('criadoem', 'desc')
            ->get();

        return view('dashboard.veiculo',
            compact(
                'pedidosDieselMesAtual',
                'totalPedidosDieselMesAtual',
                'percentualPedidosDieselMudanca',
                'totalPedidosDieselMesPassado',
                'totalPedidosQtdDieselMesAtual',
                'percentualPedidosQtdDieselMudanca',
                'totalPedidosQtdDieselMesPassado',
                'mediaConsumoMesAtual',
                'graficoPedidosDieselPorMes',
                'mediasConsumoPorVeiculo',
                'listaAbastecimento',
                'empresas',
                'contratos',
                'veiculos',
            ));
    }

    private function calcularMediaConsumoPorVeiculo($request)
    {
        $query = Nf::select([
            'nf.idnf',
            'nf.km',
            'nf.idveiculo',
            'nfitem.qtd',
            'veiculo.placa as rotulo'
        ])
        ->join('nfitem', 'nfitem.idnf', '=', 'nf.idnf')
        ->join('veiculo', 'veiculo.idveiculo', '=', 'nf.idveiculo')
        ->whereIn('nfitem.idprodserv', [20, 31])
        ->where('nf.saida', 'P')
        ->where('nfitem.qtd', '>', 0);

        // Aplicar filtros dinâmicos
        filter($request, $query, 'nf');

        // Obter dados agrupados diretamente do banco
        $resultados = $query->orderBy('nf.idveiculo')
            ->orderBy('nf.dataentrada')
            ->get();

        // Estruturas para armazenar as informações por veículo
        $informacoesPorVeiculo = [];
        foreach ($resultados as $registro) {
            $idVeiculo = $registro->idveiculo;
            $kmAtual = $registro->km;
            $litros = $registro->qtd;
            $rotulo = $registro->rotulo; // Placa do veículo

            if (!isset($informacoesPorVeiculo[$idVeiculo])) {
                $informacoesPorVeiculo[$idVeiculo] = [
                    'kmAnterior' => $kmAtual,
                    'distanciaTotal' => 0,
                    'litrosTotal' => 0,
                    'rotulo' => $rotulo // Inclui o rótulo aqui
                ];
            } else {
                $distancia = $kmAtual - $informacoesPorVeiculo[$idVeiculo]['kmAnterior'];
                $informacoesPorVeiculo[$idVeiculo]['distanciaTotal'] += max($distancia, 0);
                $informacoesPorVeiculo[$idVeiculo]['kmAnterior'] = $kmAtual;
            }
            $informacoesPorVeiculo[$idVeiculo]['litrosTotal'] += $litros;
        }

        // Preparando dados para ordenação e exibição
        $mediasConsumoPorVeiculo = [];
        foreach ($informacoesPorVeiculo as $idVeiculo => $info) {
            $mediaConsumo = $info['litrosTotal'] > 0 ? $info['distanciaTotal'] / $info['litrosTotal'] : 0;
            if ($mediaConsumo > 0) {
                $mediasConsumoPorVeiculo[] = [
                    'id' => $idVeiculo,
                    'rotulo' => $info['rotulo'],
                    'valor' => $mediaConsumo
                ];
            }
        }

        // Ordena o array pelo campo 'valor' em ordem decrescente
        usort($mediasConsumoPorVeiculo, function ($item1, $item2) {
            return $item2['valor'] <=> $item1['valor'];
        });

        return $mediasConsumoPorVeiculo;
    }

    private function getListaAbastecimento($request)
    {
        $query = Nf::select([
            'nf.idnf',
            'nf.dataentrada',
            'veiculo.placa',
            'pessoa.nome',
            'pessoa.razaosocial',
            'nf.km',
            'nfitem.qtd',
            'nf.idveiculo',
            'veiculo.dashboard',
        ])
        ->join('nfitem', 'nfitem.idnf', '=', 'nf.idnf')
        ->join('veiculo', 'veiculo.idveiculo', '=', 'nf.idveiculo')
        ->leftJoin('pessoa', 'pessoa.idpessoa', '=', 'veiculo.idpessoa')
        ->whereIn('nfitem.idprodserv', [20, 31])
        ->where('nf.saida', 'P')
        ->where('nfitem.qtd', '>', 0);

        // Aplicar filtros dinâmicos
        filter($request, $query, 'nf');

        $lista = $query->orderBy('veiculo.dashboard', 'desc')
            ->orderBy('veiculo.marca')
            ->orderBy('veiculo.modelo')
            ->orderBy('veiculo.placa')
            ->orderBy('nf.dataentrada')
            ->get();

        $dates = $this->getDates($request);

        // Processa cada resultado para inserir o último quilômetro antes do início
        $anterior = 0;
        $dataInicio = $dates['dataInicio'];
        foreach ($lista as $resultado) {
            if (!isset($ultimoKm[$resultado->idveiculo])) {
                // Se for o primeiro registro do veículo no período, busca o último KM antes do início
                $ultimoKm[$resultado->idveiculo] = getUltimoKmAntesDe($resultado->idveiculo, $dataInicio);
                $resultado->ultimokm = $ultimoKm[$resultado->idveiculo];
            } else {
                // Se já houver um registro, usa o KM anterior do mesmo período
                $resultado->ultimokm = $anterior;
            }
            // Atualiza o último KM conhecido para esse veículo
            $ultimoKm[$resultado->idveiculo] = $resultado->km;
            $anterior = $resultado->km;
        }

        return $lista;
    }
}
