<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Empresa as Model;
use App\Models\Pessoa;
use Exception;
use Illuminate\Support\Facades\DB;

class EmpresaController extends Controller
{
    public function index()
    {
        $empresas = Model::orderBy('razaosocial')
                    ->paginate(100);

        return view('admin.empresas.index')->with(compact('empresas'));
    }

    public function fetch(Request $request)
    {
        $perPage = $request->input('per_page', 100);

        // Definir os campos e suas classes CSS
        $fields = [
            'ID' => [
                'key' => 'idempresa',
                'class' => 'text-center text-xs',
            ],
            'RAZÃO SOCIAL' => [
                'key' => fn($empresa) => $empresa->razaosocial ?? '-',
                'class' => 'text-sm',
            ],
            'NOME FANTASIA' => [
                'key' => 'nomefantasia',
                'class' => 'text-center text-xs',
            ],
            'STATUS' => [
                'key' => fn($empresa) => view('partials.status-switch', [
                    'status' => $empresa->status,
                    'id' => $empresa->idempresa,
                ])->render(),
                'class' => 'no-clicked pe-3',
            ],
        ];

        // Buscar os dados paginados
        $empresas = Model::orderBy('razaosocial')
            ->paginate($perPage, ['*'], 'page', $request->input('page', 2));

        // Montar os dados processados para o frontend
        $processedData = $empresas->map(function ($empresa) use ($fields) {
            $row = [
                'url' => route('admin.empresas.edit', $empresa->idempresa),
            ];
            foreach ($fields as $label => $field) {
                $value = is_callable($field['key'])
                    ? $field['key']($empresa) // Executar a closure
                    : data_get($empresa, $field['key']); // Obter o valor diretamente
                $row[$label] = [
                    'value' => $value,
                    'class' => $field['class'],
                ];
            }
            return $row;
        });

        // Retornar os dados processados como JSON
        return response()->json([
            'columns' => array_keys($fields),
            'data' => $processedData,
            'next_page_url' => $empresas->nextPageUrl(),
        ]);
    }

    public function create()
    {

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];
        return view('admin.empresas.create')->with(compact('statusOptions'));
    }

    public function store(Request $request)
    {
        $empresaCreate = $request->all();

        // Aqui definimos o tenant_id baseado no usuário logado
        $empresaCreate['idtenant'] = auth()->user()->idtenant;

        DB::beginTransaction();
        try {
            $empresa = Model::create($empresaCreate);

            // Cria um cliente/fornecedor da empresa
            $tipos = ['fornecedor', 'cliente'];
            foreach ($tipos as $tipo) {
                $this->createPersonDefault($empresa, $tipo);
            }

            DB::commit();
            return redirect()
                    ->route('admin.empresas.edit', $empresa)
                    ->with('success', 'Empresa criada com sucesso!');

        } catch(Exception $e) {
            DB::rollBack();
             return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', $e->getMessage());
        }
    }

    public function edit($idEmpresa)
    {
        $empresa = Model::find($idEmpresa);

        if ($empresa->status == 'ATIVO'){
            $bg_status = "bg-success";
        }elseif($empresa->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.empresas.edit')->with(compact('empresa','bg_status', 'statusOptions'));
    }

    public function update($idEmpresa, Request $request)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $empresa = Model::find($idEmpresa);

        DB::beginTransaction();
        try
        {
            $empresa->update($request->all());

            if ($redirect == false) {
                DB::commit();
                return response()->json($empresa->status);
            }

            DB::commit();
            return back()->with('success', 'Usuário atualizado!');

        } catch(Exception $e)
        {
            DB::rollBack();
            return redirect()
                    ->back()
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function changeStatus($id, $status)
    {
        // Chamando a função updateStatus da trait, passando o modelo e o request
        return $this->updateStatus(Model::class, $id, $status);
    }

    public function destroy($idEmpresa)
    {
        $empresa = Model::find($idEmpresa);

        try
        {
            $empresa->delete();

            return redirect()
                    ->route('admin.empresas.index')
                    ->with('success', 'Empresa removida!');
        } catch(Exception $e)
        {
            return redirect()
                    ->back()
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function search(Request $request)
    {
        $empresas =  Model::where('razaosocial', 'like', '%'.$request->search.'%')
                        ->orWhere('nomefantasia', 'like', '%'.$request->search.'%')
                        ->orderBy('razaosocial')
                        ->paginate(100);

        $size = $empresas->count();

        // Colocando mensagens na session
        session()->forget('error');
        session()->forget('success');
        if ($size && $size > 0 )
        {
            session()->flash('success', $size.' registro(s) encontrado(s).');

        } else
        {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        return view('admin.empresas.index')
                ->with(compact('empresas'));
    }

    public static function createPersonDefault($empresa, $tipo)
    {
        $pessoa = Pessoa::create([
            'idempresa' => $empresa->idempresa,
            'nome' => $empresa->nomefantasia,
            'razaosocial' => $empresa->razaosocial,
            'cpfcnpj' => $empresa->cnpj,
            'inscestadual' => $empresa->inscestadual,
            'cep' => $empresa->cep,
            'endereco' => $empresa->endereco,
            'numero' => $empresa->numero,
            'complemento' => $empresa->complemento,
            'bairro' => $empresa->bairro,
            'cidade' => $empresa->cidade,
            'estado' => $empresa->estado,
            'status' => 'ATIVO',
            'tipo' => $tipo,
            'pessoapadrao' => true
        ]);

        return $pessoa;
    }
}
