<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use App\Models\EventoTipo;
use App\Models\Pessoa;
use App\Models\Veiculo;
use App\Services\PedidoService;
use Illuminate\Http\Request;
use App\Models\Evento as Model;
use App\Models\NfItem;
use Exception;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class EventoController extends Controller
{
    public function index(Request $request, Pessoa $pessoaModel, EventoTipo $eventoTipoModel)
    {
        // Usa eager loading para carregar as relações de 'empresa' e 'eventotipo' de uma só vez
        $eventos = Model::with(['empresa', 'eventotipo', 'cliente'])
                        ->orderBy('data', 'desc')
                        ->orderBy('hora', 'desc')
                        ->orderBy('evento')
                        ->paginate(100);

        $statusOptions = [
            'PENDENTE' => 'Pendente',
            'EM EXECUÇÃO' => 'Em Execução',
            'CONCLUÍDO' => 'ConcluÍdo',
            'CANCELADO' => 'Cancelado',
        ];

        $clientes = $pessoaModel
            ->cliente()
            ->where('status', 'Ativo')
            ->orderBy('nome')
            ->get();

        $eventoTipos = $eventoTipoModel
            ->orderBy('eventotipo')
            ->get();


        return view('admin.eventos.index',
            compact('eventos','statusOptions', 'clientes', 'eventoTipos'));
    }

    public function fetch(Request $request)
    {
        $perPage = $request->input('per_page', 100);

        // Definir os campos e suas classes CSS
        $fields = [
            'ID' => [
                'key' => 'idevento',
                'class' => 'text-center text-xs',
            ],
            'CLIENTE' => [
                'key' => fn($evento) => $evento->cliente->nome ?? $evento->cliente->razaosocial ?? '-',
                'class' => 'text-start text-xs'
            ],
            'TIPO' => [
                'key' => fn($evento) => $evento->eventotipo->eventotipo ?? '-',
                'class' => 'text-center text-xs',
            ],
            'EVENTO' => [
                'key' => 'evento',
                'class' => 'text-center text-xs',
            ],
            'INÍCIO' => [
                'key' => fn($evento) => Carbon::parse($evento->data)->format('d/m/y'),
                'class' => 'text-center text-xs',
            ],
            'FIM' => [
                'key' => fn($evento) => Carbon::parse($evento->datafim)->format('d/m/y'),
                'class' => 'text-center text-xs',
            ],
            'DURAÇÃO' => [
                'key' => fn($evento) => $this->calculateDuration($evento),
                'class' => 'text-center text-xs',
            ],
            'QTD' => [
                'key' => 'qtd',
                'class' => 'text-center text-xs',
            ],
            'STATUS' => [
                'key' => fn($evento) => view('partials.status-progress', [
                    'status' => $evento->status,
                    'id' => $evento->idformapagamento,
                    'options' => [
                        ['percentage' => 25, 'status' => 'CANCELADO', 'tooltip' => 'Cancelado', 'class' => 'bg-danger'],
                        ['percentage' => 25, 'status' => 'PENDENTE', 'tooltip' => 'Pendente', 'class' => 'bg-warning'],
                        ['percentage' => 25, 'status' => 'EM EXECUÇÃO', 'tooltip' => 'Em execução', 'class' => 'bg-info'],
                        ['percentage' => 25, 'status' => 'CONCLUÍDO', 'tooltip' => 'Concluído', 'class' => 'bg-success'],
                    ]
                ])->render(),
                'class' => 'no-clicked pe-3',
            ],
        ];

        // Buscar os dados paginados
        $eventos = Model::with(['empresa', 'eventotipo', 'cliente'])
            ->orderBy('data', 'desc')
            ->orderBy('hora', 'desc')
            ->orderBy('evento')
            ->paginate($perPage, ['*'], 'page', $request->input('page', 2));

        // Montar os dados processados para o frontend
        $processedData = $eventos->map(function ($evento) use ($fields) {
            $row = [
                'url' => route('admin.eventos.edit', $evento->idevento),
            ];
            foreach ($fields as $label => $field) {
                // Obtém o valor da chave (key) do evento
                $value = is_callable($field['key'])
                    ? $field['key']($evento) // Executa a função de chave se for uma closure
                    : data_get($evento, $field['key']); // Ou obtém diretamente do evento
                // Monta a linha com os valores, classe e estilo
                $row[$label] = [
                    'value' => $value,
                    'class' => $field['class'], // Classe CSS
                ];
            }
            return $row;
        });

        // Retornar os dados processados como JSON
        return response()->json([
            'columns' => array_keys($fields),
            'data' => $processedData,
            'next_page_url' => $eventos->nextPageUrl(),
        ]);
    }

    public function create(Empresa $empresaModel, EventoTipo $eventoTipoModel,Pessoa $pessoaModel, Veiculo $veiculoModel)
    {
        $empresas = $empresaModel
                        ->orderBy('razaosocial')
                        ->get();

        $pessoas = $pessoaModel
                        ->orderBy('nome')
                        ->funcionario()
                        ->get();

        $clientes = $pessoaModel
                        ->cliente()
                        ->where('status', 'Ativo')
                        ->orderBy('nome')
                        ->get();

        $eventoTipos = $eventoTipoModel
                        ->orderBy('eventotipo')
                        ->get();

        $veiculos = $veiculoModel
                        ->orderBy('placa')
                        ->get();

        $statusOptions = [
                        'PENDENTE' => 'Pendente',
                        'EM EXECUÇÃO' => 'Em Execução',
                        'CONCLUÍDO' => 'ConcluÍdo',
                        'CANCELADO' => 'Cancelado',
                            // Adicione mais status conforme necessário
                        ];

        return view('admin.eventos.create')->with(compact('empresas', 'eventoTipos','pessoas', 'clientes','veiculos', 'statusOptions'));
    }

    public function store(Request $request)
    {
        $camposMonetarios = ['valorun', 'valoritem'];
        $eventoCreate = $request->all();
         // Trata os campos monetários
         foreach ($camposMonetarios as $campo) {
            if (isset($eventoCreate[$campo])) {
                $eventoCreate[$campo] = str_replace(',', '.', preg_replace('/[^\d,\.]/', '', $eventoCreate[$campo]));
                $eventoCreate[$campo] = floatval($eventoCreate[$campo]);
            }
        }

        try
        {
            $evento = Model::create($eventoCreate);

            return redirect()->to('admin/eventos/'.$evento->idevento.'/edit');

        } catch(Exception $e)
        {
            return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', $e->getMessage());
        }
    }

    public function edit($idEvento, Empresa $empresaModel, EventoTipo $eventoTipoModel, Pessoa $pessoaModel,Veiculo $veiculoModel, NfItem $nfItemModel)
    {
        $evento = Model::find($idEvento);

        $empresas = $empresaModel
        ->orderBy('razaosocial')
        ->get();

        $eventoTipos = $eventoTipoModel
        ->orderBy('eventotipo')
        ->get();

        $pessoas = $pessoaModel
        ->orderBy('nome')
        ->funcionario()
        ->get();

        $clientes = $pessoaModel
        ->where('status', 'Ativo')
        ->orderBy('nome')
        ->cliente()
        ->get();


        $veiculos = $veiculoModel
        ->orderBy('placa')
        ->get();

        $nf = null; // Inicializa como null para o caso de não haver NfItem
        if (isset($evento->idnfitem)) {
            $nfItem = NfItem::with('nf')->find($evento->idnfitem);
            if ($nfItem && $nfItem->nf) {
                $nf = $nfItem->nf;
                $nf->datavencimento = new Carbon($nf->datavencimento);

                if ($nf->status == 'CONCLUÍDO'){
                    $bg_status_nf = "bg-success";
                }elseif($nf->status == 'CANCELADO'){
                    $bg_status_nf = "bg-danger";
                }elseif($nf->status == 'PENDENTE'){
                    $bg_status_nf = "bg-warning";
                }else{
                    $bg_status_nf = "bg-info";
                }


            }else{
                $bg_status_nf = '';
            }
        }else{
            $bg_status_nf = '';
        }


        if ($evento->status == 'CONCLUÍDO'){
            $bg_status = "bg-success";
        }elseif($evento->status == 'CANCELADO'){
            $bg_status = "bg-danger";
        }elseif($evento->status == 'PENDENTE'){
            $bg_status = "bg-warning";
        }else{
            $bg_status = "bg-info";
        }

        $statusOptions = [
            'PENDENTE' => 'Pendente',
            'EM EXECUÇÃO' => 'Em Execução',
            'CONCLUÍDO' => 'ConcluÍdo',
            'CANCELADO' => 'Cancelado',
                // Adicione mais status conforme necessário
            ];

        return view('admin.eventos.edit')->with(compact('evento', 'empresas','eventoTipos', 'pessoas', 'clientes', 'bg_status','veiculos', 'nf', 'statusOptions','bg_status_nf'));
    }

    public function update($idEvento, Request $request, PedidoService $pedidoService)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $evento = Model::find($idEvento);

        //preciso chamar o sincronizaPedido aqui

        //precisa atualizar o idobjeto e objeto (aqui ou no sincronizaPedido?)

        if ($redirect == true ) {
            $camposMonetarios = ['qtd','valorun','valoritem' /* outros campos */];

            if (isset($evento)){
                // Trata os campos monetários
                foreach ($camposMonetarios as $campo) {
                    if (isset($request[$campo])) {
                        $request[$campo] = str_replace(',', '.', preg_replace('/[^\d,\.]/', '', $request[$campo]));
                        $request[$campo] = floatval($request[$campo]);
                    }
                }
            }
            ($request['valorun']);
        }

        DB::beginTransaction();
        try {
            $evento->update($request->all());

            // Cria uma instância do serviço de Pedido
            //$pedidoService = new \App\Services\PedidoService();
            //$pedidoService->sincronizaPedidoEvento($evento);

            if ($redirect == false) {
                DB::commit();
                return response()->json($evento->status);
            }

            DB::commit();
            return back()->with('success', 'Evento atualizado!');

        } catch(Exception $e) {
            DB::rollBack();
            return redirect()
                    ->route('admin.eventos.index')
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function destroy($idEvento)
    {
        $evento = Model::find($idEvento);

        try
        {
            $evento->delete();

            return redirect()
                    ->route('admin.eventos.index')
                    ->with('success', 'Evento removido!');
        } catch(Exception $e)
        {
            return redirect()
                    ->back()
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function search(Request $request, Pessoa $pessoaModel, EventoTipo $eventoTipoModel)
    {
        $eventos = Model::select('evento.*')
            ->when($request->status, function ($query) use ($request) {
                $query->where('status', $request->status);
            });

        // Aplicar busca textual (search) se presente
        if ($request->filled('search')) {
            $searchTerm = '%' . $request->search . '%';
            $eventos->where(function ($query) use ($searchTerm) {
                $query->whereHas('cliente', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('nome', 'like', $searchTerm)
                        ->orWhere('razaosocial', 'like', $searchTerm);
                })->orWhereHas('eventotipo', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('eventotipo', 'like', $searchTerm);
                })->orWhere('evento', 'like', $searchTerm);
            });
        }

        // Aplica filtros dinâmicos com base nos campos
        filter($request, $eventos);

        $eventos = $eventos->orderBy('data', 'desc')->paginate(50);

        // Mensagens da sessão
        $size = $eventos->count();
        session()->forget('error');
        session()->forget('success');
        if ($size > 0) {
            session()->flash('success', $size . ' valor(es) encontrado(s).');
        } else {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        $statusOptions = [
            'PENDENTE' => 'Pendente',
            'EM EXECUÇÃO' => 'Em Execução',
            'CONCLUÍDO' => 'ConcluÍdo',
            'CANCELADO' => 'Cancelado',
        ];

        $clientes = $pessoaModel
            ->cliente()
            ->where('status', 'Ativo')
            ->orderBy('nome')
            ->get();

        $eventoTipos = $eventoTipoModel
            ->orderBy('eventotipo')
            ->get();

        return view('admin.eventos.index',
            compact(
                'eventos',
                'statusOptions',
                'clientes',
                'eventoTipos'
            ));
    }

    private function calculateDuration($evento)
    {
        // Combine data e hora
        $startDateTime = Carbon::parse($evento->data . ' ' . $evento->hora);

        // Combine datafim e horafim
        $endDateTime = Carbon::parse($evento->datafim . ' ' . $evento->horafim);

        // Calcule a diferença
        $duration = $startDateTime->diff($endDateTime);

        // Verifique se a diferença é maior que 1 dia
        if ($duration->days > 1) {
            if ($evento->eventotipo->hora == 'SIM') {
                // Formate a diferença para mostrar em dias, horas, minutos, etc.
                $durationString = $duration->format('%d dias %H:%I');
            } else {
                // Formate a diferença para mostrar em dias
                $durationString = $duration->format('%d dias');
            }
        } else {
            // Formate a diferença para mostrar apenas horas e minutos
            $durationString = $duration->format('%H:%I');
        }

        return $durationString;
    }

}
