<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use Illuminate\Http\Request;
use App\Models\Pessoa as Model;
use Exception;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class FuncionarioController extends Controller
{

	protected $admissaomotivo = [
		'atendimento a demandas regulatórias' => "Atendimento a demandas regulatórias",
		'criação de novas posições' => "Criação de novas posições",
		'desenvolvimento de competências' => "Desenvolvimento de competências",
		'desenvolvimento de novos projetos' => "Desenvolvimento de novos projetos",
		'diversificação do quadro de funcionários' => "Diversificação do quadro de funcionários",
		'expansão da equipe' => "Expansão da equipe",
		'expansão para novos mercados' => "Expansão para novos mercados",
		'implementação de novas tecnologias' => "Implementação de novas tecnologias",
		'melhoria da eficiência operacional' => "Melhoria da eficiência operacional",
		'necessidade de habilidades específicas' => "Necessidade de habilidades específicas",
		'otimização da relação custo-benefício' => "Otimização da relação custo-benefício",
		'reposicionamento estratégico' => "Reposicionamento estratégico",
		'substituição de funcionários' => "Substituição de funcionários",
		'sucessão de liderança' => "Sucessão de liderança",
		'suporte a expansão geográfica' => "Suporte a expansão geográfica"
    ];

	protected $demissaomotivo = [
		'baixo desempenho' => "Baixo desempenho",
		'conduta inadequada' => "Conduta inadequada",
		'conclusão de projeto' => "Conclusão de projeto",
		'conflitos no local de trabalho' => "Conflitos no local de trabalho",
		'desacordo cultural' => "Desacordo cultural",
		'desalinhamento de valores' => "Desalinhamento de valores",
		'desempenho insatisfatório' => "Desempenho insatisfatório",
		'dificuldades financeiras' => "Dificuldades financeiras",
		'falta de qualificação' => "Falta de qualificação",
		'falta de motivação' => "Falta de motivação",
		'falta de ética profissional' => "Falta de ética profissional",
		'incompatibilidade de equipe' => "Incompatibilidade de equipe",
		'mudança na estrutura organizacional' => "Mudança na estrutura organizacional",
		'redução de custos' => "Redução de custos",
		'reestruturação da empresa' => "Reestruturação da empresa",
		'saída voluntária' => "Saída voluntária",
		'término de contrato temporário' => "Término de contrato temporário",
		'insubordinação' => "Insubordinação",
		'problemas de saúde' => "Problemas de saúde",
		'redução de equipe' => "Redução de equipe"
    ];

    public function index(Request $request)
    {

        $module = $request->segment(2); // 'vendas' ou 'pedidos'
        $key = $module . '_dateRange'; // chave será 'vendas_dateRange' ou 'pedidos_dateRange'

        if ($request->has('clear') || session($key) === "") {
            session()->forget($key);
        }

        if (session($key)) {
            // Redirecionar para search com o date_range da session específica do módulo
            return redirect()->route('admin.funcionarios.search', ['date_range' => session($key)]);
        }

        $funcionarios = Model::with('empresa')->orderBy('nome')->funcionario()->paginate(100);

		$admissaomotivo = $this->admissaomotivo;

		$demissaomotivo = $this->demissaomotivo;

        return view('admin.funcionarios.index')->with(compact('funcionarios', 'admissaomotivo', 'demissaomotivo'));
    }

    public function fetch(Request $request)
    {
        $perPage = $request->input('per_page', 100);

        // Definir os campos e suas classes CSS
        $fields = [
            'ID' => [
                'key' => 'idpessoa',
                'class' => 'text-center text-xs',
            ],
            'EMPRESA' => [
                'key' => 'empresa.sigla',
                'class' => 'text-center text-xs',
            ],
            'NOME' => [
                'key' => 'nome',
                'class' => 'text-center text-xs',
            ],
            'CPF' => [
                'key' => 'cpfcnpj',
                'class' => 'text-center text-xs',
            ],
            'ADMISSÃO' => [
                'key' => fn($funcionario) => Carbon::parse($funcionario->admissao)->format('d/m/y'),
                'class' => 'text-center text-xs',
            ],
            'STATUS' => [
                'key' => fn($item) => view('partials.status-switch', [
                    'status' => $item->status,
                    'id' => $item->idformapagamento,
                ])->render(),
                'class' => 'no-clicked d-flex align-items-center justify-content-center',
            ],
        ];

        // Buscar os dados paginados
        $funcionarios = Model::with('empresa')
            ->orderBy('nome')
            ->funcionario()
            ->paginate($perPage, ['*'], 'page', $request->input('page', 2));

        // Montar os dados processados para o frontend
        $processedData = $funcionarios->map(function ($funcionario) use ($fields) {
            $row = [
                'url' => route('admin.funcionarios.edit', $funcionario->idpessoa),
            ];
            foreach ($fields as $label => $field) {
                $value = is_callable($field['key'])
                    ? $field['key']($funcionario) // Executar a closure
                    : data_get($funcionario, $field['key']); // Obter o valor diretamente
                $row[$label] = [
                    'value' => $value,
                    'class' => $field['class'],
                ];
            }
            return $row;
        });

        // Retornar os dados processados como JSON
        return response()->json([
            'columns' => array_keys($fields),
            'data' => $processedData,
            'next_page_url' => $funcionarios->nextPageUrl(),
        ]);
    }

    public function create(Empresa $empresaModel)
    {
        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

		$admissaomotivo = $this->admissaomotivo;

		$demissaomotivo = $this->demissaomotivo;

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.funcionarios.create')->with(compact('empresas', 'admissaomotivo', 'demissaomotivo', 'statusOptions'));
    }

    public function store(Request $request)
    {
        $pessoaCreate = $request->all();

        // Tratamento para temagenda
        if ($pessoaCreate['temagenda'] == 'on') {
            $pessoaCreate['temagenda'] = true;
        } else {
            $pessoaCreate['temagenda'] = false;
        }

        try {
           $pessoa =  Model::create($pessoaCreate);

            // @TODO: Gerar usuario com permissões padroes

            return redirect()->to('admin/funcionarios/'.$pessoa->idpessoa.'/edit');

        } catch (Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->with('error', $e->getMessage());
        }
    }

    public function edit($idPessoa, Empresa $empresaModel)
    {
        $funcionario = Model::find($idPessoa);
        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

		$admissaomotivo = $this->admissaomotivo;

		$demissaomotivo = $this->demissaomotivo;

        if ($funcionario->status == 'ATIVO'){
            $bg_status = "bg-success";
        }elseif($funcionario->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.funcionarios.edit')->with(compact('funcionario', 'empresas', 'admissaomotivo', 'demissaomotivo','bg_status', 'statusOptions'));
    }

    public function update($idPessoa, Request $request)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $pessoa = Model::find($idPessoa);
        $pessoaUpdate = $request->all();

        if ($redirect == true) {
            // Tratamento para temagenda
            if (isset($pessoaUpdate['temagenda']) && $pessoaUpdate['temagenda'] == 'on') {
                $pessoaUpdate['temagenda'] = true;
            } else {
                $pessoaUpdate['temagenda'] = false;
            }
        }

        DB::beginTransaction();
        try {
            $pessoa->update($pessoaUpdate);

            if ($redirect == false) {
                DB::commit();
                return response()->json($pessoa->status);
            }

            DB::commit();
            return back()->with('success', 'Funcionário atualizado!');

        } catch (Exception $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->withInput()
                ->with('error', $e->getMessage());
        }
    }

    public function destroy($idPessoa)
    {
        $pessoa = Model::find($idPessoa);

        try {
            $pessoa->delete();

            return redirect()
                ->route('admin.funcionarios.index')
                ->with('success', 'Funcionário removido!');
        } catch (Exception $e) {
            return redirect()
                ->back()
                ->with('error', $e->getMessage())
                ->withInput();
        }
    }


    public function search(Request $request, $tipo = 'cliente')
    {
        $intervaloDatas = explode(' - ', $request->date_range);
       // dump($intervaloDatas);
        // Inicializa as variáveis $dataInicio e $dataFim como null ou strings vazias
        $dataInicio = '';
        $dataFim = '';

        // Verifica se existem pelo menos dois elementos no array
        if (count($intervaloDatas) >= 2) {
            $dataInicio = $intervaloDatas[0];
            $dataFim = $intervaloDatas[1];

            $dataInicio = Carbon::createFromFormat('d/m/y', $intervaloDatas[0])->format('Y-m-d');
            $dataFim = Carbon::createFromFormat('d/m/y', $intervaloDatas[1])->format('Y-m-d');
        }


        // Aplicando Eager Loading para evitar N+1
        $query = Model::with('empresa')
        ->when($request->date_range, function ($query) use ($dataInicio, $dataFim) {
            $query->whereBetween('admissao', [$dataInicio, $dataFim]);
        })
        ->where(function ($query) use ($request) {
            $query->where('nome', 'like', '%' . $request->search . '%')
                ->orWhere('pessoa.razaosocial', 'like', '%' . $request->search . '%')
                ->orWhere('cpfcnpj', 'like', '%' . $request->search . '%')
                ->orWhereHas('empresa', function ($q) use ($request) {
                    $q->where('nomefantasia', 'like', '%' . $request->search . '%');
                })
                ->orWhereHas('empresa', function ($q) use ($request) {
                    $q->where('empresa.razaosocial', 'like', '%' . $request->search . '%');
                });
        });
        $query->funcionario();

        $funcionarios = $query
        ->orderBy('nome')
        ->paginate(100);

        $size = $funcionarios->count();

        // Colocando mensagens na session
        session()->forget('error');
        session()->forget('success');
        if ($size && $size > 0 )
        {
            session()->flash('success', $size.' valores encontrado(s).');

        } else
        {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        $admissaomotivo = $this->admissaomotivo;

		$demissaomotivo = $this->demissaomotivo;

        //Salvar Sessão DateRange e Busca
        $module = $request->segment(2); // 'vendas' ou 'pedidos'

        $key = $module . '_dateRange'; // chave será 'vendas_dateRange' ou 'pedidos_dateRange'

        if (session($key)) {
            $dateRange = session($key);
        }else{
            $dateRange = $request->date_range ? urldecode($request->date_range) : '';
        }

        // Armazenar dateRange na session com chave específica do módulo
        session([$key => $dateRange]);
        return view('admin.funcionarios.index')->with(compact('funcionarios', 'admissaomotivo', 'demissaomotivo','dateRange'));
    }
}
