<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Agencia;
use App\Models\Empresa;
use App\Models\FormaPagamento;
use Illuminate\Http\Request;
use App\Models\Nf as Model;
use App\Models\NfParcela;
use App\Models\NfItem;
use App\Models\ProdutoServico;
use App\Models\Pessoa;
use App\Models\TipoProdutoServico;
//use App\Models\Evento;
use App\Models\GrupoItem;
use App\Models\Item;
use Exception;
use Illuminate\Support\Facades\DB;
use DateTime;
use Carbon\Carbon;

class NfController extends Controller
{
    public function index(Request $request, Empresa $empresaModel, TipoProdutoServico $tipoProdutoServicoModel, Pessoa $pessoaModel)
    {
        // Usar eager loading para carregar as relações e evitar consultas N+1
        $compras = Model::with(['empresa', 'fornecedor', 'prodservtipo'])
                        ->where('status', '!=', 'INATIVO')
                        ->compra()
                        ->orderBy('datavencimento', 'desc')
                        ->paginate(100);

        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

        $tiposProdutoServico = $tipoProdutoServicoModel
            ->active()
            ->orderBy('prodservtipo')
            ->get();

        $pessoas = $pessoaModel
            ->Fornecedor()
            ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.compras.index')->with(compact('compras', 'statusOptions', 'empresas', 'tiposProdutoServico', 'pessoas'));
    }

    public function fetch(Request $request)
    {
        $perPage = $request->input('per_page', 100);

        // Definir os campos e suas classes CSS
        $fields = [
            'ID' => [
                'key' => 'idnf',
                'class' => 'text-center text-xs',
            ],
            'EMPRESA' => [
                'key' => 'empresa.sigla',
                'class' => 'text-center text-xs',
            ],
            'FORNECEDOR' => [
                'key' => fn($compra) => $compra->fornecedor->razaosocial ?? $compra->fornecedor->nome ?? '-',
                'class' => 'text-sm',
            ],
            'TIPO' => [
                'key' => fn($compra) => $compra->prodservtipo->prodservtipo ?? '-',
                'class' => 'text-center text-xs',
            ],
            'VALOR' => [
                'key' => fn($compra) => "R$ ". number_format($compra->totalnf, 2, ',', '.'),
                'class' => 'text-center text-xs fw-bold text-danger',
            ],
            'ENTRADA' => [
                'key' => fn($compra) => Carbon::parse($compra->dataentrada)->format('d/m/y'),
                'class' => 'text-center text-xs fw-bold',
            ],
            'VENCIMENTO' => [
                'key' => fn($compra) => $compra->datavencimento
                    ? Carbon::parse($compra->datavencimento)->format('d/m/y')
                    : '-',
                'class' => 'text-center text-xs fw-bold',
            ],
            'STATUS' => [
                'key' => fn($compra) => view('partials.status-progress', [
                    'status' => $compra->status,
                    'id' => $compra->idnf,
                    'options' => [
                        ['percentage' => 50, 'status' => 'INATIVO', 'tooltip' => 'Inativo', 'class' => 'bg-danger'],
                        ['percentage' => 50, 'status' => 'ATIVO', 'tooltip' => 'Ativo', 'class' => 'bg-info'],
                        ['percentage' => 50, 'status' => 'CONCLUÍDO', 'tooltip' => 'Concluído', 'class' => 'bg-success'],
                    ]
                ])->render(),
                'class' => 'no-clicked pe-3',
            ],
        ];

        // Buscar os dados paginados
        $compras = Model::with(['empresa', 'fornecedor', 'prodservtipo'])
            ->where('status', '!=', 'INATIVO')
            ->compra()
            ->orderBy('datavencimento', 'desc')
            ->paginate($perPage, ['*'], 'page', $request->input('page', 2));

        // Montar os dados processados para o frontend
        $processedData = $compras->map(function ($compra) use ($fields) {
            $row = [
                'url' => route('admin.compras.edit', ['compra' => $compra->idnf, 'tipo' => $compra->tipo]),
            ];
            foreach ($fields as $label => $field) {
                $value = is_callable($field['key'])
                    ? $field['key']($compra) // Executar a closure
                    : data_get($compra, $field['key']); // Obter o valor diretamente
                $row[$label] = [
                    'value' => $value,
                    'class' => $field['class'],
                ];
            }
            return $row;
        });

        // Retornar os dados processados como JSON
        return response()->json([
            'columns' => array_keys($fields),
            'data' => $processedData,
            'next_page_url' => $compras->nextPageUrl(),
        ]);
    }

    public function create(Pessoa $pessoaModel, Empresa $empresaModel, FormaPagamento $formaPagamentoModel, TipoProdutoServico $tipoProdutoServicoModel, GrupoItem $grupoItemModel, Agencia $agenciaModel)
    {
        $pessoas = $pessoaModel
                    ->orderBy('razaosocial')
                    ->orderBy('nome')
                    ->fornecedor()
                    ->get();

        $tiposProdutoServico = $tipoProdutoServicoModel
                    ->active()
                    ->orderBy('prodservtipo')
                    ->get();

        $grupoItens = $grupoItemModel
                    ->active()
                    ->orderBy('grupoitem')
                    ->get();

        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

        $formaPagamento = $formaPagamentoModel
                            ->orderBy('formapagamento')
                            ->get();

        $agencias = $agenciaModel
                            ->orderBy('nome')
                            ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.compras.create', compact('pessoas', 'empresas', 'formaPagamento','tiposProdutoServico','grupoItens', 'statusOptions','agencias'));
    }

    public function store(Request $request)
    {
        try
        {
            DB::beginTransaction();

            $compraCreate = $request->all();

            $compra = Model::create($compraCreate);

            DB::commit();
            //return back()->with('success', 'Compra criada com sucesso!');
            return redirect()->to('admin/compras/'.$compra->idnf.'/edit');

        } catch(Exception $e)
        {
            DB::rollBack();

            return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', $e->getMessage());
        }
    }

    public function edit($idNf, Pessoa $pessoaModel, Empresa $empresaModel, FormaPagamento $formaPagamentoModel,TipoProdutoServico $tipoProdutoServicoModel, NfItem $nfItemModel, ProdutoServico $prodservModel, NfParcela $nfParcelaModel, GrupoItem $grupoItemModel, Item $itemModel, Agencia $agenciaModel)
    {
        $compra = Model::find($idNf);
        $pessoas = $pessoaModel
                    ->orderBy('razaosocial')
                    ->orderBy('nome')
                    ->fornecedor()
                    ->get();

        $empresas = $empresaModel
        ->orderBy('razaosocial')
        ->get();

        $tiposProdutoServico = $tipoProdutoServicoModel
                    ->orderBy('prodservtipo')
                    ->get();

        $grupoItens = $grupoItemModel
                    ->orderBy('grupoitem')
                    ->get();

        $listaItens = $itemModel
                    ->orderBy('item')
                    ->get();

        $formaPagamento = $formaPagamentoModel
                        ->orderBy('formapagamento')
                        ->get();

        $nfItem = $nfItemModel
                        ->where('idnf', $idNf) // Filtra pelos registros com o idnf relacionado
                        ->orderBy('idnfitem')
                        ->get();


        $prodserv = $prodservModel
                    ->whereHas('tipos', function($query) use ($compra) {
                        if (isset($compra->idprodservtipo)) {
                            $query->where('prodserv_prodservtipo.idprodservtipo', $compra->idprodservtipo); // Qualifica o nome da coluna com a tabela intermediária
                        }
                    })
                    ->orderBy('prodserv')
                    ->get();



        $nfParcela = $nfParcelaModel
                        ->where('idnf', $idNf) // Filtra pelos registros com a nf
                        ->orderBy('datavencimento')
                        ->get();

     /*   $eventos = $eventoModel
                        ->where('idcliente',$compra->idpessoa)
                        ->where('status', 'ATIVO')
                        ->where(function ($query) {
                            $query->whereNull('idnfitem')
                                  ->orWhere('idnfitem', '');
                        })
                        ->get();*/

        $agencias = $agenciaModel
                        ->orderBy('nome')
                        ->get();

        if ($compra->status == 'CONCLUÍDO'){
            $bg_status = "bg-success";
        }elseif($compra->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.compras.edit', compact('compra','pessoas', 'empresas', 'formaPagamento', 'tiposProdutoServico', 'nfItem', 'prodserv', 'nfParcela','bg_status','grupoItens', 'listaItens','statusOptions','agencias'));
    }

    public function update($idNf, Request $request)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $compra = Model::find($idNf);
        $compraUpdate = $request->all();

        if ($redirect == true) {
            $camposMonetarios = ['descontonf'];

            // Trata os campos monetários
            foreach ($camposMonetarios as $campo) {
                if (isset($compraUpdate[$campo])) {
                    $compraUpdate[$campo] = str_replace(',', '.', preg_replace('/[^\d,\.]/', '', $compraUpdate[$campo]));
                    $compraUpdate[$campo] = floatval($compraUpdate[$campo]);
                }
            }
        }

        DB::beginTransaction();
        try {
            $compra->update($compraUpdate);

            if ($request->exists('idnfitem')) {
                $input = $request->all();
                $condition = $input['idnfitem'];
                $camposMonetarios = ['qtd','valorun','valoritem' /* outros campos */];

                if (isset($condition)) {
                    foreach ($condition as $key => $c) {

                        $itens = NfItem::find($input['idnfitem'][$key]);

                        // Trata os campos monetários
                        foreach ($camposMonetarios as $campo) {
                            if (isset($input[$campo][$key])) {
                                $input[$campo][$key] = str_replace(',', '.', preg_replace('/[^\d,\.]/', '', $input[$campo][$key]));
                                $input[$campo][$key] = floatval($input[$campo][$key]);
                            }
                        }

                        if (isset($input['idprodserv'][$key])) {
                            $itens->idprodserv = $input['idprodserv'][$key];

                            // Localiza o registro prodserv correspondente
                            $prodserv = ProdutoServico::find($itens->idprodserv);

                            // Verifica se o registro foi encontrado
                            if ($prodserv) {
                                // Atualiza o campo valorcompra com o novo valor
                                $prodserv->valorcompra = $input['valorun'][$key];
                                $prodserv->save();
                            }
                        }

                        if (isset($input['idprodserv'][$key])) {
                            $itens->idprodserv = $input['idprodserv'][$key];
                        }

                        $item = Item::find($input['iditem'][$key]);
                        if ($item) {
                            $item->idgrupoitem = $input['idgrupoitem'][$key];
                            $item->save();
                        }

                        $itens->idgrupoitem = $input['idgrupoitem'][$key];
                        $itens->iditem = $input['iditem'][$key];
                        $itens->qtd = $input['qtd'][$key];
                        $itens->idempresa = $compra->idempresa;
                        $itens->un = $input['un'][$key];
                        $itens->valorun = $input['valorun'][$key];
                        $itens->valoritem = $input['valoritem'][$key];
                        $itens->save();
                    }
                }
            }

            $textData = $request->input('data');

            if (isset($textData)) {
                // Quebra o texto em linhas
                $lines = explode("\n", $textData);

                foreach ($lines as $line) {
                    // Remove espaços extras no início e no fim da linha
                    $line = trim($line);

                    // Detectar o formato da linha
                    if (preg_match('/\d{2} de [a-z]{3}/i', $line)) {
                        // Captura a linha completa usando uma regex flexível
                        preg_match('/(\d{2} de [a-z]{3})\s+(.+?)\s+R\$\s*([0-9,.]+)/i', $line, $matches);

                        if (count($matches) === 4) {
                            $dataCompleta = $matches[1]; // Data
                            $lancamento = $matches[2]; // Descrição completa do produto
                            // Remover as parcelas da descrição para salvar no banco de dados como 'item'
                            $lancamentoSemParcelas = preg_replace('/\(\d+\/\d+\)/', '', $lancamento);
                            $valor = $matches[3]; // Valor

                            // Limpar o valor monetário
                            $valor = preg_replace('/[^\d,.-]/', '', $valor);

                            // Criar ou recuperar o item
                            $item = Item::firstOrCreate(
                                [
                                    'idempresa' => $compra->idempresa,
                                    'item' => $lancamentoSemParcelas
                                ]
                            );

                            // Criar NfItem
                            $nfitem = new NfItem();
                            $nfitem->idnf = $compra->idnf;
                            $nfitem->idempresa = $compra->idempresa;
                            $nfitem->idgrupoitem = $item->idgrupoitem;
                            $nfitem->iditem = $item->iditem;
                            $nfitem->importado = 1;
                            $nfitem->item = $lancamento;
                            $nfitem->qtd = 1;
                            $nfitem->valorun = $this->convertValue($valor);
                            $nfitem->valoritem = $this->convertValue($valor);
                            $nfitem->save();
                        }
                    } elseif (preg_match('/\d{2} \/\s*[a-z]{3}/i', $line)) {
                        // Formato tabulado (ex: "30 / mar Loja Doce -ct 06/10")
                        $parts = preg_split('/\t+/', $line); // Divisão por tabulação

                        if (count($parts) >= 3) {
                            $lancamento = $parts[1];

                            // Remover a parcela no formato XX/YY
                            $lancamento = preg_replace('/\d{2}\/\d{2}/', '', $lancamento);
                            $lancamento = trim($lancamento); // Remover espaços extras

                            $valor = preg_replace('/[^\d,.-]/', '', $parts[2]); // Limpar valor monetário

                            // Criar ou recuperar o item
                            $item = Item::firstOrCreate(
                                [
                                    'idempresa' => $compra->idempresa,
                                    'item' => $lancamento
                                ]
                            );

                            // Criar NfItem
                            $nfitem = new NfItem();
                            $nfitem->idnf = $compra->idnf;
                            $nfitem->idempresa = $compra->idempresa;
                            $nfitem->idgrupoitem = $item->idgrupoitem;
                            $nfitem->iditem = $item->iditem;
                            $nfitem->importado = 1;
                            $nfitem->item = $lancamento;
                            $nfitem->qtd = 1;
                            $nfitem->valorun = $this->convertValue($valor);
                            $nfitem->valoritem = $this->convertValue($valor);
                            $nfitem->save();
                        }
                    }
                }
            }

            $status = $compra['status'];
            $totalnf = $compra['totalnf'];
            $qtdparcela = $compra['qtdparcela'];
            $datavencimento = $compra['datavencimento'];

            if ($status == 'CONCLUÍDO'){
                // Verificar se existem registros relacionados na tabela nfparcela
                $count = DB::table('nfparcela')
                ->where('idnf', $idNf)
                ->count();
               $vparcela = 0;
                if ($count <= 0) {
                    $vparcela = $totalnf / $qtdparcela;

                    $parcelas = [];

                    $dataVencimento = DateTime::createFromFormat('Y-m-d', $datavencimento);
                    $dataVencimento->setTime(0, 0, 0);
                    $dataVencimentoparcela = clone $dataVencimento; // Inicializa a variável aqui

                        // Iterar para calcular as demais datas de vencimento
                        for ($i = 0; $i < $qtdparcela; $i++) {

                                // Clonar a data de vencimento anterior para criar uma nova instância
                                $dataVencimentoparcela = clone $dataVencimento;

                                // Adicionar 1 mês à data de vencimento
                                $dataVencimento->modify('+1 month');



                            // Verificar se o dia é válido
                            if (!$this->checkDiaUtil($dataVencimentoparcela)) {
                                // Se não for um dia útil, ajustar para a próxima segunda-feira
                                $dataVencimentoparcela = $this->ajustarDiaUtil($dataVencimentoparcela);
                            }

                            // Adicionar a data de vencimento ao array
                            $parcelas[$i] = [
                                'idagencia' => $compra->idagencia,
                                'tipo' => $compra->tipo,
                                'idempresa' => $compra->idempresa,
                                'parcela' => $i + 1,
                                'valor' => $vparcela,
                                'datavencimento' => $dataVencimentoparcela,
                                'status' => 'ATIVO'
                            ];

                        }

                    // TODO: gerar parcelas
                    $compra->parcela()->createMany($parcelas);
                }

            } elseif ($status == 'ATIVO') {
                // Verificar se existem registros relacionados na tabela nfparcela com status CONCLUÍDO
                $countConcluido = DB::table('nfparcela')
                                    ->where('idnf', $idNf)
                                    ->where('status', 'CONCLUÍDO')
                                    ->count();

                if ($countConcluido == 0) {
                    // Se não existem parcelas concluídas, exclui as existentes que não estão concluídas
                    DB::table('nfparcela')
                      ->where('idnf', $idNf)
                      ->where('status', '<>', 'CONCLUÍDO')
                      ->delete();
                }
            }

            if ($redirect == false) {
                DB::commit();
                return response()->json($compra->status);
            }

            DB::commit();
            return back()->with('success', 'Compra atualizada!');

        } catch(Exception $e) {
            DB::rollBack();
            return redirect()
                    ->route('admin.compras.edit', $compra)
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function destroy($idNf)
    {
        $empresa = Model::find($idNf);

        try
        {
            $empresa->delete();

            return redirect()
                    ->route('admin.compras.index')
                    ->with('success', 'Empresa removida!');
        } catch(Exception $e)
        {
            return redirect()
                    ->back()
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function search(Request $request, Empresa $empresaModel, TipoProdutoServico $tipoProdutoServicoModel, Pessoa $pessoaModel)
    {
        $compras = Model::select('nf.*')
            ->with(['pessoa', 'empresa', 'prodservtipo'])
            ->Compra()
            ->when($request->status, function ($query) use ($request) {
                return $query->where('status', $request->status);
            });

        // Aplica filtros de busca por termo se o campo `search` estiver preenchido
        if ($request->filled('search') && $request->search !== null) {
            $searchTerm = '%' . $request->search . '%';
            $compras->where(function ($query) use ($searchTerm) {
                $query->whereHas('pessoa', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('nome', 'like', $searchTerm)
                        ->orWhere('razaosocial', 'like', $searchTerm);
                })->orWhereHas('prodservtipo', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('prodservtipo', 'like', $searchTerm);
                })->orWhereHas('empresa', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('sigla', 'like', $searchTerm);
                });
            });
        }

        // Aplica filtros dinâmicos com base nos campos
        filter($request, $compras);

        // Paginação e ordenação
        $compras = $compras->orderBy('dataentrada', 'desc')->paginate(100);

        session()->forget(['error', 'success']);
        if ($compras->count() > 0) {
            session()->flash('success', $compras->count() . ' valores encontrado(s).');
        } else {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        // Carrega opções adicionais para a view
        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
        ];

        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

        $tiposProdutoServico = $tipoProdutoServicoModel
            ->active()
            ->orderBy('prodservtipo')
            ->get();

        $pessoas = $pessoaModel
            ->Fornecedor()
            ->get();

        return view('admin.compras.index',
            compact(
                'compras',
                'statusOptions',
                'empresas',
                'tiposProdutoServico',
                'pessoas'
            ));
    }


    public function removeitem($idNf){
        NfItem::find($idNf)->delete($idNf);
        return response()->json([
        'success' => 'Record deleted successfully!'
        ]);
    }

    public function additem($idNf){
        $nfitem = new NfItem();
        $compra = Model::find($idNf);
        $nfitem->idnf = $idNf;
        $nfitem->idempresa = $compra->idempresa;
        $nfitem->save();
        return response()->json([
        'success' => $nfitem->idnfitem
        ]);
    }



    public function checkDiaUtil($data) {
        $diaSemana = $data->format('N'); // 1 (segunda-feira) a 7 (domingo)
        return ($diaSemana >= 1 && $diaSemana <= 5); // Verificar se é um dia útil (segunda a sexta-feira)
    }

    public function ajustarDiaUtil($data) {
        // Verificar se o dia é sábado (6) ou domingo (7)
        if ($data->format('N') == 6) {
            // Adicionar 2 dias para ir para a próxima segunda-feira
            $data->modify('-1 days');
        } else {
            // Adicionar 1 dia para ir para a próxima segunda-feira
            $data->modify('-2 day');
        }

        return $data;
    }



    private function convertDate($dateString)
    {
        // Converta a string de data para um objeto DateTime, conforme necessário
    }

    private function convertValue($valueString)
    {
        return floatval(str_replace(['R$', ','], ['', '.'], $valueString));
    }

}
