<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Agencia;
use Illuminate\Http\Request;
use App\Models\NfParcela as Model;
use App\Models\Nf;
use App\Models\Empresa;
use App\Models\Pessoa;
use App\Models\TipoProdutoServico;
use App\Models\Veiculo;
use App\Models\FormaPagamento;
use App\Services\NfParcelaService;

use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\DB;

class NfParcelaCreditoController extends Controller
{
    public function __construct(
        protected NfParcelaService $nfParcelaService
    ) {}

    public function index(
            Request $request,
            Empresa $empresaModel,
            Pessoa $pessoaModel,
            TipoProdutoServico $tipoProdutoServicoModel,
            Nf $nfModel,
            FormaPagamento $formaPagamentoModel
        )
    {
        // Use o método `with()` para carregar as relações necessárias e faça o select para renomear o campo status
        $parcelas = Model::query()
            ->select('nfparcela.*') // Renomeando o status de nfparcela
            ->join('empresa', 'empresa.idempresa', '=', 'nfparcela.idempresa') // Join com a tabela empresa
            ->with(['empresa', 'nf.fornecedor', 'nf.prodservtipo']) // Eager loading das relações necessárias
            ->NaoPedido()
            ->credito()
            ->where('nfparcela.status', '!=', 'INATIVO')
            ->orderBy('datavencimento', 'desc')
            ->paginate(100);

        // Aplicação dinâmica de filtros com base nos campos
        $campos = $request->input('campos', []);
        $operadores = $request->input('operadores', []);
        $valores = $request->input('valores', []);

        if (count($campos) === count($operadores) && count($operadores) === count($valores)) {
            $parcelas->where(function($mainQuery) use ($campos, $operadores, $valores) {
                foreach ($campos as $index => $campo) {
                    $operador = $operadores[$index];
                    $valor = $valores[$index];

                    // Define o operador com base na entrada do usuário
                    switch ($operador) {
                        case '=':
                        case '!=':
                        case '>':
                        case '<':
                        case '>=':
                        case '<=':
                            $mainQuery->where($campo, $operador, $valor);
                            break;
                        case 'like':
                            $mainQuery->where($campo, 'like', '%' . $valor . '%');
                            break;
                        case 'in':
                            $valoresArray = explode(',', $valor);
                            $mainQuery->whereIn($campo, $valoresArray);
                            break;
                        case 'between':
                            $rangeValues = explode(',', $valor);
                            if (count($rangeValues) === 2) {
                                $mainQuery->whereBetween($campo, [$rangeValues[0], $rangeValues[1]]);
                            }
                            break;
                        default:
                            break;
                    }
                }
            });
        }

        // Calcular os valores totais com o novo nome do campo status
        $totalValue = Model::NaoPedido()
            ->credito()
            ->whereIn('nfparcela.status', ['ATIVO', 'CONCLUÍDO']) // Aqui o status ainda é o nome original no banco
            ->sum('valor');

        $totalValueCreditoAtivo = Model::NaoPedido()
            ->credito()
            ->where('nfparcela.status', 'ATIVO') // O status original é usado no cálculo
            ->sum('valor');

        $totalValueCreditoConcluido = Model::NaoPedido()
            ->credito()
            ->where('nfparcela.status', 'CONCLUÍDO') // O status original é usado no cálculo
            ->sum('valor');

        $empresas = $empresaModel
            ->orderBy('nomefantasia', 'asc')
            ->get();

        $pessoas = $pessoaModel
            ->Cliente()
            ->get();

        $nfs = $nfModel
            ->Venda()
            ->get();

        $tiposProdutoServico = $tipoProdutoServicoModel
            ->orderBy('prodservtipo')
            ->get();

        $formaPagamento = $formaPagamentoModel->orderBy('formapagamento')->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.nfparcelascredito.index',
            compact(
                'parcelas',
                'statusOptions',
                'totalValue',
                'totalValueCreditoAtivo',
                'totalValueCreditoConcluido',
                'empresas',
                'pessoas',
                'tiposProdutoServico',
                'nfs',
                'formaPagamento'
            ));
    }

    public function fetch(Request $request)
    {
        $perPage = $request->input('per_page', 100);

        // Definir os campos e suas classes CSS
        $fields = [
            'ID' => [
                'key' => 'idnfparcela',
                'class' => 'text-center text-xs',
            ],
            'EMPRESA' => [
                'key' => 'empresa.sigla',
                'class' => 'text-center text-xs',
            ],
            'NF' => [
                'key' => 'nf.idnf',
                'class' => 'text-center text-xs',
            ],
            'FORNECEDOR' => [
                'key' => 'nf.fornecedor.razaosocial',
                'class' => 'text-sm',
            ],
            'TIPO' => [
                'key' => 'nf.prodservtipo.prodservtipo',
                'class' => 'text-center text-xs',
            ],
            'PARCELA' => [
                'key' => fn($parcela) => "$parcela->parcela / {$parcela->nf->qtdparcela}",
                'class' => 'text-center text-xs',
            ],
            'AGÊNCIA' => [
                'key' => fn($parcela) => $parcela->agencia->nome ?? '-',
                'class' => 'text-center text-xs',
            ],
            'VALOR' => [
                'key' => fn($parcela) => "R$ ". number_format($parcela->valor, 2, ',', '.'),
                'class' => 'text-end text-sm fw-bold text-success',
            ],
            'VENCIMENTO' => [
                'key' => fn($parcela) => Carbon::parse($parcela->datavencimento)->format('d/m/y'),
                'class' => 'text-center text-xs fw-bold',
            ],
            'PAGAMENTO' => [
                'key' => fn($parcela) => $parcela->datapagamento
                    ? Carbon::parse($parcela->datapagamento)->format('d/m/y')
                    : '-',
                'class' => 'text-center text-xs fw-bold',
            ],
            'STATUS' => [
                'key' => fn($parcela) => view('partials.status-progress', [
                    'status' => $parcela->status,
                    'id' => $parcela->idnfparcela,
                    'options' => [
                        ['percentage' => 50, 'status' => 'INATIVO', 'tooltip' => 'Inativo', 'class' => 'bg-danger'],
                        ['percentage' => 50, 'status' => 'ATIVO', 'tooltip' => 'Ativo', 'class' => 'bg-info'],
                        ['percentage' => 50, 'status' => 'CONCLUÍDO', 'tooltip' => 'Concluído', 'class' => 'bg-success'],
                    ]
                ])->render(),
                'class' => 'no-clicked pe-3',
            ],
        ];

        // Buscar os dados paginados
        $parcelas = Model::query()
            ->select('nfparcela.*') // Renomeando o status de nfparcela
            ->join('empresa', 'empresa.idempresa', '=', 'nfparcela.idempresa') // Join com a tabela empresa
            ->with(['empresa', 'nf.fornecedor', 'nf.prodservtipo']) // Eager loading das relações necessárias
            ->NaoPedido()
            ->credito()
            ->where('nfparcela.status', '!=', 'INATIVO')
            ->orderBy('datavencimento', 'desc')
            ->paginate($perPage, ['*'], 'page', $request->input('page', 2));

        // Montar os dados processados para o frontend
        $processedData = $parcelas->map(function ($parcela) use ($fields) {
            $row = [
                'url' => route('admin.nfparcelascredito.edit', $parcela->idnfparcela),
            ];
            foreach ($fields as $label => $field) {
                $value = is_callable($field['key'])
                    ? $field['key']($parcela) // Executar a closure
                    : data_get($parcela, $field['key']); // Obter o valor diretamente
                $row[$label] = [
                    'value' => $value,
                    'class' => $field['class'],
                ];
            }
            return $row;
        });

        // Retornar os dados processados como JSON
        return response()->json([
            'columns' => array_keys($fields),
            'data' => $processedData,
            'next_page_url' => $parcelas->nextPageUrl(),
        ]);
    }

    public function create()
    {

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];
        return view('admin.nfparcelascredito.create');
    }

    public function store(Request $request)
    {
        $parcelaCreate = $request->all();

        try {
            Model::create($parcelaCreate);

            return redirect()
                ->route('admin.nfparcelascredito.index')
                ->with('success', 'Parcela criada com sucesso!');
        } catch (Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->with('error', $e->getMessage());
        }
    }

    public function edit($idNfParcela, Nf $NfModel, Empresa $empresaModel, Pessoa $pessoaModel, FormaPagamento $formaPagamentoModel, TipoProdutoServico $tipoProdutoServicoModel, Veiculo $veiculoModel, Agencia $agenciaModel)
    {
        $parcela = Model::find($idNfParcela);

        $nf = $NfModel
        ->where('idnf', $parcela->idnf) // Filtra pelos registros com o idnf relacionado
        ->first();

        $empresa = $empresaModel
        ->where('idempresa', $nf->idempresa)
        ->first();

        $pessoa = $pessoaModel
        ->where('idpessoa', $nf->idpessoa)
        ->first();

        $veiculo = $veiculoModel
        ->where('idveiculo', $nf->idveiculo)
        ->first();

        $tipoProdutoServico = $tipoProdutoServicoModel
        ->where('idprodservtipo', $nf->idprodservtipo)
        ->first();

        $formaPagamento = $formaPagamentoModel
        ->where('idformapagamento', $nf->idformapagamento)
        ->first();

        if ($parcela->status == 'CONCLUÍDO'){
            $bg_status = "bg-success";
        }elseif($parcela->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        $agencias = $agenciaModel
        ->orderBy('nome')
        ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.nfparcelascredito.edit')->with(compact('parcela', 'nf', 'empresa', 'pessoa', 'veiculo','tipoProdutoServico','formaPagamento','bg_status', 'statusOptions', 'agencias'));
    }

    public function update($idNfParcela, Request $request)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $parcela = Model::find($idNfParcela);
        $nfParcelaUpdate = $request->all();

        if ($redirect == true) {
            $camposMonetarios = ['valor'];

             // Trata os campos monetários
             foreach ($camposMonetarios as $campo) {
                if (isset($nfParcelaUpdate[$campo])) {
                    $nfParcelaUpdate[$campo] = str_replace('.', '', $nfParcelaUpdate[$campo]);
                    $nfParcelaUpdate[$campo] = str_replace(',', '.', $nfParcelaUpdate[$campo]);
                    $nfParcelaUpdate[$campo] = floatval($nfParcelaUpdate[$campo]);
                }
            }
        }

        DB::beginTransaction();
        try {
            $novoStatus = $nfParcelaUpdate['status'] ?? $parcela->status;
            $statusAtual = $parcela->status;
            $parcela->update($nfParcelaUpdate);

            if (isset($parcela->idagencia)) {
                if ($statusAtual !== 'CONCLUÍDO' && $novoStatus === 'CONCLUÍDO') {
                    $this->nfParcelaService->concluirParcela($parcela, false, true, $parcela->idagencia);
                }

                if ($statusAtual === 'CONCLUÍDO' && $novoStatus === 'ATIVO' || $statusAtual === 'CONCLUÍDO' && $novoStatus === 'INATIVO') {
                    $this->nfParcelaService->estornarParcela($parcela, false, true, $parcela->idagencia);
                }
            }

            if ($redirect == false) {
                DB::commit();
                return response()->json($parcela->status);
            }

            DB::commit();
            return redirect()->back()->with('success', 'Parcela atualizada!');

        } catch (Exception $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->with('error', $e->getMessage())
                ->withInput();
        }
    }

    public function destroy($idNfParcela)
    {
        $parcela = Model::find($idNfParcela);

        try {

            $parcela->delete();

            return redirect()
                ->route('admin.nfparcelascredito.index')
                ->with('success', 'Parcela removida!');
        } catch (Exception $e) {
            return redirect()
                ->back()
                ->with('error', $e->getMessage())
                ->withInput();
        }
    }

    public function search(
            Request $request,
            Empresa $empresaModel,
            Pessoa $pessoaModel,
            Nf $nfModel,
            TipoProdutoServico $tipoProdutoServicoModel,
            FormaPagamento $formaPagamentoModel
        )
    {
        // Cria a query inicial com joins e filtragem básica
        $creditos = Model::select('nfparcela.*', 'nf.idpessoa', 'nf.idprodservtipo')
            ->join('nf', 'nfparcela.idnf', '=', 'nf.idnf')
            ->with(['nf.pessoa', 'nf.prodservtipo', 'empresa'])
            ->where('nfparcela.tipo', 'C')
            ->when($request->status, function ($query) use ($request) {
                return $query->where('nfparcela.status', $request->status);
            });

        // Aplica filtros de busca por termo se o campo `search` estiver preenchido
        if ($request->filled('search')) {
            $searchTerm = '%' . $request->search . '%';
            $creditos->where(function ($query) use ($searchTerm) {
                $query->whereHas('nf.pessoa', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('nome', 'like', $searchTerm);
                })->orWhereHas('nf.prodservtipo', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('prodservtipo', 'like', $searchTerm);
                })->orWhereHas('empresa', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('sigla', 'like', $searchTerm);
                });
            });
        }

        // Aplica filtros dinâmicos com base nos campos
        filter($request, $creditos);

        // Cálculo dos valores totais
        $totalValue = $creditos->clone()->sum('valor');
        $totalValueCreditoAtivo = $creditos->clone()->where('nfparcela.status', 'ativo')->sum('valor');
        $totalValueCreditoConcluido = $creditos->clone()->where('nfparcela.status', 'concluido')->sum('valor');

        // Obter os resultados da busca
        $parcelas = $creditos->paginate(100);

        // Gerenciamento de mensagens de sessão
        session()->forget(['error', 'success']);
        session()->flash($parcelas->count() ? 'success' : 'error', $parcelas->count() ? $parcelas->count() . ' valores encontrado(s).' : 'Nenhum registro encontrado.');

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
        ];

        $tiposProdutoServico = $tipoProdutoServicoModel->orderBy('prodservtipo')->get();
        $empresas = $empresaModel->orderBy('nomefantasia', 'asc')->get();
        $pessoas = $pessoaModel->Cliente()->get();
        $formaPagamento = $formaPagamentoModel->orderBy('formapagamento')->get();

        return view('admin.nfparcelascredito.index',
            compact(
                'parcelas',
                'totalValue',
                'totalValueCreditoAtivo',
                'totalValueCreditoConcluido',
                'statusOptions',
                'tiposProdutoServico',
                'empresas',
                'pessoas',
                'formaPagamento'
            ));
    }
}
