<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Funcionalidade;
use App\Models\Plano;
use App\Models\PlanoFuncionalidade;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PlanoFuncionalidadeController extends Controller
{

    public function index(PlanoFuncionalidade $planoFuncionalidadeModel)
    {
        $planoFuncionalidades = $planoFuncionalidadeModel
            ->with('plano', 'funcionalidade')
            ->orderBy('criadoem')
            ->paginate(100);

        return view('admin.plano-funcionalidade.index', compact('planoFuncionalidades'));
    }

    public function fetch(Request $request)
    {
        $perPage = $request->input('per_page', 100);

        // Definir os campos e suas classes CSS
        $fields = [
            'ID' => [
                'key' => 'idplanofuncionalidade',
                'class' => 'text-center text-xs',
            ],
            'PLANO' => [
                'key' => 'plano.nome',
                'class' => 'text-center text-xs',
            ],
            'FUNCIONALIDADE' => [
                'key' => 'funcionalidade.nome',
                'class' => 'text-center text-xs',
            ],
            'QUANTIDADE' => [
                'key' => 'qtd',
                'class' => 'text-center text-xs',
            ],
            'STATUS' => [
                'key' => fn($planoFuncionalidade) => view('partials.status-switch', [
                    'status' => $planoFuncionalidade->status,
                    'id' => $planoFuncionalidade->idplanofuncionalidade,
                ])->render(),
                'class' => 'no-clicked d-flex align-items-center justify-content-center',
            ],
        ];

        // Buscar os dados paginados
        $planoFuncionalidades = PlanoFuncionalidade::with('plano', 'funcionalidade')
            ->orderBy('criadoem')
            ->paginate($perPage, ['*'], 'page', $request->input('page', 2));

        // Montar os dados processados para o frontend
        $processedData = $planoFuncionalidades->map(function ($planoFuncionalidade) use ($fields) {
            $row = [
                'url' => route('admin.forma-pagamento.edit', $planoFuncionalidade->idplanofuncionalidade),
            ];
            foreach ($fields as $label => $field) {
                $value = is_callable($field['key'])
                    ? $field['key']($planoFuncionalidade) // Executar a closure
                    : data_get($planoFuncionalidade, $field['key']); // Obter o valor diretamente
                $row[$label] = [
                    'value' => $value,
                    'class' => $field['class'],
                ];
            }
            return $row;
        });

        // Retornar os dados processados como JSON
        return response()->json([
            'columns' => array_keys($fields),
            'data' => $processedData,
            'next_page_url' => $planoFuncionalidades->nextPageUrl(),
        ]);
    }

    public function create(Plano $planoModel, Funcionalidade $funcionalidadeModel)
    {
        $planos = $planoModel
            ->orderBy('nome')
            ->get();

        $funcionalidades = $funcionalidadeModel
            ->orderBy('nome')
            ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.plano-funcionalidade.create', compact('planos', 'funcionalidades', 'statusOptions'));
    }

    public function store(Request $request, PlanoFuncionalidade $planoFuncionalidadeModel)
    {
        $planoFuncionalidadeRequest = $request->all();

        DB::beginTransaction();
        try {
            $planoFuncionalidade = $planoFuncionalidadeModel->create($planoFuncionalidadeRequest);
            DB::commit();
            return redirect()
                ->route('admin.plano-funcionalidade.edit', $planoFuncionalidade)
                ->with('success', 'Plano funcionalidade criado com sucesso!');
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()
                ->route('admin.plano-funcionalidade.create')
                ->with('error', $e->getMessage());
        }
    }

    public function edit(PlanoFuncionalidade $planoFuncionalidade, Plano $planoModel, Funcionalidade $funcionalidadeModel)
    {
        $planos = $planoModel
            ->orderBy('nome')
            ->get();

        $funcionalidades = $funcionalidadeModel
            ->orderBy('nome')
            ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
        ];

        if ($planoFuncionalidade->status == 'ATIVO'){
            $bg_status = "bg-success";
        }elseif($planoFuncionalidade->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        return view('admin.plano-funcionalidade.edit', compact('planoFuncionalidade', 'planos', 'funcionalidades', 'statusOptions', 'bg_status'));
    }

    public function update(Request $request, PlanoFuncionalidade $planoFuncionalidade)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $planoFuncionalidadeRequest = $request->all();

        DB::beginTransaction();
        try {
            $planoFuncionalidade->update($planoFuncionalidadeRequest);

            if ($redirect == false) {
                DB::commit();
                return response()->json($planoFuncionalidade->status);
            }

            DB::commit();
            return redirect()
                ->route('admin.plano-funcionalidade.edit', $planoFuncionalidade)
                ->with('success', 'Plano funcionalidade atualizado com sucesso!');
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()
                ->route('admin.plano-funcionalidade.edit', $planoFuncionalidade)
                ->with('error', $e->getMessage());
        }
    }

    public function destroy(PlanoFuncionalidade $planoFuncionalidade)
    {
        DB::beginTransaction();
        try {
            $planoFuncionalidade->delete();
            DB::commit();
            return redirect()
                ->route('admin.plano-funcionalidade.index')
                ->with('success', 'Plano funcionalidade excluido com sucesso!');
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()
                ->route('admin.plano-funcionalidade.index')
                ->with('error', $e->getMessage());
        }
    }

    public function search(Request $request, PlanoFuncionalidade $planoFuncionalidade)
    {
        // Definindo os filtros
        $query = $request->search;

        $planoFuncionalidades = $planoFuncionalidade
            ->with('plano', 'funcionalidade')
            ->whereHas('plano', function($subQuery) use ($query) {
                $subQuery->where('nome', 'like', "%{$query}%");
            })
            ->orWhereHas('funcionalidade', function($subQuery) use ($query) {
                $subQuery->where('nome', 'like', "%{$query}%");
            })
            ->orWhere('status', 'like', "%{$query}%")
            ->get();


        return view('admin.plano-funcionalidade.index', compact('planoFuncionalidades'));
    }
}
