<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use App\Models\Pessoa;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RepresentanteController extends Controller
{
    public function index()
    {
        $representantes = Pessoa::with('empresa')
            ->representante()
            ->paginate(100);

        $statusOptions = [
            'INATIVO' => 'Inativo',
            'SUSPENSO' => 'Suspenso',
            'POTENCIAL' => 'Potencial',
            'ATIVO' => 'Ativo',
        ];

        return view('admin.representantes.index', compact('representantes', 'statusOptions'));
    }

    public function fetch(Request $request)
    {
        $perPage = $request->input('per_page', 100);

        // Definir os campos e suas classes CSS
        $fields = [
            'ID' => [
                'key' => 'idpessoa',
                'class' => 'text-center text-xs',
            ],
            'EMPRESA' => [
                'key' => 'empresa.sigla',
                'class' => 'text-center text-xs',
            ],
            'NOME' => [
                'key' => 'empresa.nome',
                'class' => 'text-start text-xs',
            ],
            'RAZÃO SOCIAL' => [
                'key' => 'razaosocial',
                'class' => 'text-center text-xs',
            ],
            'CPF / CNPJ' => [
                'key' => 'cpfcnpj',
                'class' => 'text-center text-xs',
            ],
            'STATUS' => [
                'key' => fn($representante) => view('partials.status-progress', [
                    'status' => $representante->status,
                    'id' => $representante->idpessoa,
                    'options' => [
                        ['percentage' => 25, 'status' => 'INATIVO', 'tooltip' => 'Inativo', 'class' => 'bg-danger'],
                        ['percentage' => 25, 'status' => 'SUSPENSO', 'tooltip' => 'Suspenso', 'class' => 'bg-warning'],
                        ['percentage' => 25, 'status' => 'POTENCIAL', 'tooltip' => 'Potencial', 'class' => 'bg-info'],
                        ['percentage' => 25, 'status' => 'ATIVO', 'tooltip' => 'Ativo', 'class' => 'bg-success'],
                    ]
                ])->render(),
                'class' => 'no-clicked pe-3',
            ],
        ];

        // Buscar os dados paginados
        $representantes = Pessoa::with('empresa')
            ->representante()
            ->paginate($perPage, ['*'], 'page', $request->input('page', 2));

        // Montar os dados processados para o frontend
        $processedData = $representantes->map(function ($representante) use ($fields) {
            $row = [
                'url' => route('admin.representantes.edit', $representante->idpessoa),
            ];
            foreach ($fields as $label => $field) {
                $value = is_callable($field['key'])
                    ? $field['key']($representante) // Executar a closure
                    : data_get($representante, $field['key']); // Obter o valor diretamente
                $row[$label] = [
                    'value' => $value,
                    'class' => $field['class'],
                ];
            }
            return $row;
        });

        // Retornar os dados processados como JSON
        return response()->json([
            'columns' => array_keys($fields),
            'data' => $processedData,
            'next_page_url' => $representantes->nextPageUrl(),
        ]);
    }

    public function create()
    {
        $empresas = Empresa::orderBy('razaosocial')->get();

        $statusOptions = [
            'INATIVO' => 'Inativo',
            'SUSPENSO' => 'Suspenso',
            'POTENCIAL' => 'Potencial',
            'ATIVO' => 'Ativo',
        ];

        return view('admin.representantes.create', compact('empresas', 'statusOptions'));
    }

    public function store(Request $request)
    {
        $representanteRequest = $request->all();

        DB::beginTransaction();
        try {
            $representante = Pessoa::create($representanteRequest);
            DB::commit();
            return redirect()->route('admin.representantes.edit', $representante)
                    ->with('success', 'Representante criado com sucesso!');
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function edit(Pessoa $representante)
    {
        $empresas = Empresa::orderBy('razaosocial')->get();

        $statusOptions = [
            'INATIVO' => 'Inativo',
            'SUSPENSO' => 'Suspenso',
            'POTENCIAL' => 'Potencial',
            'ATIVO' => 'Ativo',
        ];

        if ($representante->status == 'ATIVO') {
            $bg_status = "bg-success";
        } elseif ($representante->status == 'POTENCIAL') {
            $bg_status = "bg-info";
        } elseif ($representante->status == 'SUSPENSO') {
            $bg_status = "bg-warning";
        } elseif ($representante->status == 'INATIVO') {
            $bg_status = "bg-danger";
        }

        return view('admin.representantes.edit', compact('representante', 'empresas', 'statusOptions', 'bg_status'));
    }

    public function update(Request $request, Pessoa $representante)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $representanteRequest = $request->all();

        DB::beginTransaction();
        try {
            $representante->update($representanteRequest);
            DB::commit();

            if ($redirect == false) {
                return response()->json($representante->status);
            }

            return redirect()->route('admin.representantes.edit', $representante);
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function destroy(Pessoa $representante)
    {
        DB::beginTransaction();
        try {
            $representante->delete();
            DB::commit();
            return redirect()->route('admin.representantes.index')
                    ->with('success', 'Representante excluido com sucesso!');
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
}
