<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use App\Models\Pessoa;
use App\Models\Tag;
use App\Models\TagTipo;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TagController extends Controller
{
    public function index(Tag $tagModel)
    {
        $tags = $tagModel
            ->with('tagTipo', 'empresa')
            ->where('status', '!=', 'INATIVO')
            ->orderBy('tag')
            ->paginate(100);

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'MANUTENCAO' => 'Manutenção',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.tags.index', compact('tags', 'statusOptions'));
    }

    public function fetch(Request $request)
    {
        $perPage = $request->input('per_page', 100);

        // Definir os campos e suas classes CSS
        $fields = [
            'ID' => [
                'key' => 'idtag',
                'class' => 'text-center text-xs',
            ],
            'EMPRESA' => [
                'key' => 'empresa.sigla',
                'class' => 'text-center text-xs',
            ],
            'TIPO DE TAG' => [
                'key' => fn($tag) => $tag->tagTipo->tagtipo,
                'class' => 'text-center text-xs',
            ],
            'TAG' => [
                'key' => 'tag',
                'class' => 'text-center text-xs',
            ],
            'STATUS' => [
                'key' => fn($item) => view('partials.status-progress', [
                    'status' => $item->status,
                    'id' => $item->idformapagamento,
                    'options' => [
                        ['percentage' => 50, 'status' => 'INATIVO', 'tooltip' => 'Inativo', 'class' => 'bg-danger'],
                        ['percentage' => 50, 'status' => 'MANUTENCAO', 'tooltip' => 'Manutenção', 'class' => 'bg-warning'],
                        ['percentage' => 50, 'status' => 'ATIVO', 'tooltip' => 'Ativo', 'class' => 'bg-success'],
                    ]
                ])->render(),
                'class' => 'no-clicked pe-3',
            ],
        ];

        // Buscar os dados paginados
        $tags = Tag::with('tagTipo', 'empresa')
            ->where('status', '!=', 'INATIVO')
            ->orderBy('tag')
            ->paginate($perPage);

        // Montar os dados processados para o frontend
        $processedData = $tags->map(function ($tag) use ($fields) {
            $row = [
                'url' => route('admin.tags.edit', $tag->idtag),
            ];
            foreach ($fields as $label => $field) {
                $value = is_callable($field['key'])
                    ? $field['key']($tag) // Executar a closure
                    : data_get($tag, $field['key']); // Obter o valor diretamente
                $row[$label] = [
                    'value' => $value,
                    'class' => $field['class'],
                ];
            }
            return $row;
        });

        // Retornar os dados processados como JSON
        return response()->json([
            'columns' => array_keys($fields),
            'data' => $processedData,
            'next_page_url' => $tags->nextPageUrl(),
        ]);
    }

    public function create(TagTipo $tagTipoModel, Empresa $empresaModel, Pessoa $pessoaModel)
    {
        $tagTipos = $tagTipoModel
            ->orderBy('tagtipo')
            ->get();

        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

        $pessoas = $pessoaModel
            ->orderBy('nome')
            ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'MANUTENCAO' => 'Manutenção',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.tags.create', compact('tagTipos', 'empresas', 'statusOptions', 'pessoas'));
    }

    public function store(Request $request, Tag $tagModel)
    {
        $tagRequest = $request->all();

        try {
            $tag = $tagModel->create($tagRequest);

            return redirect()
                ->route('admin.tags.edit', $tag)
                ->with('success', 'Tag criada com sucesso!');
        } catch (Exception $e) {

            return redirect()
                ->back()
                ->withInput()
                ->with('error', $e->getMessage());
        }
    }

    public function edit($idTag, Tag $tagModel, TagTipo $tagTipoModel, Empresa $empresaModel, Pessoa $pessoaModel)
    {
        if (! $tag = $tagModel->find($idTag)) {
            return redirect()->route('admin.tags.index')->with('error', 'Tag não encontrada!');
        }

        $tagTipos = $tagTipoModel
            ->with('empresa')
            ->orderBy('tagtipo')
            ->get();

        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

        $pessoas = $pessoaModel
            ->orderBy('nome')
            ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'MANUTENÇÃO' => 'Manutenção',
            'INATIVO' => 'Inativo',
        ];

        if ($tagModel->status == 'ATIVO'){
            $bg_status = "bg-success";
        }elseif($tagModel->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-warning";
        }

        return view('admin.tags.edit', compact('tag', 'tagTipos', 'empresas', 'statusOptions', 'bg_status', 'pessoas'));
    }

    public function update(Tag $tag, Request $request)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $tagRequest = $request->only('idtagtipo', 'idempresa', 'tag', 'nome', 'status', 'descricao',  'criadopor', 'alteradopor');

        DB::beginTransaction();

        try {
            $tag->update($tagRequest);

            if ($redirect == false) {
                DB::commit();
                return response()->json($tag->status);
            }

            DB::commit();

            return redirect()->back()->with('success', 'Tag atualizada com sucesso!');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withInput()->with('error', $e->getMessage());
        }
    }

    public function destroy(Tag $tag)
    {
        if (! $tag) {
            return redirect()->route('admin.tag.index')->with('error', 'Tag não encontrada!');
        }

        try {
            $tag->delete();

            return redirect()->route('admin.tag.index')->with('success', 'Tag excluída com sucesso!');

        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('error', $e->getMessage());
        }

    }

    public function search(Request $request, Tag $tagModel)
    {
        // Chamando o método filter com os filtros e paginando o resultado
        $tags = $tagModel
            ->with('tagTipo', 'empresa')
            ->where('tag', 'LIKE', '%' . $request->search . '%')
            ->whereHas('empresa', function ($q) use ($request) {
                $q->where('razaosocial', 'LIKE', '%' . $request->search . '%');
            })
            ->whereHas('tagTipo', function ($q) use ($request) {
                $q->where('tagtipo', 'LIKE', '%' . $request->search . '%');
            })
            ->where('status', $request->status)
            ->orderBy('tag')
            ->orderBy('tag.idempresa')
            ->orderBy('tag.idtagtipo')
            ->paginate(100);

        // Limpa as mensagens anteriores da sessão
        session()->forget('error');
        session()->forget('success');

        // Define mensagens de sucesso ou erro
        if ($tags->isNotEmpty()) {
            session()->flash('success', $tags->count() . ' valores encontrado(s).');
        } else {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        // Define as opções de status
        $statusOptions = [
            'ATIVO' => 'Ativo',
            'MANUTENCAO' => 'Manutenção',
            'INATIVO' => 'Inativo',
        ];

        // Retorna a view com os resultados
        return view('admin.tags.index', compact('tags', 'statusOptions'));
    }
}
