<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use App\Models\Tag;
use App\Models\TagTipo;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class TagTipoController extends Controller
{
    public function index(TagTipo $tagTipoModel)
    {
        $tagTipos = $tagTipoModel
            ->with('empresa', 'criador')
            ->orderBy('tagtipo')
            ->paginate(100);

        return view('admin.tagtipos.index', compact('tagTipos'));
    }

    public function fetch(Request $request)
    {
        $perPage = $request->input('per_page', 100);

        // Definir os campos e suas classes CSS
        $fields = [
            'ID' => [
                'key' => 'idtagtipo',
                'class' => 'text-center text-xs',
            ],
            'EMPRESA' => [
                'key' => 'empresa.sigla',
                'class' => 'text-center text-xs',
            ],
            'TIPO DE TAG' => [
                'key' => 'tagtipo',
                'class' => 'text-center text-xs',
            ],
            'CRIADOR' => [
                'key' => fn($tagTipo) => $tagTipo->criador->name ?? $tagTipo->criadopor ?? '-',
                'class' => 'text-center text-xs',
            ],
            'STATUS' => [
                'key' => fn($tagTipo) => view('partials.status-switch', [
                    'status' => $tagTipo->status,
                    'id' => $tagTipo->idtagtipo,
                ])->render(),
                'class' => 'no-clicked d-flex align-items-center justify-content-center',
            ]
        ];

        // Buscar os dados paginados
        $tagTipos = TagTipo::with('empresa', 'criador')
            ->orderBy('tagtipo')
            ->paginate($perPage, ['*'], 'page', $request->input('page', 2));

        // Montar os dados processados para o frontend
        $processedData = $tagTipos->map(function ($tagTipo) use ($fields) {
            $row = [
                'url' => route('admin.tagtipos.edit', $tagTipo->idtagtipo),
            ];
            foreach ($fields as $label => $field) {
                $value = is_callable($field['key'])
                    ? $field['key']($tagTipo) // Executar a closure
                    : data_get($tagTipo, $field['key']); // Obter o valor diretamente
                $row[$label] = [
                    'value' => $value,
                    'class' => $field['class'],
                ];
            }
            return $row;
        });

        // Retornar os dados processados como JSON
        return response()->json([
            'columns' => array_keys($fields),
            'data' => $processedData,
            'next_page_url' => $tagTipos->nextPageUrl(),
        ]);
    }

    public function create(Empresa $empresaModel)
    {
        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.tagtipos.create', compact('empresas', 'statusOptions'));
    }

    public function store(Request $request, TagTipo $tagTipoModel)
    {
        $tagTipoRequest = $request->all();

        $campos = [
            'marca',
            'modelo',
            'anofabricacao',
            'numeroserie',
            'altura',
            'largura',
            'comprimento',
            'peso',
            'material',
            'combustivel',
            'datamanutencao',
            'dataproximamanutencao',
            'garantia',
            'voltagem',
            'processador',
            'memoriaram',
            'cor',
            'sistemaoperacional',
        ];

        // Tratamento de campos
        foreach ($campos as $campo) {
            if (isset($tagTipoRequest[$campo]) && $tagTipoRequest[$campo] == 'on') {
                $tagTipoRequest[$campo] = 1;
            } else {
                $tagTipoRequest[$campo] = 0;
            }
        }

        DB::beginTransaction();
        try {
            $tagTipoModel->create($tagTipoRequest);

            return redirect()
                ->back()
                ->with('success', 'Tag Tipo criado com sucesso');
        } catch (Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->with('error', $e->getMessage());
        }
    }

    public function edit($idTagTipo, Empresa $empresaModel)
    {
        if (! $tagTipo = TagTipo::find($idTagTipo)) {
            return redirect()
                ->back()
                ->with('error', 'Tag Tipo não encontrado');
        }

        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
        ];

        if ($tagTipo->status == 'ATIVO'){
            $bg_status = "bg-success";
        }elseif($tagTipo->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-warning";
        }

        return view('admin.tagtipos.edit', compact('tagTipo', 'empresas', 'statusOptions', 'bg_status'));
    }

    public function update($idTagTipo, Request $request, Tag $tagModel)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $tagTipoRequest = $request->all();

        // Verifica se o Tag Tipo existe
        if (! $tagTipo = TagTipo::find($idTagTipo)) {
            return redirect()
                ->back()
                ->with('error', 'Tag Tipo não encontrada');
        }

        // Recupera todas as tags associadas ao Tag Tipo e verifica se estão ativas
        $tagsAtivas = $tagModel->where('idtagtipo', $tagTipo->idtagtipo) // Use o 'idtagtipo' do modelo encontrado
                            ->where('status', 'ATIVO')
                            ->get();

        // Impede a atualização para INATIVO se houver tags ativas associadas
        if ($tagsAtivas->count() > 0 && $tagTipoRequest['status'] === 'INATIVO') {
            if ($redirect == true) {
                return redirect()
                    ->back()
                    ->with('error', 'Existem tags ativas associadas a este Tag Tipo. Portanto, o Tag Tipo deve permanecer ativo.');
            }

            return response()->json(['error' => 'Existem tags ativas associadas a este Tag Tipo. Portanto, o Tag Tipo deve permanecer ativo.']);
        }

        if ($redirect == true) {
            $campos = [
                'marca',
                'modelo',
                'anofabricacao',
                'numeroserie',
                'altura',
                'largura',
                'comprimento',
                'peso',
                'material',
                'combustivel',
                'datamanutencao',
                'dataproximamanutencao',
                'garantia',
                'voltagem',
                'processador',
                'memoriaram',
                'cor',
                'sistemaoperacional',
            ];

            // Tratamento de campos
            foreach ($campos as $campo) {
                if (isset($tagTipoRequest[$campo]) && $tagTipoRequest[$campo] == 'on') {
                    $tagTipoRequest[$campo] = 1;
                } else {
                    $tagTipoRequest[$campo] = 0;
                }
            }
        }

        DB::beginTransaction();
        try {
            // Realiza a atualização
            $tagTipo->update($tagTipoRequest);

            if ($redirect == false) {
                DB::commit();
                return response()->json($tagTipo->status);
            }

            DB::commit();
            return redirect()
                ->back()
                ->with('success', 'Tag Tipo alterado com sucesso');

        } catch (Exception $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->with('error', $e->getMessage());
        }
    }


    public function destroy($idTagTipo, TagTipo $tagTipoModel)
    {
        if (! $tagTipo = $tagTipoModel->find($idTagTipo)) {
            return redirect()
                ->back()
                ->with('error', 'Tag Tipo não encontrada');
        }

        try {
            $tagTipo->delete();

            return redirect()
                ->back()
                ->with('success', 'Excluído com sucesso');
        } catch (Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->with('error', $e->getMessage());
        }
    }

    public function search(Request $request)
    {
        $tagTipos = TagTipo::query()
        ->with('empresa')
        ->where('tagtipo', 'like', '%' . $request->search . '%')
        ->orderBy('tagtipo.tagtipo')
        ->paginate(100);

        $size = $tagTipos->count();

        // Colocando mensagens na session
        session()->forget('error');
        session()->forget('success');
        if ($size && $size > 0 )
        {
            session()->flash('success', $size.' valor(es) encontrado(s).');

        } else
        {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        return view('admin.tagtipos.index')
                ->with(compact('tagTipos'));
    }
}
