<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Agencia;
use App\Models\GrupoItem;
use App\Models\Nf;
use App\Models\NfItem;
use App\Models\NfParcela;
use App\Models\Pessoa;
use App\Models\ProdutoServico;
use App\Models\TipoProdutoServico;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\STr;

class TransferenciaController extends Controller
{
    public function index()
    {
        $nfs = Nf::with('parcela', 'empresa', 'agencia', 'nfitem')
            ->whereNotNull('transferkey')
            ->paginate(100)
            ->groupBy('transferkey')
            ->map(function ($nfs) {
                return [
                    'nf1' => $nfs->first(),
                    'nf2' => $nfs->skip(1)->first(), // Pega a segunda NF do grupo, se existir
                ];
            });

        $agencias = Agencia::with('empresa')
                            ->orderBy('nome')
                            ->get();

        $centroCustos = GrupoItem::all();

        $statusOptions = [
            'INATIVO' => 'Inativo',
            'SUSPENSO' => 'Suspenso',
            'POTENCIAL' => 'Potencial',
            'ATIVO' => 'Ativo'
        ];

        return view('admin.transferencias.index', compact('nfs', 'agencias', 'centroCustos', 'statusOptions'));
    }

    public function fetch(Request $request)
    {
        $perPage = $request->input('per_page', 100);

        // Definir os campos e suas classes CSS
        $fields = [
            'EMPRESA' => [
                'key' => fn($group) => $group['nf1']->empresa->sigla,
                'class' => 'text-center text-xs',
            ],
            'AGÊNCIA ORIGEM' => [
                'key' => fn($group) => $group['nf1']->agencia->nome,
                'class' => 'text-center text-xs',
            ],
            'AGÊNCIA DESTINO' => [
                'key' => fn($group) => $group['nf2']->agencia->nome,
                'class' => 'text-center text-xs',
            ],
            'VALOR' => [
                'key' => fn($group) => number_format($group['nf1']->valornf, 2, ',', '.'),
                'class' => 'text-center text-xs',
            ],
            'DATA TRANSFERÊNCIA' => [
                'key' => fn($group) => Carbon::parse($group['nf1']->valornf)->format('d/m/y'),
                'class' => 'text-center text-xs',
            ],
        ];

        // Buscar os dados paginados
        $nfs = Nf::with('parcela', 'empresa', 'agencia', 'nfitem')
            ->whereNotNull('transferkey')
            ->paginate($perPage, ['*'], 'page', $request->input('page', 2));

        $nextPage = $nfs->nextPageUrl();

        $nfs->groupBy('transferkey')
            ->map(function ($nfs) {
                return [
                    'nf1' => $nfs->first(),
                    'nf2' => $nfs->skip(1)->first(), // Pega a segunda NF do grupo, se existir
                ];
            });

        // Montar os dados processados para o frontend
        $processedData = $nfs->map(function ($group) use ($fields) {
            $row = [
                'dataAttribute' => [
                    'data-data' => $group['nf1']->dataemissao,
                    'data-valor' => $group['nf1']->valornf,
                    'data-observacao' => $group['nf1']->observacao,
                    'data-agencia-saida' => $group['nf1']->idagencia,
                    'data-agencia-entrada' => $group['nf2']->idagencia,
                    'data-centro-custo' => $group['nf1']->nfitem[0]->idgrupoitem
                ]
            ];
            foreach ($fields as $label => $field) {
                $value = is_callable($field['key'])
                    ? $field['key']($group) // Executar a closure
                    : data_get($group, $field['key']); // Obter o valor diretamente
                $row[$label] = [
                    'value' => $value,
                    'class' => $field['class'],
                ];
            }
            return $row;
        });

        // Retornar os dados processados como JSON
        return response()->json([
            'columns' => array_keys($fields),
            'data' => $processedData,
            'next_page_url' => $nextPage,
        ]);
    }

    public function store(Request $request)
    {
        $transferRequest = $request->all();

        // Tratamento para valor
        $transferRequest['valortransferencia'] = str_replace(['.', ','], ['', '.'], $transferRequest['valortransferencia']);

        // Cria uma chave aleatoria para vincular as NFs
        $transferRequest['chave'] = Str::random(32);

        DB::beginTransaction();
        try {
            // Registra uma NF para ambas as operações. Crédito e Débito
            $operacoes = [
                "D" => [
                    "idagencia" => $transferRequest['agenciasaida'],
                    "tipoPessoa" => "C"
                ],
                "C" => [
                    "idagencia" => $transferRequest['agenciaentrada'],
                    "tipoPessoa" => "F"
                ]];

            foreach ($operacoes as $key => $operacao) {
                $agencia = Agencia::find($operacao['idagencia']);
                $empresa = $agencia->empresa;

                // Busca o cliente/fornecedor padrão da empresa
                $pessoa = Pessoa::where('idempresa', $empresa->idempresa)
                            ->where('tipo', $operacao['tipoPessoa'])
                            ->where('pessoapadrao', true)
                            ->first();

                if (!$pessoa) {
                    $pessoa = EmpresaController::createPersonDefault($empresa, $operacao['tipoPessoa']);
                }

                // Busca o prodservtipo padrão da empresa
                $prodservtipo = TipoProdutoServico::firstOrCreate(
                    ['prodservtipo' => 'Movimentação', 'idempresa' => $empresa->idempresa],
                    [
                        'padrao' => true,
                        'status' => 'ATIVO',
                    ]
                );

                $prodserv = ProdutoServico::firstOrCreate(
                    ['prodserv' => 'Transferência entre contas', 'idempresa' => $empresa->idempresa],
                    [
                        'idprodservtipo' => $prodservtipo->idprodservtipo,
                        'tipo' => 'S',
                        'unidade' => 'UN',
                        'status' => 'ATIVO',
                    ]
                );

                // Verifica se já existe o vínculo entre prodserv e prodservtipo
                if (!$prodserv->tipos()->where('prodservtipo.idprodservtipo', $prodservtipo->idprodservtipo)->exists()) {
                    // Cria o vínculo
                    $prodserv->tipos()->attach($prodservtipo->idprodservtipo);
                }

                $nf = Nf::create([
                    'idpessoa' => $pessoa->idpessoa,
                    'idempresa' => $empresa->idempresa,
                    'idprodservtipo' => $prodservtipo->idprodservtipo,
                    'idagencia' => $agencia->idagencia,
                    'saida' => $key == "D" ? "N" : "V",
                    'tipo' => $key,
                    'dataemissao' => now(),
                    'dataentrada' => $transferRequest['datatransferencia'],
                    'qtdparcela' => 1,
                    'status' => 'CONCLUÍDO',
                    'valornf' => $transferRequest['valortransferencia'],
                    'datavencimento' => $transferRequest['datatransferencia'],
                    'transferkey' => $transferRequest['chave'],
                    'totalnf' => $transferRequest['valortransferencia'],
                    'observacao' => $transferRequest['observacaotransferencia'] ?? null,
                ]);

                NfItem::create([
                    'idnf' => $nf->idnf,
                    'idempresa' => $nf->idempresa,
                    'idprodserv' => $prodserv->idprodserv,
                    'idgrupoitem' => $transferRequest['centrodecustotransferencia'],
                    'qtd' => 1,
                    'un' => 'UN',
                    'valorun' => $transferRequest['valortransferencia'],
                    'valoritem' => $transferRequest['valortransferencia'],
                ]);

                NfParcela::create([
                    'idnf' => $nf->idnf,
                    'idagencia' => $agencia->idagencia,
                    'idempresa' => $nf->idempresa,
                    'idpessoa' => $pessoa->idpessoa,
                    'tipo' => $key,
                    'parcela' => 1,
                    'valor' => $transferRequest['valortransferencia'],
                    'datavencimento' => $transferRequest['datatransferencia'],
                    'datatransacao' => $transferRequest['datatransferencia'],
                    'status' => 'CONCLUÍDO',
                    'saldoinicial' => $agencia->saldoinicial,
                    'saldoatual' => (float) $agencia->saldoinicial - (float) $transferRequest['valortransferencia'],
                ]);

                if ($key == "D") {
                    $agencia->decrementarSaldo($transferRequest['valortransferencia']);
                } else {
                    $agencia->incrementarSaldo($transferRequest['valortransferencia']);
                }
            }

            DB::commit();
            return back()->with('success', 'Transferência realizada com sucesso!');
        } catch (Exception $e) {
            DB::rollBack();
            return back()->with('error', $e->getMessage());
        }
    }

    public function search(Request $request)
    {
        $nfs = Nf::with('parcela', 'empresa', 'agencia', 'nfitem')
                    ->whereNotNull('transferkey')
                    ->orWhereHas('empresa', function ($q) use ($request) {
                        $q->where('nomefantasia', 'like', '%' . $request->search . '%')
                        ->orWhere('razaosocial', 'like', '%' . $request->search . '%')
                        ->orWhere('sigla', 'like', '%' . $request->search . '%');
                    })
                    ->where('status', $request->status)
                    ->get()
                    ->groupBy('transferkey')
                    ->map(function ($nfs) {
                        return [
                            'nf1' => $nfs->first(),
                            'nf2' => $nfs->skip(1)->first(), // Pega a segunda NF do grupo, se existir
                        ];
                    });

        $agencias = Agencia::with('empresa')
                            ->orderBy('nome')
                            ->get();

        $centroCustos = GrupoItem::all();

        $statusOptions = [
            'CONCLUÍDO' => 'Concluído',
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.transferencias.index', compact('nfs', 'agencias', 'centroCustos', 'statusOptions'));
    }
}
