<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use App\Models\Pessoa;
use App\Models\Veiculo as Model;
use App\Models\ContratoVeiculo;
use Exception;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class VeiculoController extends Controller
{
    protected $marcas = [
		'audi' => "Audi",
		'bmw' => "BMW",
		'chevrolet' => "Chevrolet",
        'FACHHINI' => "FACHHINI",
		'FIAT' => "FIAT",
		'FORD' => "FORD",
        'GM' => "GM",
		'honda' => "Honda",
		'hyundai' => "Hyundai",
        'IDEROL' => "IDEROL",
        'IVECO' => "IVECO",
		'jaguar' => "Jaguar",
		'jeep' => "Jeep",
        'John Deere' => 'John Deere',
		'kia' => "Kia",
		'landrover' => "Land Rover",
        'LIU GONG' => "LIU GONG",
  		'mazda' => "Mazda",
        'MICHIGAN' => "MICHIGAN",
		'MB' => "MB",
        'New Holland' => 'New Holland',
		'nissan' => "Nissan",
		'porsche' => "Porsche",
        'RANDON' => "RANDON",
        'ROBERTO' => "ROBERTO",
		'renault' => "Renault",
		'subaru' => "Subaru",
		'tesla' => "Tesla",
		'toyota' => "Toyota",
        'VOLVO' => "VOLVO",
		'VW' => "VW",
        'outros' => "Outros"
    ];

	 protected $tipoveiculo = [
		'bicicleta' => "Bicicleta",
        'caminhao' => "Caminhão",
        'carro' => "Carro",
        'maquina' => "Máquina",
        'motocicleta' => "Motocicleta",
        'onibus' => "Ônibus"
    ];

	protected $cor = [
		'amarelo' => "Amarelo",
		'azul' => "Azul",
		'bege' => "Bege",
		'branco' => "Branco",
		'cinza' => "Cinza",
		'dourado' => "Dourado",
		'laranja' => "Laranja",
		'marrom' => "Marrom",
		'preto' => "Preto",
		'prata' => "Prata",
		'rosa' => "Rosa",
		'roxo' => "Roxo",
		'verde' => "Verde",
		'vermelho' => "Vermelho"
	];

    public function index(Request $request, Empresa $empresaModel, Pessoa $pessoaModel)
    {
        // Aplicando eager loading para empresa e pessoa, evitando N+1 queries
        $veiculos = Model::with(['empresa', 'pessoa'])
            ->select('veiculo.*', 'veiculo.status as status')
            ->join('empresa', 'empresa.idempresa', '=', 'veiculo.idempresa')
            ->where('veiculo.status', '!=', 'INATIVO')
            ->orderBy('empresa.sigla')
            ->orderBy('veiculo.marca')
            ->orderBy('veiculo.placa')
            ->paginate(100);

        $marcas = $this->marcas;
        $tipoveiculo = $this->tipoveiculo;
        $cor = $this->cor;
        $empresas = $empresaModel->orderBy('razaosocial')->get();
        $pessoas = $pessoaModel->orderBy('nome')->get();

        return view('admin.veiculos.index',
            compact(
                'veiculos',
                'marcas',
                'tipoveiculo',
                'cor',
                'empresas',
                'pessoas'
            )
        );
    }

    public function fetch(Request $request)
    {
        $perPage = $request->input('per_page', 100);

        // Definir os campos e suas classes CSS
        $fields = [
            'ID' => [
                'key' => 'idveiculo',
                'class' => 'text-center text-xs',
            ],
            'EMPRESA' => [
                'key' => 'empresa.sigla',
                'class' => 'text-center text-xs',
            ],
            'RESPONSÁVEL' => [
                'key' => fn($veiculo) => $veiculo->pessoa->nome ?? 'Sem pessoa',
                'class' => 'text-center text-xs',
            ],
            'TIPO' => [
                'key' => fn($veiculo) => $veiculo->tipoveiculo ?? '-',
                'class' => 'text-center text-xs',
            ],
            'SIGLA/APELIDO' => [
                'key' => fn($veiculo) => $veiculo->sigla ?? '-',
                'class' => 'text-center text-xs',
            ],
            'MARCA' => [
                'key' => 'marca',
                'class' => 'text-end text-sm fw-bold text-success',
            ],
            'MODELO' => [
                'key' => fn($veiculo) => $veiculo->modelo ?? '-',
                'class' => 'text-center text-xs fw-bold',
            ],
            'PLACA' => [
                'key' => fn($veiculo) => $veiculo->placa ?? '-',
                'class' => 'text-center text-xs fw-bold',
            ],
            'VENCIMENTO' => [
                'key' => fn($venda) => $venda->datavencimento
                    ? Carbon::parse($venda->datavencimento)->format('d/m/y')
                    : '-',
                'class' => 'text-center text-xs fw-bold',
            ],
            'STATUS' => [
                'key' => fn($venda) => view('partials.status-switch', [
                    'status' => $venda->status,
                    'id' => $venda->idnf,
                ])->render(),
                'class' => 'no-clicked pe-3',
            ],
        ];

        // Buscar os dados paginados
        $veiculos = Model::with(['empresa', 'pessoa'])
            ->select('veiculo.*', 'veiculo.status as status')
            ->join('empresa', 'empresa.idempresa', '=', 'veiculo.idempresa')
            ->orderBy('empresa.sigla')
            ->orderBy('veiculo.marca')
            ->orderBy('veiculo.placa')
            ->paginate($perPage, ['*'], 'page', $request->input('page', 2));

        // Montar os dados processados para o frontend
        $processedData = $veiculos->map(function ($veiculo) use ($fields) {
            $row = [
                'url' => route('admin.veiculos.edit', $veiculo->idveiculo),
            ];
            foreach ($fields as $label => $field) {
                $value = is_callable($field['key'])
                    ? $field['key']($veiculo) // Executar a closure
                    : data_get($veiculo, $field['key']); // Obter o valor diretamente
                $row[$label] = [
                    'value' => $value,
                    'class' => $field['class'],
                ];
            }
            return $row;
        });

        // Retornar os dados processados como JSON
        return response()->json([
            'columns' => array_keys($fields),
            'data' => $processedData,
            'next_page_url' => $veiculos->nextPageUrl(),
        ]);
    }

    public function create(Pessoa $pessoaModel, Empresa $empresaModel)
    {
        $pessoas = $pessoaModel
                    ->orderBy('nome')
                    ->get();

        $empresas = $empresaModel
                        ->orderBy('razaosocial')
                        ->get();

        $fornecedores = $pessoaModel
                        ->orderBy('nome')
                        ->fornecedor()
                        ->get();

        $marcas = $this->marcas;

		$tipoveiculo = $this->tipoveiculo;

		$cor = $this->cor;

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.veiculos.create')->with(compact('pessoas', 'empresas', 'marcas', 'tipoveiculo', 'cor','fornecedores', 'statusOptions'));
    }

    public function store(Request $request)
    {
        $veiculoCreate = $request->all();

        // Defina o valor do campo dashboard com base na presença do checkbox no formulário
        $veiculoCreate['dashboard'] = $request->has('dashboard') ? 1 : 0;

        DB::beginTransaction();
        try {
            $veiculo = Model::create($veiculoCreate);

            DB::commit();
            return redirect()->to('admin/veiculos/'.$veiculo->idveiculo.'/edit');

        } catch(Exception $e) {
            DB::rollBack();
            return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', $e->getMessage());
        }
    }

    public function edit($idVeiculo, Empresa $empresaModel, Pessoa $pessoaModel, ContratoVeiculo $contratoVeiculoModel)
    {
        $veiculo = Model::find($idVeiculo);
        $pessoas = $pessoaModel
                    ->orderBy('nome')
                    ->get();
        $empresas = $empresaModel
                    ->orderBy('razaosocial')
                    ->get();

        $fornecedores = $pessoaModel
                    ->orderBy('nome')
                    ->fornecedor()
                    ->get();

        $marcas = $this->marcas;

		$tipoveiculo = $this->tipoveiculo;

		$cor = $this->cor;

        $contratosveiculos = $contratoVeiculoModel
        ->where('idveiculo', $idVeiculo) // Filtra pelos registros com o idnf relacionado
        ->get();


        if ($veiculo->status == 'ATIVO'){
            $bg_status = "bg-success";
        }elseif($veiculo->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.veiculos.edit')->with(compact('veiculo', 'empresas', 'pessoas', 'marcas', 'tipoveiculo', 'cor','bg_status','contratosveiculos','fornecedores', 'statusOptions'));
    }

    public function update($idVeiculo, Request $request)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $veiculo = Model::find($idVeiculo);
        $veiculoRequest = $request->all();

        DB::beginTransaction();
        try {
            if ($redirect == true) {
                // Atualiza o valor do campo 'dashboard' com base no estado do checkbox
                $veiculo->dashboard = $request->input('dashboard') ? 1 : 0;

                // Atualiza os outros campos do veículo com os dados recebidos do formulário
                $veiculo->update($request->except('dashboard'));
            }

            if ($redirect == false) {
                $veiculo->update($veiculoRequest);

                DB::commit();
                return response()->json($veiculo->status);
            }

            DB::commit();
            return back()->with('success', 'Veículo atualizado!');

        } catch(Exception $e) {
            DB::rollBack();
            return redirect()
            ->back()
            ->with('error', $e->getMessage())
            ->withInput();
        }
    }

    public function destroy($idVeiculo)
    {
        $veiculo = Model::find($idVeiculo);

        try
        {
            $veiculo->delete();

            return redirect()
                    ->route('admin.veiculos.index')
                    ->with('success', 'Veículo removido!');
        } catch(Exception $e)
        {
            return redirect()
                    ->back()
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function search(Request $request, Empresa $empresaModel, Pessoa $pessoaModel)
    {
        // Aplicando eager loading para evitar N+1 queries
        $query = Model::query()
            ->join('empresa', 'veiculo.idempresa', '=', 'empresa.idempresa') // Inclua um join para acessar 'empresa.sigla'
            ->with(['empresa', 'pessoa'])
            ->when($request->status, function ($query, $status) {
                $query->where('veiculo.status', $status);
            })
            ->orderBy('empresa.sigla')
            ->orderBy('veiculo.marca')
            ->orderBy('veiculo.placa');

        // Aplica filtros de busca por termo se o campo `search` estiver preenchido
        if ($request->filled('search') && $request->search !== null) {
            $searchTerm = '%' . $request->search . '%';
            $query->where(function ($query) use ($searchTerm) {
                $query->whereHas('pessoa', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('nome', 'like', $searchTerm)
                        ->orWhere('razaosocial', 'like', $searchTerm);
                })->orWhereHas('empresa', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('sigla', 'like', $searchTerm);
                })->orWhere('veiculo.placa', 'like', $searchTerm)
                ->orWhere('veiculo.marca', 'like', $searchTerm)
                ->orWhere('veiculo.modelo', 'like', $searchTerm);
            });
        }

        filter($request, $query);

        $veiculos = $query->paginate(100, ['*'], 'page', $request->page ?? 1);

        // Tamanho da consulta
        $size = $veiculos->count();

        // Definindo mensagens de sessão
        session()->forget(['error', 'success']);
        session()->flash($size > 0 ? 'success' : 'error', $size > 0 ? $size . ' valores encontrado(s).' : 'Nenhum registro encontrado.');

        // Recuperar as listas de marcas, tipos de veículos e cores
        $marcas = $this->marcas;
        $tipoveiculo = $this->tipoveiculo;
        $cor = $this->cor;
        $empresas = $empresaModel->orderBy('razaosocial')->get();
        $pessoas = $pessoaModel->orderBy('nome')->get();

        return view('admin.veiculos.index',
            compact(
                'veiculos',
                'marcas',
                'tipoveiculo',
                'cor',
                'empresas',
                'pessoas'
            )
        );
    }


}
