<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Admin\EmpresaController;
use App\Models\Agencia;
use App\Models\Empresa;
use App\Models\Nf;
use App\Models\NfItem;
use App\Models\NfParcela;
use App\Models\Pessoa;
use App\Models\Plano;
use App\Models\Tenant;
use App\Models\User;
use App\Services\PagarMeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use stdClass;

class CadastroController extends Controller
{
    protected $pagarmeService;

    public function __construct(PagarMeService $pagarmeService)
    {
        $this->pagarmeService = $pagarmeService;
    }

    public function create(Plano $plano, Pessoa $pessoa = null)
    {
        if ($pessoa && $pessoa->tipo == 'R') {
            return view('pages.cadastro', compact('plano', 'pessoa'));
        }

        return view('pages.cadastro', compact('plano'));
    }

    public function store(Request $request, Plano $plano)
    {
        // Tratamento para tipo de documento
        if ($request['documento'] && strlen($request['documento']) == 14) {
            $tipoDocumento = 'CPF';
            $tipoCliente = 'individual';
        } elseif ($request['documento'] && strlen($request['documento']) == 18) {
            $tipoDocumento = 'CNPJ';
            $tipoCliente = 'company';
        }

        // Tratamento para documento
        $documento = str_replace(['.', '-', '/'], '', $request['documento']);
        // Tratamento para celular sem DDD. Está vindo: Ex: (00) 0 0000-0000 e queremos sem o dd: 00000000
        $celular = str_replace(['(', ')', ' ', '-'], '', substr($request['celular'], 3));
        // Tratamento para ddd. Esta vindo: (00) 0 0000-0000 e queremos: 00
        $ddd = str_replace(['(', ')', ' ', '-'], '', substr($request['celular'], 0, 3));
        // Tratamento para CEP. está vindo: 00.000-000 e queremos: 00000000
        $cep = str_replace(['.', '-'], '', $request['cep']);
        // Tratamento para valor. esta vindo: 29.90 e queremos em centavos: 2990
        $valor = str_replace(['.', ','], ['', '.'], $plano->valor);

        DB::beginTransaction();
        try {
            if ($plano->gratuito == true) {
                $v_status = "ATIVO";
            } else {
                $v_status = "INATIVO";
            }

            // Registra o usuário no Banco de Dados
            if (! $this->verificaSeDocumentoExiste(Pessoa::class, 'cpfcnpj', $request['documento'])) {
                // Registra a pessoa no Banco de Dados, caso não exista
                $newPessoa = $this->createPerson($request, $plano, $cep, 'cliente', $v_status);
                // Registra o tenant no Banco de Dados, caso não exista
                $newTenant = $this->createTenant($newPessoa, $request, $v_status, $plano);
            } else {
                // Recupera a pessoa no Banco de Dados
                $newPessoa = Pessoa::where('cpfcnpj', $request['documento'])->first();
                // Recupera o tenant no Banco de Dados
                $newTenant = Tenant::where('idpessoa', $newPessoa->idpessoa)->first();
            }

            // Verifica se o user existe
            if (! $this->verificaSeDocumentoExiste(User::class, 'email', request('email'))) {
                // Registra o user no Banco de Dados
                $this->createUser($newTenant, $request, $v_status);
            }

            // Cria uma nova empresa no Banco de Dados
            $sigla = implode('', array_map(function($word) {
                return isset($word[0]) ? strtoupper($word[0]) : '';
            }, explode(' ', $request['nome'])));
            if (! $this->verificaSeDocumentoExiste(Empresa::class, 'idtenant', $newTenant->idtenant)) {
                $empresaRequest = $this->createEmpresaRequest($newTenant, $request, $sigla, $documento, $cep, $v_status);
                $newEmpresa = $this->createEmpresa($empresaRequest);
            } else {
                $newEmpresa = Empresa::where('idtenant', $newTenant->idtenant)->first();
            }

            // Cria 2 novas pessoas default para a empresa (para transferencias)
            if ($this->verificaSeDocumentoExiste(Pessoa::class, 'cpfcnpj', $request['documento'])) {
                $tipos = ['fornecedor', 'cliente'];
                foreach ($tipos as $tipo) {
                    EmpresaController::createPersonDefault($newEmpresa, $tipo);
                }
            }

            if ($plano->gratuito == false) {
                // Procura agencia padrao
                $empresa = Empresa::find($plano->idempresa);
                $agencia = Agencia::where('idempresa', $empresa->idempresa)
                                    ->where('padrao', true)
                                    ->first();

                // Registra a nota no Banco de dados
                if (! $this->verificaSeDocumentoExiste(Pessoa::class, 'cpfcnpj', $request['documento'])) {
                    $newNF = $this->createNf($plano, $newPessoa, $agencia);

                    // Registra a NF Parcela
                    $newNfParcela = $this->createNfParcela($newNF, $newPessoa, $plano);
                } else {
                    // Recupera a nota no Banco de dados
                    $nf = Nf::where('idpessoa', $newPessoa->idpessoa)
                        ->where('status', 'CONCLUÍDO')
                        ->where('datavencimento', '>=', now()->format('Y-m-d'))
                        ->first();

                    $nfParcela = NfParcela::where('idnf', $nf->idnf)
                        ->where('status', 'ATIVO')
                        ->where('datavencimento', '>=', now()->format('Y-m-d'))
                        ->first();

                    if (! $nfParcela) {
                        $newNF = $this->createNf($plano, $newPessoa, $agencia);
                        $newNfParcela = $this->createNfParcela($newNF, $newPessoa, $plano);
                    }
                }

                if (! isset($nfParcela)) {
                    // Mescla o request para o PagarMe
                    $request = $this->createRequestOrderCheckoutPagarme($request, $plano, $newTenant, $newNfParcela, $cep, $ddd, $celular, $documento, $tipoCliente, $tipoDocumento, $valor);
                    // Gera o checkout do PagarMe
                    $response = $this->pagarmeService->createOrderCheckoutPagarme($request->all());
                }

                // Verifica se a resposta contém dados válidos
                if (isset($response->data) && isset($response->status)) {
                    // Atualiza os dados da NF
                    $newNF->update([
                        'qrcodeurl' => $response->data->charges[0]->last_transaction->qr_code_url,
                        'chavepix' => $response->data->charges[0]->last_transaction->qr_code,
                    ]);

                    // Atualiza os dados da NF Parcela
                    $newNfParcela->update(['idtransacao' => $response->data->id]);

                    DB::commit();
                    return response()->json([
                        'data' => $response->data,
                        'idtenant' => $newTenant->idtenant,
                    ], $response->status);
                } else if (isset($nfParcela)) {

                    DB::commit();
                    return response()->json([
                        'nf' => $nf,
                        'idtenant' => $newTenant->idtenant,
                    ]);
                } else {
                    DB::rollBack();
                    return response()->json(['error' => 'Erro ao gerar o checkout'], 500);
                }
            }

            DB::commit();
            return response()->json([
                'idtenant' => $newTenant->idtenant,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    private function verificaSeDocumentoExiste($model, $coluna, $request): bool
    {
        $documento = $model::where($coluna, $request)->exists();
        if ($documento) {
            return true;
        }
        return false;
    }

    private function createPerson($request, $plano, $cep, $tipo, $v_status)
    {
        $newPessoa = Pessoa::create([
            'idrepresentante' => $request['idrepresentante'] ? $request['idrepresentante'] : null,
            'idempresa' => $plano->idempresa,
            'nome' => $request['nome'],
            'cpfcnpj' => $request['documento'],
            'cep' => $cep,
            'endereco' => $request['logradouro'],
            'numero' => $request['numero'],
            'bairro' => $request['bairro'],
            'cidade' => $request['cidade'],
            'estado' => $request['estado'],
            'celular' => $request['celular'],
            'tipo' => $tipo,
            'status' => $v_status
        ]);

        return $newPessoa;
    }

    private function createTenant($newPessoa, $request, $v_status, $plano)
    {
        $newTenant = Tenant::create([
            'idpessoa' => $newPessoa->idpessoa,
            'nome' => $request['nome'],
            'status' => $v_status,
            'is_tenant_admin' => 0,
            'idplano' => $plano->idplano
        ]);

        return $newTenant;
    }

    private function createUser($newTenant, $request, $v_status)
    {
        $user = User::create([
            'idtenant' => $newTenant->idtenant,
            'name' => $request['nome'],
            'email' => $request['email'],
            'status' => $v_status,
            'password' => $request['password'],
            'api_token' => Str::random(80),
        ])->assignRole('ADMIN');

        return $user;
    }

    private function createEmpresaRequest($newTenant, $request, $sigla, $documento, $cep, $v_status)
    {
        $empresaRequest = [
            'idtenant' => $newTenant->idtenant,
            'razaosocial' => $request['nome'],
            'nomefantasia' => $request['nomefantasia'] ?? null,
            'sigla' => $sigla,
            'cnpj' => $documento,
            'cep' => $cep,
            'endereco' => $request['logradouro'],
            'numero' => $request['numero'],
            'complemento' => $request['complemento'] ?? null,
            'bairro' => $request['bairro'],
            'cidade' => $request['cidade'],
            'estado' => $request['estado'],
            'status' => $v_status,
        ];

        return $empresaRequest;
    }

    private function createEmpresa($empresaRequest)
    {
        return Empresa::create($empresaRequest);
    }

    private function createNf($plano, $newPessoa, $agencia)
    {
        $newNF = Nf::create([
            'idprodservtipo' => $plano->prodserv->idprodservtipo,
            'idpessoa' => $newPessoa->idpessoa,
            'idempresa' => $plano->idempresa,
            'idagencia' => $agencia->idagencia ?? null,
            'dataentrada' => now(),
            'dataemissao' => now(),
            'datavencimento' => now()->addMonth(),
            'tipo' => 'C',
            'saida' => 'V',
            'status' => 'CONCLUÍDO',
            'valornf' => $plano->valor,
            'totalnf' => $plano->valor,
        ]);

        // Cria nfitem
        $this->createNfItem($newNF, $plano);

        return $newNF;
    }

    private function createNfItem($newNF, $plano)
    {
        $nfItem = NfItem::create([
            'idnf' => $newNF->idnf,
            'idprodserv' => $plano->idprodserv,
            'idempresa' => $plano->idempresa,
            'item' => $plano->nome,
            'qtd' => 1,
            'valorun' => $plano->valor,
            'valoritem' => $plano->valor,
            'un' => 'UNIDADE'
        ]);

        return $nfItem;
    }

    private function createNfParcela($newNF, $newPessoa, $plano)
    {
        $newNfParcela = NfParcela::create([
            'idnf' => $newNF->idnf,
            'idpessoa' => $newPessoa->idpessoa,
            'idempresa' => $plano->idempresa,
            'idagencia' => $agencia->idagencia ?? null,
            'localtransacao' => 'pagarme',
            'tipo' => 'C',
            'parcela' => 1,
            'item' => $plano->nome,
            'qtd' => 1,
            'valor' => $plano->valor,
            'status' => 'ATIVO',
            'datavencimento' => now()->addMonth(),
            'saldoinicial' => $agencia->saldoatual ?? 0,
        ]);

        return $newNfParcela;
    }

    private function createRequestOrderCheckoutPagarme(
        $request,
        $plano,
        $newTenant,
        $newNfParcela,
        $cep,
        $ddd,
        $celular,
        $documento,
        $tipoCliente,
        $tipoDocumento,
        $valor)
    {
        $requestMerged = $request->merge([
            'estado' => $request['estado'],
            'cidade' => $request['cidade'],
            'cep' => $cep,
            'logradouro' => $request['logradouro'],
            'ddd' => $ddd,
            'celular' => $celular,
            'nome' => $request['nome'],
            'email' => $request['email'],
            'documento' => $documento,
            'tipo_cliente' => $tipoCliente,
            'tipo_documento' => $tipoDocumento,
            'valor' => $valor,
            'descricao' => "Plano: $plano->nome",
            'idnfparcela' => $newNfParcela->idnfparcela,
            'idplano' => $plano->idplano,
            'idtenant' => $newTenant->idtenant,
        ]);

        return $requestMerged;
    }
}
