<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Str;

class ForgotPasswordController extends Controller
{
    private string $SECRET_API_KEY;

    public function __construct() {
        $this->SECRET_API_KEY = env('BREVO_API_KEY');
    }

    public function sendResetLinkEmail(Request $request)
    {
        $request->validate(['email' => 'required|email']);

        // Obtém o usuário pelo e-mail
        $user = User::where('email', $request->email)->first();

        if (!$user) {
            return back()->withErrors(['email' => 'Usuário não encontrado.']);
        }

        // // Gera o token de redefinição de senha
        // $status = Password::sendResetLink($request->only('email'));

        // if ($status != Password::RESET_LINK_SENT) {
        //     return back()->withErrors(['email' => __($status)]);
        // }

        $token = Password::createToken($user); // Obtém o token corretamente

        $data = [
            'sender' => [
                'name' => config('app.name'),
                'email' => config('mail.from.address'),
            ],
            'to' => [
                [
                    'email' => $request->email,
                    'name' => $request->email
                ]
            ],
            'subject' => 'Redefinição de Senha',
            'textContent' => "Por favor, clique no link abaixo para redefinir sua senha: " . url('/reset-password/' . $token . '?email=' . $request->email),
            'htmlContent' => 'Clique no link abaixo para redefinir sua senha:<br><a href="' . url('/reset-password/' . $token . '?email=' . $request->email) . '">Redefinir Senha</a>',
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://api.brevo.com/v3/smtp/email');
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Accept: application/json',
            'api-key: ' . $this->SECRET_API_KEY
        ]);

        $result = curl_exec($ch);
        if (curl_errno($ch)) {
            return back()->withErrors(['email' => 'Erro ao enviar o link de recuperação.']);
        }
        curl_close($ch);

        if ($result === false) {
            return back()->withErrors(['email' => 'Erro ao enviar o link de recuperação.']);
        }

        return back()->with('status', 'O link de recuperação foi enviado!');
    }


    public function reset(Request $request)
    {
        $request->validate([
            'token' => 'required',
            'email' => 'required|email',
            'password' => 'required|min:8|confirmed',
        ]);

        $status = Password::reset(
            $request->only('email', 'password', 'password_confirmation', 'token'),
            function ($user, $password) {
                $user->forceFill([
                    'password' => $password
                ])->setRememberToken(Str::random(60));

                $user->save();
                event(new PasswordReset($user));
            }
        );

        return $status === Password::PASSWORD_RESET
                    ? redirect()->route('login')->with('success', __($status))
                    : back()->withErrors(['email' => [__($status)]]);
    }

    public function showResetForm(Request $request, $token = null)
    {
        if (is_null($token)) {
            return redirect()->route('login');
        }

        $request->validate([
            'email' => 'required|email',
            'password' => 'required|min:8|confirmed',
        ]);
        return view('sessions.password.reset', ['token' => $token, 'email' => $request->email]);
    }

    public function showLinkRequestForm()
    {
        return view('sessions.password.verify');
    }
}
