<?php

namespace App\Http\Controllers;

use App\Models\Agencia;
use App\Models\Nf;
use App\Models\NfItem;
use App\Models\NfParcela;
use App\Models\Pessoa;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class WebhookController extends Controller
{
    public function handle(Request $request)
    {
        Log::info('Recebendo webhook do Pagar.me:', $request->all());

        // Valida a assinatura do Pagar.me para garantir que a requisição é legítima
        $this->validatePagarmeSignature($request);

        // Pega os dados do webhook
        $payload = $request->all();

        // Verifica o tipo de evento
        if (isset($payload['type']) && $payload['type'] === 'order.paid') {
            $this->processChargeSucceeded($payload);
            // Retorne uma resposta de sucesso para o Pagar.me
            return response()->json(['status' => 'success'], 200);
        }

        return response()->json(['status' => 'event_not_handled'], 400);
    }

    private function validatePagarmeSignature(Request $request)
    {
        $signature = $request->header('X-Hub-Signature');
        $secret = config('pagarme.api_key');

        if (!$signature) {
            abort(403, 'Assinatura não fornecida');
        }

        $calculatedSignature = 'sha1=' . hash_hmac('sha1', $request->getContent(), $secret);

        if (!hash_equals($calculatedSignature, $signature)) {
            abort(403, 'Assinatura inválida');
        }
    }

    // Processa o evento de cobrança concluída
    private function processChargeSucceeded($payload)
    {
        DB::beginTransaction();
        try {
            // Recupera o idtenant do webhook
            $idtenant = $payload['data']['customer']['metadata']['idtenant'] ?? null;
            $tenant = Tenant::find($idtenant);

            if ($tenant) {
                // Atualiza o status do tenant para ativo
                if ($tenant->status !== 'ATIVO') {
                    $tenant->status = 'ATIVO';
                    $tenant->save();
                }

                // Atualiza o status da nfParcela para concluído
                $idnfparcela = $payload['data']['order']['metadata']['idnfparcela'] ?? null;
                $nfParcela = NfParcela::find($idnfparcela);

                if ($nfParcela && $nfParcela->status !== 'CONCLUÍDO') {
                    $agencia = Agencia::find($nfParcela->idagencia);
                    $nfParcela->status = 'CONCLUÍDO';
                    $nfParcela->datatransacao = now();
                    $nfParcela->saldoatual = $agencia->saldoatual + $nfParcela->valor;
                    $nfParcela->save();
                }

                // Atualiza o status de todos os usuários do tenant para ativo
                $users = User::where('idtenant', $idtenant)->get();
                foreach ($users as $user) {
                    if ($user->status !== 'ATIVO') {
                        $user->status = 'ATIVO';
                        $user->save();
                    }
                }

                // Atualiza o status da pessoa
                $pessoa = Pessoa::where('idpessoa', $tenant->idpessoa)->first();
                if ($pessoa && $pessoa->status !== 'ATIVO') {
                    $pessoa->status = 'ATIVO';
                    $pessoa->save();
                }

                // Crie uma nf para o representante caso tenha.
                if ($pessoa->idrepresentante) {
                    $comissao = $nfParcela->valor * ($pessoa->representante->comissao / 100);
                    $nf = Nf::create([
                        'idpessoa' => $pessoa->idoessoa,
                        'idempresa' => $pessoa->representante->idempresa,
                        'idprodservtipo' => $nfParcela->nf->idprodservtipo,
                        'saida' => 'V',
                        'tipo' => 'C',
                        'dataemissao' => now(),
                        'dataentrada' => now(),
                        'datainicio' => now(),
                        'datavencimento' => now(),
                        'valornf' => $comissao,
                        'totalnf' => $comissao,
                        'qtdparcela' => 1,
                        'status' => 'CONCLUÍDO',
                        'observacao' => "Comissão de indicação de {$pessoa->representante->nome}",
                    ]);

                    // Procura nf de acordo com nfparcela que está vindo na req
                    $nfCliente = Nf::where('idnf', $nfParcela->idnf)->first();
                    $nfitem = $nfCliente->nfitem()->where('idnf', $nfCliente->idnf)->first();
                    NfItem::create([
                        'idnf' => $nf->idnf,
                        'idempresa' => $nf->idempresa,
                        'idprodserv' => $nfitem->idprodserv,
                        'idgrupoitem' => $nfitem->idgrupoitem,
                        'idtipoitem' => $nfitem->idtipoitem,
                        'iditem' => $nfitem->iditem,
                        'item' => $nfitem->item,
                        'qtd' => 1,
                        'un' => 'UN',
                        'valorun' => $comissao,
                        'valoritem' => $comissao,
                    ]);

                    NfParcela::create([
                        'idnf' => $nf->idnf,
                        'idagencia' => $nf->idagencia ?? null,
                        'idempresa' => $nf->idempresa,
                        'idpessoa' => $nf->idpessoa,
                        'tipo' => 'C',
                        'parcela' => 1,
                        'valor' => $comissao,
                        'datavencimento' => now(),
                        'datatransacao' => now(),
                        'status' => 'ATIVO',
                    ]);
                }

                DB::commit();
            }
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Erro ao processar webhook do Pagar.me: ' . $e->getMessage());
        }
    }
}
