<?php

namespace App\Http\Livewire;

use Livewire\Component;
use App\Models\Evento as Model;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class CalendarEventoComponent extends Component
{
    public $eventos;

    public function mount()
    {
        $visualizacao = request()->query('visualizacao', 'duracao'); // 'duracao' como padrão

        $inicioMesAtual = Carbon::now()->startOfMonth();
        $finalMesAtual = Carbon::now()->endOfMonth();


        $this->eventos = Model::query()
                              ->select('evento.*', 
                              DB::raw("IFNULL(cliente.razaosocial, cliente.nome) as pessoaRazaoSocial"),
                              DB::raw("CASE WHEN evento.status = 'CONCLUÍDO' THEN '#eeeeee' ELSE cliente.cor END as backgroundColor"),
                              DB::raw("CONCAT(evento.data, ' ', IF(evento.hora = '00:00:00', '08:00:00', evento.hora)) as dataHoraInicio"),
                            DB::raw("CONCAT(evento.datafim, ' ', IF(evento.horafim = '00:00:00', '18:00:00', evento.horafim)) as dataHoraFim"))
                              ->join('empresa', 'empresa.idempresa', '=', 'evento.idempresa')
                              ->leftJoin('pessoa as cliente', 'cliente.idpessoa', '=', 'evento.idcliente')
                              
                              ->get()
                              ->map(function ($evento) use ($visualizacao) {

                                $start = $evento->dataHoraInicio;
                                $end = $evento->dataHoraFim;
                                $urlEvento = '/admin/eventos/'.$evento->idevento.'/edit';

                                 // Ajusta 'start' e 'end' com base na visualização selecionada
                                switch ($visualizacao) {
                                    
                                    case 'inicio':
                                        
                                        $end = ''; // Ignora a data/hora de término
                                        break;
                                    case 'termino':
                                        //$start = $end; // Usa a data/hora de término como início
                                        $start = $end;
                                        $end = ''; // Opcional: remove o 'end' para evitar confusão
                                        break;
                                    // 'duracao' não precisa de ajustes, pois usa $start e $end originais
                                }
                                
                                $className = 'event-color-' . Str::slug($evento->idevento);
                                $textColor = $this->textColorForBackground($evento->backgroundColor); // Calcula a cor do texto com base no background
                                return [
                                    'title' => $evento->pessoaRazaoSocial,
                                    'start' => $start,
                                    'end' => $end,
                                    'className' => $className,
                                    'backgroundColor' => $evento->backgroundColor, // Agora incluímos corretamente
                                    'textColor' => $textColor, // Adicionando a cor do texto determinada
                                    'url' => $urlEvento,
                                    // Removido 'backgroundColor' pois agora usaremos 'className'
                                ];
                              });
    }

    public function render()
    {
        return view('livewire.calendar-evento-component', [
            'eventos' => $this->eventos,
        ]);
    }

    public function textColorForBackground($backgroundColor) {
        // Remover '#' se estiver presente
        $color = ltrim($backgroundColor, '#');
    
        // Convertendo o HEX em componentes RGB
        $r = hexdec(substr($color, 0, 2));
        $g = hexdec(substr($color, 2, 2));
        $b = hexdec(substr($color, 4, 2));
    
        // Calculando a luminância
        $luminance = (0.299 * $r + 0.587 * $g + 0.114 * $b) / 255;
    
        // Se a luminância for menor que 0.5, a cor de fundo é escura, então usamos branco; senão, usamos #333
        return $luminance < 0.5 ? '#FFFFFF' : '#333333';
    }
}



