<?php

namespace App\Http\Livewire;

use Livewire\Component;
use App\Models\Nf;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Services\DateRangeService;

class MediaPedidoDieselComponent extends Component
{
    public $mediaConsumoMesAtual;

    public $datarange;
    public $idempresa;
    public $idveiculo;
    public $idcontrato;

    protected $listeners = ['filtersApplied' => 'onFilterUpdate'];

    public function mount()
    {
        $this->loadInitialData();
    }

    public function loadInitialData()
    {
        if (empty($this->datarange)) {
            $this->setInitialDateRange();
        }
        $this->loadData();
    }

    private function setInitialDateRange()
    {
        $dataInicio = Carbon::now()->subMonths(3)->startOfMonth()->format('d-m-Y');
        $dataFim = Carbon::now()->format('d-m-Y');
        $this->datarange = $dataInicio . ' até ' . $dataFim;
    }

    private function loadData()
    {
        $dates = DateRangeService::generateDateRange($this->datarange);

        $this->mediaConsumoMesAtual = $this->calcularMediaConsumo($dates['dataInicio'], $dates['dataFim']);

    }

    public function render()
    {
        return view('livewire.media-pedido-diesel-component', [
            'mediaConsumoMesAtual' => $this->mediaConsumoMesAtual

        ]);
    }

    public function onFilterUpdate($filters)
    {
        $this->datarange = $filters['datarange'] ?? null;
        $this->idempresa = $filters['idempresa'] ?? null;
        $this->idveiculo = $filters['idveiculo'] ?? null;
        $this->idcontrato = $filters['idcontrato'] ?? null;
        $this->loadData();
    }


    function calcularMediaConsumo($inicio, $fim) {
        // Consulta para obter a média de consumo (km/l) do período especificado
        $query = "
        SELECT
            n.idnf,
            ni.qtd,
            n.km,
            n.idveiculo
        FROM nf n
        JOIN nfitem ni ON n.idnf = ni.idnf
        JOIN veiculo v ON v.idveiculo = n.idveiculo and v.dashboard = 1
        WHERE ni.idprodserv IN (20, 31)
        AND n.dataentrada BETWEEN ? AND ?
        AND n.saida = 'P'
        AND ni.qtd > 0";

        $bindings = [$inicio, $fim];

        if ($this->idempresa) {
            $query .= " AND n.idempresa = ?";
            $bindings[] = $this->idempresa;
        }

        if (!empty($this->idcontrato)) {
            $query .= " AND EXISTS (
                SELECT 1
                FROM contratoveiculo cv
                WHERE cv.idveiculo = n.idveiculo
                AND cv.idcontrato = ?
            )";
            $bindings[] = $this->idcontrato;
        }

        if ($this->idveiculo) {
            $query .= " AND n.idveiculo = ?";
            $bindings[] = $this->idveiculo;
        }

        $query .= " ORDER BY n.idveiculo, n.dataentrada";

        $resultados = DB::select(DB::raw($query), $bindings);

        // Estruturas para armazenar as informações por veículo
        $informacoesPorVeiculo = [];
        foreach ($resultados as $registro) {
            $idVeiculo = $registro->idveiculo;
            $kmAtual = $registro->km;
            $litros = $registro->qtd;

            if (!isset($informacoesPorVeiculo[$idVeiculo])) {
                //implementar aqui
                $kmAnterior = $this->getUltimoKmAntesDe($idVeiculo, $inicio);
                $informacoesPorVeiculo[$idVeiculo] = ['kmAnterior' => $kmAnterior, 'distanciaTotal' => 0, 'litrosTotal' => 0];
            }

            $distancia = $kmAtual - $informacoesPorVeiculo[$idVeiculo]['kmAnterior'];
            $informacoesPorVeiculo[$idVeiculo]['distanciaTotal'] += max($distancia, 0);
            $informacoesPorVeiculo[$idVeiculo]['kmAnterior'] = $kmAtual;

            $informacoesPorVeiculo[$idVeiculo]['litrosTotal'] += $litros;
        }

        // Calcula a média de consumo
        $somaMedias = 0;
        $veiculosComMedia = 0;
        foreach ($informacoesPorVeiculo as $idVeiculo => $info) {
            $mediaConsumo = $info['litrosTotal'] > 0 ? $info['distanciaTotal'] / $info['litrosTotal'] : 0;
            if ($mediaConsumo > 0) {
                $somaMedias += $mediaConsumo;
                $veiculosComMedia++;
            }
        }

        // Calcula e retorna a média geral de consumo
        if ($veiculosComMedia > 0) {
            return $somaMedias / $veiculosComMedia;
        } else {
            return 0; // Retorna 0 se não houver dados suficientes
        }
    }

    function getUltimoKmAntesDe($idVeiculo, $dataInicio) {
        $ultimoRegistro = DB::table('nf')
            ->join('nfitem', 'nfitem.idnf', '=', 'nf.idnf')
            ->where('nf.idveiculo', $idVeiculo)
            ->where('nf.dataentrada', '<', $dataInicio)
            ->whereIn('nfitem.idprodserv', [20, 31])  // Correção aqui, usando whereIn corretamente
            ->orderByDesc('nf.dataentrada')
            ->first();

        return $ultimoRegistro ? $ultimoRegistro->km : 0;
    }

}


