<?php

namespace App\Http\Livewire;

use Livewire\Component;
use App\Models\TagRetira;
use Carbon\Carbon;

class TagComponent extends Component
{
    public $tagsMesAtual;
    public $tagsMesPassado;
    public $totalTagsMesAtual;
    public $totalTagsMesPassado;
    public $percentualTagsMudanca;
    public $tagsEmAbertoMesAtual;
    public $tagsEmAbertoMesPassado;
    public $tagsReservadasMesAtual;
    public $tagsReservadasMesPassado;
    public $tagsPendentes;
    public $tagsEmExecucao;
    public $tagsVencer;
    public $tagsVencidas;

    public function mount()
    {
        $inicioMesAtual = Carbon::now()->startOfMonth();
        $finalMesAtual = Carbon::now()->endOfMonth();

        $inicioMesPassado = Carbon::now()->subMonthNoOverflow()->startOfMonth();
        $finalMesPassado = Carbon::now()->subMonthNoOverflow()->endOfMonth();

        // Tags retiradas deste mês
        $this->tagsMesAtual = $this->buscarTags($inicioMesAtual, $finalMesAtual);

        // Tags retiradas do mês passado
        $this->tagsMesPassado = $this->buscarTags($inicioMesPassado, $finalMesPassado);

        // Tags deste mês em Aberto
        $this->tagsEmAbertoMesAtual = $this->buscarTags($inicioMesAtual, $finalMesAtual, 'aberto');

        // Tags do mês passado em Aberto
        $this->tagsEmAbertoMesPassado = $this->buscarTags($inicioMesPassado, $finalMesPassado, 'aberto');

        // Tags reservadas deste mês
        $this->tagsReservadasMesAtual = $this->buscarTags($inicioMesAtual, $finalMesAtual, 'reservada');

        // Tags reservadas do mês passado
        $this->tagsReservadasMesPassado = $this->buscarTags($inicioMesPassado, $finalMesPassado, 'reservada');

        // Tags pendentes
        $this->tagsPendentes = $this->buscarTags(null, null, 'PENDENTE');

        // Tags em execução
        $this->tagsEmExecucao = $this->buscarTags(null, null, 'EM ANDAMENTO');

        // Tags a vencer (3 dias ou menos para a datafim)
        $this->tagsVencer = $this->buscarTags(null, null)
        ->filter(function($tag) {
            return $tag->datafim
                && Carbon::parse($tag->datafim)->isBetween(Carbon::now(), Carbon::now()->addDays(3))
                && $tag->status === 'EM ANDAMENTO';
        });

        // Tags vencidas (datafim já passou e ainda não foram concluídas)
        $this->tagsVencidas = $this->buscarTags(null, null)
        ->filter(function($tag) {
            return $tag->datafim
                && Carbon::parse($tag->datafim)->isPast()
                && in_array($tag->status, ['EM ANDAMENTO', 'RESERVADA']);
        });



        // Calcula o total e percentual de mudança
        $this->totalTagsMesAtual = $this->calcularTotalTags($this->tagsMesAtual);
        $this->totalTagsMesPassado = $this->calcularTotalTags($this->tagsMesPassado);
        $this->percentualTagsMudanca = $this->calcularPercentualMudanca($this->totalTagsMesPassado, $this->totalTagsMesAtual);
    }

    private function buscarTags($inicio = null, $fim = null, $status = null)
    {
        $query = TagRetira::query()
            ->where('status', '!=', 'CANCELADO');

        if ($status) {
            $query->where('status', $status);
        }

        if ($inicio && $fim) {
            $query->whereBetween('datainicio', [$inicio, $fim]);
        } elseif ($inicio) {
            $query->where('datainicio', '>=', $inicio);
        } elseif ($fim) {
            $query->where('datainicio', '<=', $fim);
        }

        return $query->get();
    }

    private function calcularTotalTags($tags)
    {
        return $tags->count();
    }

    private function calcularPercentualMudanca($totalMesPassado, $totalMesAtual)
    {
        return $totalMesPassado > 0
            ? (($totalMesAtual - $totalMesPassado) / $totalMesPassado) * 100
            : 0;
    }

    public function render()
    {
        return view('livewire.tag-component', [
            'tagsMesAtual' => $this->tagsMesAtual,
            'tagsMesPassado' => $this->tagsMesPassado,
            'percentualTagsMudanca' => $this->percentualTagsMudanca,
            'tagsEmAbertoMesAtual' => $this->tagsEmAbertoMesAtual,
            'tagsEmAbertoMesPassado' => $this->tagsEmAbertoMesPassado,
            'tagsReservadasMesAtual' => $this->tagsReservadasMesAtual,
            'tagsReservadasMesPassado' => $this->tagsReservadasMesPassado,
            'tagsPendentes' => $this->tagsPendentes,
            'tagsEmExecucao' => $this->tagsEmExecucao,
            'tagsVencer' => $this->tagsVencer,
            'tagsVencidas' => $this->tagsVencidas,
            'totalTagsMesAtual' => $this->totalTagsMesAtual,
            'totalTagsMesPassado' => $this->totalTagsMesPassado,
            'percentualTagsMudanca' => $this->percentualTagsMudanca
        ]);
    }
}
