<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasOne;
use App\Scopes\TenantScope;
use App\Services\TagService;
use Illuminate\Support\Facades\Auth;

class Tag extends Model
{
    use HasFactory;

    // Definindo os campos personalizados de timestamps
    const CREATED_AT = 'criadoem';
    const UPDATED_AT = 'alteradoem';

    // Nome da tabela e chave primária
    protected $table = 'tag';
    protected $primaryKey = 'idtag';

    // Campos que podem ser preenchidos em massa
    protected $fillable = [
        'idtagtipo',
        'idempresa',
        'idnf',
        'tag',
        'status',
        'criadopor',
        'alteradopor',

        // campos
        'marca',
        'modelo',
        'anofabricacao',
        'numeroserie',
        'altura',
        'largura',
        'comprimento',
        'peso',
        'material',
        'combustivel',
        'datamanutencao',
        'dataproximamanutencao',
        'garantia',
        'voltagem',
        'processador',
        'memoriaram',
        'cor',
        'sistemaoperacional',
        'observacao',
    ];

    // Relacionamentos

    /**
     * Relacionamento com o model TagTipo (um Tag pertence a um TagTipo).
     */
    public function tagTipo()
    {
        return $this->belongsTo(TagTipo::class, 'idtagtipo', 'idtagtipo')->withDefault();
    }

    /**
     * Relacionamento com o model Empresa (um Tag pertence a uma Empresa).
     */
    public function empresa() : HasOne
    {
        return $this->hasOne(Empresa::class, 'idempresa', 'idempresa');
    }

    /**
     * Relacionamento com o model Pessoa (um Tag pertence a uma Pessoa).
     */
    public function pessoa() : HasOne
    {
        return $this->hasOne(Pessoa::class, 'idpessoa', 'idpessoa');
    }

    // Escopo global
    protected static function booted()
    {
        static::addGlobalScope(new TenantScope);

        // Gera automaticamente o campo 'tag' ao criar um novo registro
        static::creating(function ($tag) {
            if (is_null($tag->tag)) {
                $tagService = new TagService();
                $tag->tag = $tagService->generateTagNumber($tag->idempresa);
            }
        });

        // Vincula automaticamente o usuário logado ao criar um novo registro
        static::creating(function ($tag) {
            if (is_null($tag->criadopor)) {
                $tag->criadopor = Auth::user()->id;
            }
        });
    }

    // Método para verificar se a coluna existe na tabela
    public static function hasColumn($column)
    {
        return in_array($column, (new static)->getFillable());
    }

    public function filter($filters = false, $return = 'get', $paginate = '*')
    {
        if ($filters) {
            return $this->latest()
                ->with('empresa', 'tagtipo') // Carrega os relacionamentos necessários
                ->where(function ($query) use ($filters) {
                    if (isset($filters['filter']) && isset($filters['query'])) {
                        foreach ($filters['filter'] as $filter) {
                            if ($filter == 'empresa') {
                                // Filtro para o relacionamento empresa
                                $query->orWhereHas('empresa', function ($q) use ($filters) {
                                    $q->where('sigla', 'like', '%' . $filters['query'] . '%');
                                });
                            } elseif ($filter == 'tagtipo') {
                                // Filtro para o relacionamento tagtipo
                                $query->orWhereHas('tagtipo', function ($q) use ($filters) {
                                    $q->where('tagtipo', 'like', '%' . $filters['query'] . '%');
                                });

                            } else {
                                // Filtro direto nas colunas da tabela 'tagretira'
                                $query->orWhere($filter, 'like', '%' . $filters['query'] . '%');
                            }
                        }
                    }
                })
                ->when($return === 'paginate', function($query) use ($paginate) {
                    return $query->paginate($paginate);
                }, function($query) use ($return) {
                    return $query->$return();
                });
        } else {
            return $this->getAll($return, $paginate);
        }
    }
}
