<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Agendamento</title>
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        ::-webkit-scrollbar {
            background-color: transparent;
            width: 8px;
            border-radius: 15px;
        }
        ::-webkit-scrollbar-thumb {
            background-color: #ccc;
            border-radius: 15px;
        }
        .time-slots-container {
            max-height: 0;
            overflow: hidden;
            transition: max-height 0.3s ease-out;
        }
        .time-slots-container.expanded {
            max-height: 300px;
            overflow-y: auto;
        }
    </style>
</head>
<body class="bg-gray-100 flex items-center justify-center min-h-screen p-4">
    <div class="container mx-auto max-w-4xl">
        <div class="bg-white rounded-lg shadow-md overflow-hidden">
            <div class="p-6">
                <div class="flex flex-col md:flex-row gap-6">
                    <!-- Informações da Agenda -->
                    <div class="md:w-1/3">
                        <h2 class="text-2xl font-bold mb-4"><i class='bx bx-calendar-event'></i> <span id="estabelecimentoNome"></span></h2>
                        <div id="agendaInfo" class="text-sm"></div>
                    </div>

                    <!-- Calendário -->
                    <div class="md:w-2/3">
                        <h3 class="text-lg font-semibold mb-2">Selecione uma data</h3>
                        <div id="calendar"></div>
                    </div>
                </div>

                <!-- Seleção de Horários (inicialmente oculta) -->
                <div class="time-slots-container mt-6">
                    <h3 class="text-lg font-semibold mb-2">Disponibilidade</h3>
                    <div id="timeSlots" class="grid grid-cols-3 sm:grid-cols-4 md:grid-cols-5 gap-2"></div>
                </div>
            </div>
        </div>
    </div>
    {!! NoCaptcha::renderJs() !!}

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script>
        // Global variables
        const agenda = @json($agenda);
        let lastClickedDay = null;
        const diasSemana = ['dom', 'seg', 'ter', 'qua', 'qui', 'sex', 'sab'];
        let selectedDate = null;
        let currentMonth = new Date();

        // Initialization functions
        const initializeAgendaInfo = () => {
            document.getElementById('estabelecimentoNome').textContent = agenda.nome;
            if (agenda.idpessoa) {
                const pessoa = agenda.pessoa;
                document.getElementById('agendaInfo').innerHTML += `<p class="border border-1 p-2 mt-3 rounded-lg"><strong>Colaborador:</strong> ${pessoa.nome}</p>`;
            }
            initializeEventTypeSelect();
        };

        const initializeEventTypeSelect = () => {
            if (agenda.agenda_eventos_tipos) {
                const tipos = agenda.agenda_eventos_tipos;
                const select = createEventTypeSelect(tipos);
                const valorTipo = createValorTipoDiv();
                document.getElementById('agendaInfo').appendChild(select);
                document.getElementById('agendaInfo').appendChild(valorTipo);
            }
        };

        // Helper functions
        const createEventTypeSelect = (tipos) => {
            const select = document.createElement('select');
            select.classList.add('form-select', 'text-sm', 'block', 'w-full', 'border', 'border-1', 'rounded-lg', 'py-2', 'px-3', 'mt-3');
            select.name = 'tiposevento[]';
            select.id = 'select-tiposevento';

            const defaultOption = document.createElement('option');
            defaultOption.value = '';
            defaultOption.selected = true;
            defaultOption.disabled = true;
            defaultOption.textContent = 'Selecione o tipo de agendamento';
            select.appendChild(defaultOption);

            tipos.forEach(tipo => {
                const option = document.createElement('option');
                option.value = tipo.ideventotipo;
                option.dataset.valor = tipo.evento_tipo.prodserv.valorvenda ? tipo.evento_tipo.prodserv.valorvenda.replace('.', ',') : '';
                option.dataset.idprodserv = tipo.evento_tipo.idprodserv;
                option.dataset.tipodereserva = tipo.evento_tipo.hora === 'SIM' ? 'hora' : 'data';
                if (tipo.evento_tipo.hora === 'SIM') {
                    option.dataset.horapadrao = tipo.evento_tipo.horapadrao;
                } else {
                    option.dataset.datapadrao = tipo.evento_tipo.datapadrao;
                }
                option.textContent = tipo.evento_tipo.eventotipo;
                select.appendChild(option);
            });

            select.addEventListener('change', updateValorTipo);
            return select;
        };

        const createValorTipoDiv = () => {
            const valorTipo = document.createElement('div');
            valorTipo.classList.add('text-sm', 'hidden', 'w-full', 'border', 'border-1', 'rounded-lg', 'py-2', 'px-3', 'mt-3');
            return valorTipo;
        };

        const updateValorTipo = function() {
            const selectedTipo = agenda.agenda_eventos_tipos.find(tipo => tipo.ideventotipo == this.value);
            const valorTipo = document.querySelector('#agendaInfo div:last-child');

            if (selectedTipo && selectedTipo.evento_tipo.prodserv && selectedTipo.evento_tipo.prodserv.valorvenda) {
                valorTipo.classList.remove('hidden');
                valorTipo.innerHTML = `<span class="font-bold">Valor: </span> R$ ${selectedTipo.evento_tipo.prodserv.valorvenda.replace('.', ',')}`;
            } else {
                valorTipo.classList.add('hidden');
                valorTipo.innerHTML = '';
            }
        };

        // Calendar functions
        const renderCalendar = () => {
            const calendarEl = document.getElementById('calendar');
            calendarEl.innerHTML = '';

            const monthYearEl = createMonthYearElement();
            calendarEl.appendChild(monthYearEl);

            const weekdaysEl = createWeekdaysElement();
            calendarEl.appendChild(weekdaysEl);

            const daysEl = createDaysElement();
            calendarEl.appendChild(daysEl);

            addMonthNavigationListeners();
        };

        const createMonthYearElement = () => {
            const monthYearEl = document.createElement('div');
            monthYearEl.className = 'flex justify-between items-center mb-4';
            monthYearEl.innerHTML = `
                <button id="prevMonth" class="text-gray-600 hover:text-blue-500"><i class='bx bx-left-arrow-circle text-3xl'></i></button>
                <h3 class="text-lg font-semibold">${currentMonth.toLocaleString('default', { month: 'long', year: 'numeric' })}</h3>
                <button id="nextMonth" class="text-gray-600 hover:text-blue-500"><i class='bx bx-right-arrow-circle text-3xl'></i></button>
            `;
            return monthYearEl;
        };

        const createWeekdaysElement = () => {
            const weekdaysEl = document.createElement('div');
            weekdaysEl.className = 'grid grid-cols-7 gap-1 mb-2';
            diasSemana.forEach(dia => {
                const dayEl = document.createElement('div');
                dayEl.className = 'text-center text-sm font-medium text-gray-700';
                dayEl.textContent = dia.charAt(0).toUpperCase() + dia.slice(1, 3);
                weekdaysEl.appendChild(dayEl);
            });
            return weekdaysEl;
        };

        const createDaysElement = () => {
            const daysEl = document.createElement('div');
            daysEl.className = 'grid grid-cols-7 gap-1';

            const firstDay = new Date(currentMonth.getFullYear(), currentMonth.getMonth(), 1);
            const lastDay = new Date(currentMonth.getFullYear(), currentMonth.getMonth() + 1, 0);
            const today = new Date();

            for (let i = 0; i < firstDay.getDay(); i++) {
                daysEl.appendChild(document.createElement('div'));
            }

            for (let day = 1; day <= lastDay.getDate(); day++) {
                const date = new Date(currentMonth.getFullYear(), currentMonth.getMonth(), day);
                const dayEl = createDayElement(date, today);
                daysEl.appendChild(dayEl);
            }

            return daysEl;
        };

        const createDayElement = (date, today) => {
            const diaSemana = diasSemana[date.getDay()];
            const disponivel = temHorariosDisponiveis(diaSemana);
            const isPastDate = date < today.setHours(0, 0, 0, 0);
            const isBeforeMinimumAdvance = !meetsMinimumAdvance(date);

            const dayEl = document.createElement('button');
            dayEl.textContent = date.getDate();

            const isToday = date.toDateString() === today.toDateString();
            const noDaySelected = !lastClickedDay;

            const isLastClickedDay = lastClickedDay && lastClickedDay.textContent === date.getDate().toString();
            dayEl.className = getDayElementClass(isToday, noDaySelected, disponivel, isPastDate, isLastClickedDay, isBeforeMinimumAdvance);
            dayEl.disabled = !disponivel || isPastDate || isBeforeMinimumAdvance;

            if (disponivel && !isPastDate && !isBeforeMinimumAdvance) {
                dayEl.addEventListener('click', (e) => handleDayClick(e, date));
            }

            return dayEl;
        };

        const meetsMinimumAdvance = (date) => {
            const now = new Date();
            const antecedenciaMinima = agenda.antecedenciaminima ? agenda.antecedenciaminima.split(':') : [0, 0];
            const antecedenciaMinutos = parseInt(antecedenciaMinima[0]) * 60 + parseInt(antecedenciaMinima[1]);
            const diffMinutes = (date.getTime() - now.getTime()) / (1000 * 60);
            return diffMinutes >= antecedenciaMinutos;
        };

        const getDayElementClass = (isToday, noDaySelected, disponivel, isPastDate, isLastClickedDay, isBeforeMinimumAdvance) => {
            if (isToday && noDaySelected) {
                return 'w-full py-2 rounded-lg text-white bg-blue-700 hover:text-white hover:bg-blue-600 font-bold';
            } else if (isLastClickedDay) {
                return 'w-full py-2 rounded-lg bg-blue-600 text-white font-bold';
            } else {
                return `w-full py-2 rounded-lg ${
                    disponivel && !isPastDate && !isBeforeMinimumAdvance
                        ? 'bg-blue-500 text-white hover:bg-blue-600'
                        : 'bg-gray-200 text-gray-400 cursor-not-allowed'
                }`;
            }
        };

        const handleDayClick = (e, date) => {
            if ($('#select-tiposevento').val()) {
                if (lastClickedDay) {
                    lastClickedDay.classList.remove('bg-blue-600', 'font-bold');
                    lastClickedDay.classList.add('bg-blue-500');
                }
                e.target.classList.remove('bg-blue-500');
                e.target.classList.add('bg-blue-600', 'font-bold');
                lastClickedDay = e.target;

                selectDate(date);
            } else {
                alert('Selecione um tipo de evento antes de selecionar uma data.');
            }
        };

        const addMonthNavigationListeners = () => {
            document.getElementById('prevMonth').addEventListener('click', () => {
                currentMonth.setMonth(currentMonth.getMonth() - 1);
                renderCalendar();
            });

            document.getElementById('nextMonth').addEventListener('click', () => {
                currentMonth.setMonth(currentMonth.getMonth() + 1);
                renderCalendar();
            });
        };

        // Time slot functions
        const selectDate = async (date) => {
            selectedDate = date;
            const formattedDate = date.toISOString().split('T')[0];
            const idpessoa = agenda.idpessoa;
            const tipoReserva = $('#select-tiposevento').find(':selected').data('tipodereserva');

            const horariosOcupados = await fetchHorariosOcupados(formattedDate, idpessoa);

            if (tipoReserva === 'hora') {
                const slots = gerarSlots(diasSemana[date.getDay()]).filter(slot =>
                    isSlotAvailable(slot, horariosOcupados, 'hora')
                );
                renderTimeSlots(slots);
            } else if (tipoReserva === 'data') {
                const estaDisponivel = isSlotAvailable(formattedDate, horariosOcupados, 'data');
                const slots = estaDisponivel ? ['Agendar'] : [];
                renderTimeSlots(slots);
            }

            document.querySelector('.time-slots-container').classList.add('expanded');
        };

        const isSlotAvailable = (slot, horariosOcupados, tipoReserva) => {
            const now = new Date();
            const slotDate = new Date(selectedDate);

            if (tipoReserva === 'hora') {
                const [hours, minutes] = slot.split(':').map(Number);
                slotDate.setHours(hours, minutes, 0, 0);
            }

            // Verificar antecedência mínima
            const antecedenciaMinima = agenda.antecedenciaminima ? agenda.antecedenciaminima.split(':') : [0, 0];
            const antecedenciaMinutos = parseInt(antecedenciaMinima[0]) * 60 + parseInt(antecedenciaMinima[1]);
            const diffMinutes = (slotDate.getTime() - now.getTime()) / (1000 * 60);

            if (diffMinutes < antecedenciaMinutos) {
                return false; // Não atende à antecedência mínima
            }

            if (agenda.choquehorario === null) {
                return true; // Permite sobreposições ilimitadas
            }

            if (tipoReserva === 'hora') {
                const slotMinutos = converterParaMinutos(slot);

                if (agenda.choquehorario) {
                    const choqueCount = horariosOcupados.filter(ocupado => {
                        if (!ocupado.horaInicio) return false;
                        const inicioMinutos = converterParaMinutos(ocupado.horaInicio);
                        const fimMinutos = ocupado.horafim ? converterParaMinutos(ocupado.horafim) : inicioMinutos + 60;
                        return slotMinutos >= inicioMinutos && (slotMinutos < fimMinutos || fimMinutos < inicioMinutos);
                    }).length;
                    return choqueCount < agenda.choquehorario;
                } else {
                    return !horariosOcupados.some(ocupado => {
                        if (!ocupado.horaInicio) return false;
                        const inicioMinutos = converterParaMinutos(ocupado.horaInicio);
                        const fimMinutos = ocupado.horafim ? converterParaMinutos(ocupado.horafim) : inicioMinutos + 60;
                        return (slotMinutos >= inicioMinutos && (slotMinutos < fimMinutos || fimMinutos < inicioMinutos)) ||
                            (slotMinutos < inicioMinutos && (slotMinutos + agenda.intervaloMinutos) > inicioMinutos);
                    });
                }
            } else if (tipoReserva === 'data') {
                const dataReserva = new Date(slot);

                if (agenda.choquehorario) {
                    const choqueCount = horariosOcupados.filter(ocupado => {
                        const dataInicio = new Date(ocupado.data);
                        const dataFim = ocupado.datafim ? new Date(ocupado.datafim) : dataInicio;
                        return dataReserva >= dataInicio && dataReserva <= dataFim;
                    }).length;
                    return choqueCount < agenda.choquehorario;
                } else {
                    return !horariosOcupados.some(ocupado => {
                        const dataInicio = new Date(ocupado.data);
                        const dataFim = ocupado.datafim ? new Date(ocupado.datafim) : dataInicio;
                        return dataReserva >= dataInicio && dataReserva <= dataFim;
                    });
                }
            }

            return false; // Tipo de reserva não reconhecido
        };

        const isDateOccupied = (formattedDate, ocupado) => {
            const dataInicio = new Date(ocupado.data);
            const dataFim = new Date(ocupado.datafim);
            const dataSelecionada = new Date(formattedDate);
            return dataSelecionada >= dataInicio && dataSelecionada <= dataFim;
        };

        const renderTimeSlots = (slots) => {
            const timeSlotsEl = document.getElementById('timeSlots');
            timeSlotsEl.innerHTML = '';
            if (slots.length === 0) {
                timeSlotsEl.innerHTML = '<p class="col-span-full text-center text-gray-600">Nenhum horário disponível para este dia.</p>';
                return;
            }
            slots.forEach(slot => {
                const button = createTimeSlotButton(slot);
                timeSlotsEl.appendChild(button);
            });
        };

        const createTimeSlotButton = (slot) => {
            const button = document.createElement('button');
            button.textContent = slot;
            button.className = 'bg-green-500 text-white p-2 rounded-lg hover:bg-green-600';
            button.addEventListener('click', () => {
                if (document.getElementById('select-tiposevento').value === '') {
                    alert('Selecione o tipo de agendamento antes de escolher um horário.');
                    return;
                }
                document.querySelectorAll('#timeSlots button').forEach(btn => btn.classList.remove('selected'));
                button.classList.add('selected');
                selectTimeSlot(slot);
            });
            return button;
        };

        // Modal functions
        const selectTimeSlot = (slot) => {
            const modal = createModal(slot);
            document.body.appendChild(modal);
            addModalEventListeners(modal);
        };

        const createModal = (slot) => {
            const modal = document.createElement('div');
            modal.className = 'modal w-full fixed inset-0 z-50 flex items-center justify-center bg-black bg-opacity-50 p-4';
            modal.innerHTML = `
                <div class="bg-white p-6 rounded-lg shadow-lg w-full max-w-lg">
                    <h2 class="text-xl font-semibold mb-2 text-gray-800" id="modalTitle"><i class='bx bx-calendar-check text-2xl'></i> Confirmar Agendamento</h2>
                    <p class="text-gray-600 mb-4" id="modalDescription">Preencha seus dados e confirme o agendamento.</p>

                    <!-- Nav Tabs -->
                    <div class="flex mb-4 border-b">
                        <button class="hidden tab-btn px-4 py-2 text-gray-600 hover:text-blue-500 focus:outline-none border-b-2 border-transparent active" data-tab-target="dados"></button>
                        <button class="hidden tab-btn px-4 py-2 text-gray-600 hover:text-blue-500 focus:outline-none border-b-2 border-transparent" data-tab-target="confirmacao"></button>
                    </div>

                    <!-- Tab Content -->
                    <div class="tab-content">
                        <form method="POST" id="form-agendamento">
                            <input type="hidden" name="idprodserv" value="${$('#select-tiposevento').val()}">
                            <!-- Aba de Dados -->
                            <div class="tab-pane active" id="dados">
                                <div class="mb-4 space-y-4">
                                    <div>
                                        <label for="name" class="block text-sm font-medium text-gray-700">Nome</label>
                                        <input type="text" id="nome" name="nome" class="mt-1 block w-full p-2 border border-gray-300 rounded-md focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm transform capitalize">
                                    </div>
                                    <div>
                                        <label for="email" class="block text-sm font-medium text-gray-700">Email</label>
                                        <input type="email" id="email" name="email" class="mt-1 block w-full p-2 border border-gray-300 rounded-md focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm">
                                    </div>
                                    <div>
                                        <label for="phone" class="block text-sm font-medium text-gray-700">Telefone</label>
                                        <input type="text" id="telefone" maxlength="15" name="telefone" class="mt-1 block w-full p-2 border border-gray-300 rounded-md focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm">
                                    </div>
                                </div>
                                <hr class="my-4">
                                <div class="flex justify-end">
                                    <button onclick="document.querySelector('.modal').remove()" class="text-gray-600 font-semibold px-4 py-2 rounded-lg border border-2 hover:bg-red-600 hover:text-white transition-all duration-300">Cancelar</button>
                                    <button type="button" class="ml-2 bg-blue-500 text-white px-4 py-2 rounded-lg hover:bg-blue-600 transition-all duration-300" id="btnAvancar">Avançar</button>
                                </div>
                            </div>

                            <!-- Aba de Confirmação -->
                            <div class="tab-pane hidden" id="confirmacao">
                                <!-- Data escolhido e horário -->
                                <h1><b>${agenda.empresa.endereco.charAt(0).toUpperCase()}${agenda.empresa.endereco.slice(1)}, ${agenda.empresa.numero} - ${agenda.empresa.bairro.charAt(0).toUpperCase()}${agenda.empresa.bairro.slice(1)}</b></h1>

                                <!-- Endereço -->
                                <h3 class="text-gray-500 mb-4">${selectedDate.toLocaleDateString('pt-BR', { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' })}, ${slot}</h3>

                                <!-- Serviço -->
                                <div class="grid grid-cols-4 mb-4">
                                    <h2>Serviço</h2>
                                    <h2 class="col-span-3"><b>${$('#select-tiposevento').find(':selected').text()}</b></h2>
                                </div>

                                <!-- Colaborador -->
                                <div class="grid grid-cols-4 items-center mb-4">
                                    <h2>Colaborador</h2>
                                    <div class="flex flex-row gap-2 col-span-3">
                                        <span class="flex items-center justify-center border border-2 rounded-lg ps-3 pe-3 pt-2 pb-2"><i class='bx bxs-user text-2xl'></i></span>
                                        <div>
                                            <h2 class="capitalize"><b>{{ Str::limit($agenda->pessoa->nome, 29, '...') }}</b></h2>
                                            <h2>{{ $agenda->pessoa->email ?? ($agenda->pessoa->celular ?? $agenda->pessoa->celular2) }}</h2>
                                        </div>
                                    </div>
                                </div>

                                <!-- Cliente -->
                                <div class="grid grid-cols-4 items-center mb-4">
                                    <h2>Cliente</h2>
                                    <div class="flex flex-row gap-2 col-span-3">
                                        <span class="flex items-center justify-center border border-2 rounded-lg ps-3 pe-3 pt-2 pb-2"><i class='bx bxs-user text-2xl'></i></span>
                                        <div class="cols-12">
                                            <h2 class="font-bold capitalize" id="client-name">${$('#nome').val()}</h2>
                                            <h2 id="client-contact">${$('#email').val()}</h2>
                                        </div>
                                    </div>
                                </div>

                                <!-- Valor -->
                                <div class="grid grid-cols-4 mb-4 ${$('#select-tiposevento').find(':selected').data('valor') ? '' : 'hidden'}">
                                    <h2>Valor</h2>
                                    <h2><b>R$ ${$('#select-tiposevento').find(':selected').data('valor') || ''}</b></h2>
                                </div>

                                <!-- Duração -->
                                <div class="grid grid-cols-4 mb-5">
                                    <h2>Duração</h2>
                                    <h2 class="col-span-3">
                                        <b>
                                            ${(() => {
                                                const tipoReserva = $('#select-tiposevento').find(':selected').data('tipodereserva');
                                                if (tipoReserva === 'hora') {
                                                    const horapadrao = $('#select-tiposevento').find(':selected').data('horapadrao');
                                                    const [horas, minutos] = horapadrao.split(':').map(Number);
                                                    if (horas > 0) {
                                                        return `${horas} hora${horas > 1 ? 's' : ''}${minutos > 0 ? ` e ${minutos} minuto${minutos > 1 ? 's' : ''}` : ''}`;
                                                    } else {
                                                        return `${minutos} minuto${minutos > 1 ? 's' : ''}`;
                                                    }
                                                } else if (tipoReserva === 'data') {
                                                    const datapadrao = $('#select-tiposevento').find(':selected').data('datapadrao');
                                                    return `${datapadrao} dia${datapadrao > 1 ? 's' : ''}`;
                                                }
                                            })()}
                                        </b>
                                    </h2>
                                </div>

                                <!-- Endereço -->
                                <div class="grid grid-cols-4 mb-5">
                                    <h2>Endereço</h2>
                                    <div class="col-span-2">
                                        <a target="_blank" href="https://www.google.com/maps/search/?api=1&query=${agenda.empresa.endereco},${agenda.empresa.numero}-${agenda.empresa.bairro}" class="border border-2 rounded-lg ps-3 pe-3 pt-2 pb-2 w-full font-semibold"><i class='bx bx-map'></i> Abrir no Google Maps</a>
                                    </div>
                                </div>

                                <!-- Status -->
                                <div class="grid grid-cols-4 mb-4 items-center mb-4">
                                    <h2>Status</h2>
                                    <h2 class="bg-blue-100 rounded-lg text-center font-semibold p-1 col-span-2" id="status">Aguardando confirmação</h2>
                                </div>

                                {!! NoCaptcha::display() !!}

                                <hr class="mb-4">

                                <div class="flex justify-end mt-4">
                                    <button type="button" class="text-gray-600 font-semibold px-4 py-2 rounded-lg border border-2 hover:bg-red-600 hover:text-white transition-all duration-300" id="btnVoltar">Voltar</button>
                                    <button type="button" id="btnAgendar" class="ml-2 bg-blue-500 text-white px-4 py-2 rounded-lg hover:bg-blue-600">Agendar</button>
                                </div>
                            </>
                        </form>
                    </div>
                </div>
            `;
            return modal;
        };

        const addModalEventListeners = (modal) => {
            $(document).on('input', '#nome', updateClientName);
            $(document).on('input', '#email, #telefone', updateClientContact);

            document.getElementById('btnAvancar').addEventListener('click', handleAdvanceClick);
            document.getElementById('btnVoltar').addEventListener('click', handleBackClick);
            document.getElementById('btnAgendar').addEventListener('click', handleScheduleClick);
        };

        const showConfirmationTab = () => {
            document.querySelector('.tab-btn[data-tab-target="confirmacao"]').classList.add('active', 'border-blue-500');
            document.querySelector('.tab-pane#confirmacao').classList.remove('hidden');
            document.querySelector('.tab-btn[data-tab-target="dados"]').classList.remove('active', 'border-blue-500');
            document.querySelector('.tab-pane#dados').classList.add('hidden');
        };

        const showDataTab = () => {
            document.querySelector('.tab-btn[data-tab-target="dados"]').classList.add('active', 'border-blue-500');
            document.querySelector('.tab-pane#dados').classList.remove('hidden');
            document.querySelector('.tab-btn[data-tab-target="confirmacao"]').classList.remove('active', 'border-blue-500');
            document.querySelector('.tab-pane#confirmacao').classList.add('hidden');
        };

        // Event handlers
        const updateClientName = function() {
            const nome = $(this).val();
            const capitalizedNome = nome.replace(/\b\w/g, (char) => char.toUpperCase());
            $(this).val(capitalizedNome);
            $('#client-name').text(capitalizedNome);
        };

        const updateClientContact = function() {
            const email = $('#email').val();
            const telefone = $('#telefone').val();
            $('#telefone').val($('#telefone').val().replace(/\D/g, '').replace(/^(\d{2})(\d{1})(\d{4})(\d{4})$/, '($1) $2 $3-$4'));

            if (email && telefone) {
                $('#client-contact').text(`${email}`);
            } else if (email) {
                $('#client-contact').text(email);
            } else if (telefone) {
                $('#client-contact').text(telefone);
            }
        };

        const handleAdvanceClick = () => {
            if ($('#nome').val() && ($('#email').val() || $('#telefone').val())) {
                showConfirmationTab();
            } else {
                alert('Por favor, é necessário informar o nome e pelo menos um contato (email ou telefone).');
            }
        };

        const handleBackClick = () => {
            showDataTab();
        };

        const handleScheduleClick = (event) => {
            event.preventDefault();
            const selectedSlot = document.querySelector('#timeSlots .selected')?.textContent;
            if (!selectedSlot) {
                alert('Por favor, selecione um horário antes de agendar.');
                return;
            }
            const dados = gatherFormData(selectedSlot);
            submitAppointment(dados);
        };

        // API functions
        const fetchHorariosOcupados = async (data, idpessoa) => {
            try {
                const response = await fetch(`/api/getHorariosOcupados?data=${data}&idpessoa=${idpessoa}`);
                const eventos = await response.json();
                return eventos.map(evento => ({
                    horaInicio: evento.hora ? evento.hora.slice(0, 5) : null,
                    horafim: evento.horafim ? evento.horafim.slice(0, 5) : null,
                    data: evento.data,
                    datafim: evento.datafim
                }));
            } catch (error) {
                console.error("Erro ao buscar horários ocupados", error);
                return [];
            }
        };

        const submitAppointment = (dados) => {
            fetch(`{{ route('agenda.agendar', ['agenda' => $agenda]) }}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify(dados)
            })
            .then(response => response.json())
            .then(data => {
                updateModalAfterSubmission(data);
            })
            .catch(error => {
                console.error('Erro:', error);
            });
        };

        // Utility functions
        const temHorariosDisponiveis = (diaSemana) => {
            return agenda[`${diaSemana}_horainicio1`] !== null || agenda[`${diaSemana}_horainicio2`] !== null;
        };

        const gerarSlots = (diaSemana) => {
            const slots = [];
            const inicio1 = agenda[`${diaSemana}_horainicio1`];
            const fim1 = agenda[`${diaSemana}_horafim1`];
            const inicio2 = agenda[`${diaSemana}_horainicio2`];
            const fim2 = agenda[`${diaSemana}_horafim2`];

            adicionarSlots(inicio1, fim1);
            adicionarSlots(inicio2, fim2);

            return slots;

            function adicionarSlots(inicio, fim) {
                if (inicio && fim) {
                    let hora = new Date(`2000-01-01T${inicio}`);
                    const horaFim = new Date(`2000-01-01T${fim}`);
                    const tempoAgendamento = agenda.tempo.split(':');
                    const minutos = parseInt(tempoAgendamento[0]) * 60 + parseInt(tempoAgendamento[1]);

                    while (hora < horaFim) {
                        const slotTime = hora.toTimeString().slice(0, 5);
                        if (meetsMinimumAdvance(new Date(selectedDate.getFullYear(), selectedDate.getMonth(), selectedDate.getDate(), hora.getHours(), hora.getMinutes()))) {
                            slots.push(slotTime);
                        }
                        hora = new Date(hora.getTime() + minutos * 60000);
                    }
                }
            }
        };

        const converterParaMinutos = (hora) => {
            if (!hora) {
                return 0;
            }
            const [horas, minutos] = hora.split(':').map(Number);
            return horas * 60 + minutos;
        };

        const somarHorarios = (horapadrao, hora) => {
            const minutosHorapadrao = horapadrao.includes(':') ? converterParaMinutos(horapadrao) : Number(horapadrao);
            const [horas, minutos] = hora.split(':').map(Number);
            const totalMinutos = horas * 60 + minutos + minutosHorapadrao;

            const novasHoras = Math.floor(totalMinutos / 60) % 24;
            const novosMinutos = totalMinutos % 60;

            const ultrapassaMeiaNoite = Math.floor(totalMinutos / 60) >= 24;

            return {
                hora: `${String(novasHoras).padStart(2, '0')}:${String(novosMinutos).padStart(2, '0')}`,
                ultrapassaMeiaNoite
            };
        };

        const gatherFormData = (slot) => {
            const tipoReserva = $('#select-tiposevento').find(':selected').data('tipodereserva');
            const dados = {
                'ideventotipo': $('#select-tiposevento').val(),
                'idprodserv': $('#select-tiposevento').find(':selected').data('idprodserv'),
                'idagenda': agenda.idagenda.toLocaleString(),
                'idempresa': agenda.idempresa.toLocaleString(),
                'idpessoa': agenda.idpessoa.toLocaleString(),
                'evento': `${$('#select-tiposevento').find(':selected').text()} para ${$('#nome').val()}`,
                'item': $('#select-tiposevento').find(':selected').text(),
                'nome': $('#nome').val(),
                'email': $('#email').val(),
                'telefone': $('#telefone').val(),
                'data': selectedDate.toISOString().split('T')[0],
                'tipodereserva': tipoReserva,
            };

            if (tipoReserva === 'hora') {
                dados.horapadrao = $('#select-tiposevento').find(':selected').data('horapadrao');
                dados.hora = slot;

                const resultado = somarHorarios(dados.horapadrao, slot);
                dados.horafim = resultado.hora;
                dados.datafim = resultado.ultrapassaMeiaNoite ? new Date(selectedDate.getTime() + 86400000).toISOString().split('T')[0] : selectedDate.toISOString().split('T')[0];
            } else if (tipoReserva === 'data') {
                dados.datapadrao = $('#select-tiposevento').find(':selected').data('datapadrao');
                dados.datafim = new Date(selectedDate.getTime() + (dados.datapadrao * 86400000)).toISOString().split('T')[0];
            }

            dados.valoritem = parseFloat($('#select-tiposevento').find(':selected').data('valor').replace(',', '.'));
            dados.valorun = parseFloat($('#select-tiposevento').find(':selected').data('valor').replace(',', '.'));
            dados.qtd = 1;

            return dados;
        };

        const updateModalAfterSubmission = (data) => {
            $('#modalTitle').text('Agendado');
            $('#modalDescription').text('Seu agendamento foi realizado com sucesso.');
            if (data.status) {
                $('#status').text('Agendado');
                $('#status').removeClass('bg-blue-100').addClass('bg-green-200');
                $('#status').removeClass('col-span-2').addClass('col-span-1');
            }
            $('#btnVoltar').remove();
            $('#btnAgendar').remove();
        };

        // Initialize the application
        initializeAgendaInfo();
        renderCalendar();

        // Event listener for select change
        document.getElementById('select-tiposevento').addEventListener('change', () => {
            const data = selectDate.value;
            const idpessoa = agenda.idpessoa;
            if (data) {
                fetchHorariosOcupados(data, idpessoa);
            }
        });
    </script>
</body>
</html>
