<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use App\Models\FormaPagamento;
use Illuminate\Http\Request;
use App\Models\Nf as Model;
use App\Models\NfParcela;
use App\Models\NfItem;
use App\Models\ProdutoServico;
use App\Models\Pessoa;
use App\Models\TipoProdutoServico;
use App\Models\Veiculo;
use App\Services\EstoqueService;
use Exception;
use Illuminate\Support\Facades\DB;
use DateTime;
use Carbon\Carbon;

class PedidoController extends Controller
{
    private $estoqueService;

    public function __construct(EstoqueService $estoqueService)
    {
        $this->estoqueService = $estoqueService;
    }

    public function index(Request $request)
    {

        $module = $request->segment(2); // 'vendas' ou 'pedidos'
        $key = $module . '_dateRange'; // chave será 'vendas_dateRange' ou 'pedidos_dateRange'
    
        if (session($key)) {
            // Redirecionar para search com o date_range da session específica do módulo
            return redirect()->route('admin.pedidos.search', ['date_range' => session($key)]);
        }

        $pedidos = Model::pedido()
                    ->orderBy('dataentrada','desc')
                    ->paginate(1000);

        return view('admin.pedidos.index')->with(compact('pedidos'));
    }

    public function create(Pessoa $pessoaModel, Empresa $empresaModel, FormaPagamento $formaPagamentoModel, TipoProdutoServico $tipoProdutoServicoModel, Veiculo $veiculoModel)
    {
        $pessoas = $pessoaModel
                    ->orderBy('nome')
                    ->funcionarioOuCliente()
                    ->get();

        $tiposProdutoServico = $tipoProdutoServicoModel
                    ->active()
                    ->orderBy('prodservtipo')
                    ->get();

        $empresas = $empresaModel
                    //->where('idtenant', auth()->user()->idtenant)
                    ->orderBy('razaosocial')
                    ->get();

        $formaPagamento = $formaPagamentoModel
                    ->orderBy('formapagamento')
                    ->get();

        $veiculos = $veiculoModel
                    ->orderBy('placa')
                    ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.pedidos.create', compact('pessoas', 'empresas', 'formaPagamento','tiposProdutoServico','veiculos', 'statusOptions'));
    }

    public function store(Request $request, NfParcela $nfParcelaModel)
    {
        

        try
        {
            DB::beginTransaction();

            $pedidoCreate = $request->all();

            $pedido = Model::create($pedidoCreate);
                
            DB::commit();
            //return back()->with('success', 'pedido criada com sucesso!');
            return redirect()->to('admin/pedidos/'.$pedido->idnf.'/edit');
           
        } catch(Exception $e)
        {
            DB::rollBack();

            return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', $e->getMessage());
        }
    }

    public function edit($idNf, Pessoa $pessoaModel, Empresa $empresaModel, FormaPagamento $formaPagamentoModel,TipoProdutoServico $tipoProdutoServicoModel, NfItem $nfItemModel, ProdutoServico $prodservModel, Veiculo $veiculoModel,NfParcela $nfParcelaModel)
    {
        $pedido = Model::find($idNf);
        $pessoas = $pessoaModel
                        ->orderBy('nome')
                        ->funcionarioOuCliente()
                        ->get();

        $veiculos = $veiculoModel
                        ->orderBy('placa')
                        ->get();

        $empresas = $empresaModel
                        ->orderBy('razaosocial')
                        ->get();

        $tiposProdutoServico = $tipoProdutoServicoModel
                        ->orderBy('prodservtipo')
                        ->get();

        $formaPagamento = $formaPagamentoModel
                        ->orderBy('formapagamento')
                        ->get();

        $nfItem = $nfItemModel
                        ->where('idnf', $idNf) // Filtra pelos registros com o idnf relacionado
                        ->orderBy('idprodserv')
                        ->get();
        
        $produtoServico = $prodservModel
                        ->where('idprodservtipo', $pedido->idprodservtipo) // Filtra pelos registros com o idprodservtipo correspondente
                        ->orderBy('prodserv')
                        ->get();

        $nfParcela = $nfParcelaModel
                        ->where('idnf', $idNf) // Filtra pelos registros com a nf
                        ->orderBy('datavencimento')
                        ->get();

        if ($pedido->status == 'CONCLUÍDO'){
            $bg_status = "bg-success";
        }elseif($pedido->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        // Obter a quantidade de estoque atual para cada produto
        $estoqueAtualPorProduto = [];
        foreach ($produtoServico as $produto) {
            $estoqueAtualPorProduto[$produto->idprodserv] = $this->estoqueService->calcularEstoqueAtual($produto->idprodserv);
        }

        return view('admin.pedidos.edit', [ 
            'pedido' => $pedido,
            'pessoas' => $pessoas,
            'empresas' => $empresas,
            'formaPagamento' => $formaPagamento,
            'tiposProdutoServico' => $tiposProdutoServico,
            'nfItem' => $nfItem,
            'produtoServico' => $produtoServico,
            'veiculos' => $veiculos,
            'estoqueAtualPorProduto' => $estoqueAtualPorProduto,
            'bg_status' => $bg_status,
            'nfParcela' => $nfParcela,
            'statusOptions' => $statusOptions,
        ]);
    }

    public function update($idNf, Request $request)
    {
      
        $pedido = Model::find($idNf);
        $camposMonetarios = ['descontonf'];

        $pedidoUpdate = $request->all();
         // Trata os campos monetários
         foreach ($camposMonetarios as $campo) {
            if (isset($pedidoUpdate[$campo])) {
                $pedidoUpdate[$campo] = str_replace(',', '.', preg_replace('/[^\d,\.]/', '', $pedidoUpdate[$campo]));
                $pedidoUpdate[$campo] = floatval($pedidoUpdate[$campo]);
            }
        }

        try
        {
            $pedido->update($pedidoUpdate);
            if ($request->exists('idnfitem')){
                $input = $request->all();
                $condition = $input['idnfitem'];
                $camposMonetarios = ['qtd','valorun','valoritem' /* outros campos */];
                if (isset($condition)){
                    foreach ($condition as $key => $condition) {
                        $item = NfItem::find($input['idnfitem'][$key]);

                        if (isset($item)){

                            // Trata os campos monetários
                            foreach ($camposMonetarios as $campo) {
                                if (isset($input[$campo][$key])) {
                                    $input[$campo][$key] = str_replace(',', '.', preg_replace('/[^\d,\.]/', '', $input[$campo][$key]));
                                    $input[$campo][$key] = floatval($input[$campo][$key]);
                                }
                            }
                            if (isset($input['idprodserv'][$key]))
                                $item->idprodserv = $input['idprodserv'][$key];
                            $item->qtd = $input['qtd'][$key];
                            $item->idempresa = $pedido->idempresa;
                            $item->un = $input['un'][$key];
                            $item->valorun = $input['valorun'][$key];
                            $item->valoritem = $input['valoritem'][$key];
                            $item->save();

                        }

                        
                    }
                }
            }

          

            if ($pedido->status === 'CONCLUÍDO') {
                $this->geraNfParcelas($pedido);
            }

            return back()->with('success', 'Pedido atualizado!');

        } catch(Exception $e)
        {
            return redirect()
                    ->route('admin.pedidos.index')
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function destroy($idNf)
    {
        $empresa = Model::find($idNf);

        try
        {
            $empresa->delete();

            return redirect()
                    ->route('admin.pedidos.index')
                    ->with('success', 'Empresa removida!');
        } catch(Exception $e)
        {
            return redirect()
                    ->back()
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function search(Request $request)
    {
        $intervaloDatas = explode(' - ', $request->date_range);
       // dump($intervaloDatas);

        // Inicializa as variáveis $dataInicio e $dataFim como null ou strings vazias
        $dataInicio = '';
        $dataFim = '';

        // Verifica se existem pelo menos dois elementos no array
        if (count($intervaloDatas) >= 2) {
            $dataInicio = $intervaloDatas[0];
            $dataFim = $intervaloDatas[1];

            $dataInicio = Carbon::createFromFormat('d/m/y', $intervaloDatas[0])->format('Y-m-d');
            $dataFim = Carbon::createFromFormat('d/m/y', $intervaloDatas[1])->format('Y-m-d');
        }

        $pedidos =  Model::with(['nfitem.prodserv'])
                            ->pedido()
                            ->join('empresa', 'empresa.idempresa', '=', 'nf.idempresa')
                            ->when($request->date_range, function ($query) use ($dataInicio, $dataFim) {
                                $query->whereBetween('dataentrada', [$dataInicio, $dataFim]);
                            })
                            ->where(function ($query) use ($request) {
                                $query->where('idnf', 'like', '%' . $request->search . '%')
                                ->orwhereHas('nfitem.prodserv', function ($q) use ($request) {
                                    $q->where('prodserv', 'like', '%' . $request->search . '%');
                                })
                                ->orWhereHas('empresa', function ($q) use ($request) {
                                    $q->where('nomefantasia', 'like', '%' . $request->search . '%');
                                })
                                ->orWhereHas('empresa', function ($q) use ($request) {
                                    $q->where('razaosocial', 'like', '%' . $request->search . '%');
                                })
                                ->orWhereHas('pessoa', function ($q) use ($request) {
                                    $q->where('nome', 'like', '%' . $request->search . '%');
                                })
                                ->orWhereHas('pessoa', function ($q) use ($request) {
                                    $q->where('razaosocial', 'like', '%' . $request->search . '%');
                                })
                                ->orWhereHas('prodservtipo', function ($q) use ($request) { // Adicionando condição para prodservtipo
                                    $q->where('prodservtipo', 'like', '%' . $request->search . '%');
                                });
                            })
                            ->orderBy('dataentrada','desc')
                            ->paginate(1000);


        $size = $pedidos->count();

        // Colocando mensagens na session
        session()->forget('error');
        session()->forget('success');
        if ($size && $size > 0 )
        {
            session()->flash('success', $size.' valores econtrados.');

        } else
        {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        //Salvar Sessão DateRange e Busca
        $module = $request->segment(2); // 'vendas' ou 'pedidos'
       
        $key = $module . '_dateRange'; // chave será 'vendas_dateRange' ou 'pedidos_dateRange'
       
        if (session($key)) {
            $dateRange = session($key);
        }else{
            $dateRange = $request->date_range ? urldecode($request->date_range) : '';
        }

        // Armazenar dateRange na session com chave específica do módulo
        session([$key => $dateRange]);

        return view('admin.pedidos.index')->with(compact('pedidos','dateRange'));
    }

    public function removeitem($idNf){
        NfItem::find($idNf)->delete($idNf);
        return response()->json([
        'success' => 'Record deleted successfully!'
        ]);
    }

    public function additem($idNf){
        $nfitem = new NfItem();
        $pedido = Model::find($idNf);
        $nfitem->idnf = $idNf;
        $nfitem->idempresa = $pedido->idempresa;
        $nfitem->save();
        return response()->json([
        'success' => $nfitem->idnfitem
        ]);
    }


    
    public function checkDiaUtil($data) {
        $diaSemana = $data->format('N'); // 1 (segunda-feira) a 7 (domingo)
        return ($diaSemana >= 1 && $diaSemana <= 5); // Verificar se é um dia útil (segunda a sexta-feira)
    }
    
    public function ajustarDiaUtil($data) {
        // Verificar se o dia é sábado (6) ou domingo (7)
        if ($data->format('N') == 6) {
            // Adicionar 2 dias para ir para a próxima segunda-feira
            $data->modify('+2 days');
        } else {
            // Adicionar 1 dia para ir para a próxima segunda-feira
            $data->modify('+1 day');
        }
    
        return $data;
    }

    public function geraNfParcelas($pedido)
    {
        // Verificar se existem registros relacionados na tabela nfparcela
        $count = DB::table('nfparcela')
            ->where('idnf', $pedido->id)
            ->count();

        if ($count <= 0 and $pedido['qtdparcela'] > 0) {
            $vparcela = $pedido['totalnf'] / $pedido['qtdparcela'];
            $parcelas = [];

            $dataVencimento = DateTime::createFromFormat('Y-m-d', $pedido['datavencimento']);
            $dataVencimento->setTime(0, 0, 0);
            $dataVencimentoparcela = clone $dataVencimento;

            for ($i = 0; $i < $pedido['qtdparcela']; $i++) {
                $dataVencimentoparcela = clone $dataVencimento;
                $dataVencimento->modify('+1 month');

                if (!$this->checkDiaUtil($dataVencimentoparcela)) {
                    $dataVencimentoparcela = $this->ajustarDiaUtil($dataVencimentoparcela);
                }

                $parcelas[$i] = [
                    'tipo' => $pedido->tipo,
                    'parcela' => $i + 1,
                    'idempresa' => (int) $pedido->idempresa,
                    'valor' => $vparcela,
                    'datavencimento' => $dataVencimentoparcela,
                    'status' => 'ATIVO'
                ];
            }

            $pedido->parcela()->createMany($parcelas);
        }
    }


}
