<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

class PermissionController extends Controller
{
    protected $permissionsModel;
    protected $rolesModel;
    protected $permissions;
    protected $pagesPermission;

    public function __construct(
        Permission $permissionsModel, Role $rolesModel
    )
    {
        $this->permissionsModel = $permissionsModel;
        $this->rolesModel = $rolesModel;
        $this->permissions = [
            'view' => 'Visualizar',
            'create' => 'Cadastrar',
            'edit' => 'Alterar',
            'remove' => 'Remover'
        ];

        $this->pagesPermission = [
            'dashboard' => 'Dashboard',
            'profiles' => 'Perfil',
            'profile' => 'Perfil do usúario',
            'userManagement' => 'Administração de usúarios',
            'products' => 'Produtos',
            'categories' => 'Categorias',
            'config' => 'Configurações',
            'permissions' => 'Permissões'
        ];
    }

    public function index()
    {
        $roles = $this->rolesModel
                        ->paginate(100);

        return view('admin.permissions.index')
                ->with(compact('roles'));
    }

    public function create()
    {
        $permissions = $this->permissions;
        $pagesPermission = $this->pagesPermission;

        return view('admin.permissions.create')
                ->with(compact('permissions', 'pagesPermission'));
    }

    public function store(Request $request)
    {
        $permissions = $request->permissions;

        $role = Role::findOrCreate($request->role);
        foreach($permissions as $permission)
        {
            $permissionsCreate = Permission::findOrCreate($permission);

            $role->givePermissionTo($permissionsCreate);
        }

        return redirect()
                ->route('admin.permissions.index')
                ->with('success', 'Permissão cadastrada com sucesso.');
    }

    public function edit($id)
    {
        $role = $this->rolesModel->find($id);
        $allPermissions = array_merge($this->permissions, $this->pagesPermission);
        $permissions = $role->getAllPermissions();

        return view('admin.permissions.edit')
                ->with(compact('role', 'allPermissions', 'permissions'));
    }

    public function update(Request $request, $id)
    {
        $role = $this->rolesModel->find($id);
        $permissions = $role->getAllPermissions();
        $permissionsUpdate = $request->permissions;

        // dd($permissionsUpdate, $permissions);

        // Adicionando permissoes novas
        foreach($permissionsUpdate as $permission)
        {
            $permissionsCreate = Permission::findOrCreate($permission);

            foreach($permissions as $rolePermission)
            {
                if(!in_array($rolePermission->name, $permissionsUpdate))
                {
                    $role->revokePermissionTo($rolePermission->name);
                }
            }

            if(!$role->hasPermissionTo($permissionsCreate->name))
            {
                $role->givePermissionTo($permissionsCreate);
            }
        }

        return redirect()->route('admin.permissions.index')->with('success', 'Permissão atualizada com sucesso.');
    }

    public function destroy($id)
    {
        $role = $this->rolesModel->find($id);
        $permissions = $role->getAllPermissions();

        if ($role) {
            // Removendo permissoes antigas
            foreach($permissions as $permission)
            {
                $permission->delete();
            }
            if ($role->delete()) {
                return back()
                        ->with('success', 'Permissão apagada com sucesso.');
            } else {
                return redirect()
                        ->back()
                        ->with('error', 'Um erro ocorreu, tente novamente!')
                        ->withInput();
            }
        } else {
            return redirect()
                    ->back()
                    ->with('error', 'Um erro ocorreu, tente novamente!')
                    ->withInput();
        }
    }

    public function search(Request $request)
    {
        $roles =  $this->rolesModel
                        ->where('name', 'like', '%'.$request->search.'%')
                        ->orderBy('name')
                        ->paginate(15);

        $size = $roles->count();

        // Colocando mensagens na session
        session()->forget('error');
        session()->forget('success');
        if ($size && $size > 0 )
        {
            session()->flash('success', $size.' valores econtrados.');

        } else
        {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        return view('admin.permissions.index')
                ->with(compact('roles'));
    }
}
