<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use Illuminate\Http\Request;
use App\Models\ProdutoServico as Model;
use App\Models\TipoProdutoServico;
use App\Services\EstoqueService;
use Exception;

class ProdutoServicoController extends Controller
{
    private $estoqueService;

    public function __construct(EstoqueService $estoqueService)
    {
        $this->estoqueService = $estoqueService;
    }
    protected $unidadesDeMedida = [
        'BOBINA' => 'BOBINA',
        'CX' => 'CAIXA',
        'EA' => 'EA',
        'EQUIP' => 'EQUIPAMENTO',
        'FL' => 'FOLHA',
        'FR' => 'FRASCO',
        'GL' => 'GALAO',
        'GR' => 'GRAMA',
        'KG' => 'KILOGRAMA',
        'LT' => 'LITRO',
        'MG' => 'MG',
        'mL' => 'ML',
        'MT' => 'METRO',
        'Pares' => 'PARES',
        'PC' => 'PEÇA',
        'PCT' => 'PACOTE',
        'ROLO' => 'ROLO',
        'SACO' => 'SACO',
        'TB' => 'TUBO',
        'UI' => 'UI',
        'UN' => 'UNIDADE',
        'DS' => 'DOSE',
        'POT10' => 'POTÊNCIA 10',
        'MIL' => 'MILHEIRO',
        'TESTES' => 'TESTES',
        'BLD' => 'BALDE',
        'FDO' => 'FARDO',
        'TA' => 'TAMBOR',
        'TO' => 'TONELADA',
        'CM' => 'CENTÍMETRO',
        'M2' => 'm²',
        'M3' => 'M³',
        'JOGO' => 'JG',
    ];

    public function index()
    {
        $produtosServicos = Model::with('prodservtipo')->orderBy('prodserv')
        ->orderBy('tipo')
        ->orderBy('idprodservtipo')
        ->orderBy('prodserv')
        ->paginate(100);

        // Obter a quantidade de estoque atual para cada produto
        $estoqueAtualPorProduto = [];
        foreach ($produtosServicos as $produtoServico) {
            $estoqueAtualPorProduto[$produtoServico->idprodserv] = $this->estoqueService->calcularEstoqueAtual($produtoServico->idprodserv);
        }

        return view('admin.produto-servico.index')->with(compact('produtosServicos','estoqueAtualPorProduto'));
    }

    public function create(Empresa $empresaModel, TipoProdutoServico $tipoProdutoServicoModel)
    {
        $empresas = $empresaModel
                        ->orderBy('razaosocial')
                        ->get();

        $prodservTipo = $tipoProdutoServicoModel
                                ->active()
                                ->orderBy('prodservtipo')
                                ->get();

        $unidadesDeMedida = $this->unidadesDeMedida;

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.produto-servico.create')->with(compact('empresas', 'prodservTipo', 'unidadesDeMedida', 'statusOptions'));
    }

    public function store(Request $request)
    {
        $produtoServicoCreate = $request->all();

        try
        {
            $prodserv = Model::create($produtoServicoCreate);

            return redirect()->to('admin/produto-servico/'.$prodserv->idprodserv.'/edit');

        } catch(Exception $e)
        {
            return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', $e->getMessage());
        }
    }

    public function edit($idProdutoServico, Empresa $empresaModel, TipoProdutoServico $tipoProdutoServicoModel)
    {
        $produtoServico = Model::find($idProdutoServico);
        $empresas = $empresaModel
                    ->orderBy('razaosocial')
                    ->get();
        $prodservTipo = $tipoProdutoServicoModel
            ->active()
            ->orderBy('prodservtipo')
            ->get();

        if ($produtoServico->status == 'ATIVO'){
            $bg_status = "bg-success";
        }elseif($produtoServico->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        $unidadesDeMedida = $this->unidadesDeMedida;

        // Obter a quantidade de estoque atual para cada produto
        $estoqueAtualPorProduto = $this->estoqueService->calcularEstoqueAtual($idProdutoServico);
       
        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.produto-servico.edit')->with(compact('produtoServico', 'prodservTipo', 'empresas', 'unidadesDeMedida','estoqueAtualPorProduto','bg_status', 'statusOptions'));
    }

    public function update($idProdutoServico, Request $request) 
    {
        $produtoServico = Model::find($idProdutoServico);

        try
        {
            $produtoServico->update($request->all());

            return back()->with('success', 'Produto atualizado!');
            
        } catch(Exception $e)
        {
            return redirect()
                    ->route('admin.produto-servico.index')
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function destroy($idProdutoServico)
    {
        $produtoServico = Model::find($idProdutoServico);

        try
        {
            $produtoServico->delete();

            return redirect()
                    ->route('admin.produto-servico.index')
                    ->with('success', 'Produto/Serviço removido!');
        } catch(Exception $e)
        {
            return redirect()
                    ->back()
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function search(Request $request)
    {
        $produtosServicos =  Model::query()
        ->join('empresa', 'empresa.idempresa', '=', 'prodserv.idempresa')
        ->where(function ($query) use ($request) {        
            $query->where('prodserv', 'like', '%' . $request->search . '%') // Adicionando a busca por 'marca'
            ->orwhere('tipo', 'like', '%' . $request->search . '%')
            ->orwhereHas('prodservtipo', function ($q) use ($request) {
                $q->where('prodservtipo', 'like', '%' . $request->search . '%');
            })
            ->orwhereHas('empresa', function ($q) use ($request) {
                $q->where('nomefantasia', 'like', '%' . $request->search . '%');
            })
            ->orWhereHas('empresa', function ($q) use ($request) {
                $q->where('razaosocial', 'like', '%' . $request->search . '%');
            });
        })
        ->orderBy('empresa.sigla')
        ->orderBy('prodserv.prodserv')
        ->paginate(100);

        $size = $produtosServicos->count();

        // Colocando mensagens na session
        session()->forget('error');
        session()->forget('success');
        if ($size && $size > 0 )
        {
            session()->flash('success', $size.' valor(es) econtrados.');

        } else
        {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        // Obter a quantidade de estoque atual para cada produto
        $estoqueAtualPorProduto = [];
        foreach ($produtosServicos as $produtoServico) {
            $estoqueAtualPorProduto[$produtoServico->idprodserv] = $this->estoqueService->calcularEstoqueAtual($produtoServico->idprodserv);
        }

        return view('admin.produto-servico.index')
                ->with(compact('produtosServicos','estoqueAtualPorProduto'));
    }

    
}
