<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use App\Models\FormaPagamento;
use Illuminate\Http\Request;
use App\Models\Nf as Model;
use App\Models\NfParcela;
use App\Models\NfItem;
use App\Models\ProdutoServico;
use App\Models\Pessoa;
use App\Models\TipoProdutoServico;
use App\Models\Veiculo;
use App\Models\Evento;

use App\Services\EstoqueService;
use Exception;
use Illuminate\Support\Facades\DB;
use DateTime;
use Carbon\Carbon;

class VendaController extends Controller
{
    private $estoqueService;

    public function __construct(EstoqueService $estoqueService)
    {
        $this->estoqueService = $estoqueService;
    }

    public function index(Request $request)
    {

        $module = $request->segment(2); // 'vendas' ou 'pedidos'
        $key = $module . '_dateRange'; // chave será 'vendas_dateRange' ou 'pedidos_dateRange'
    
        if (session($key)) {
            // Redirecionar para search com o date_range da session específica do módulo
            return redirect()->route('admin.vendas.search', ['date_range' => session($key)]);
        }

        $vendas = Model::latest()
                    ->venda()
                    ->orderBy('datavencimento','desc')
                    ->paginate(100);

        return view('admin.vendas.index')->with(compact('vendas'));
    }

    public function create(Pessoa $pessoaModel, Empresa $empresaModel, FormaPagamento $formaPagamentoModel, TipoProdutoServico $tipoProdutoServicoModel, Veiculo $veiculoModel)
    {
        $pessoas = $pessoaModel
                    ->where('status', 'Ativo')
                    ->orderBy('nome')
                    ->funcionarioOuCliente()
                    ->get();

        $tiposProdutoServico = $tipoProdutoServicoModel
                    ->active()
                    ->orderBy('prodservtipo')
                    ->get();

        $empresas = $empresaModel
                    //->where('idtenant', auth()->user()->idtenant)
                    ->orderBy('razaosocial')
                    ->get();

        $formaPagamento = $formaPagamentoModel
                    ->orderBy('formapagamento')
                    ->get();

        $veiculos = $veiculoModel
                    ->orderBy('placa')
                    ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.vendas.create', compact('pessoas', 'empresas', 'formaPagamento','tiposProdutoServico','veiculos','statusOptions'));
    }

    public function store(Request $request, NfParcela $nfParcelaModel)
    {
        

        try
        {
            DB::beginTransaction();

            $vendaCreate = $request->all();

            $venda = Model::create($vendaCreate);

            if ($venda->status == 'ATIVO'){
                $this->sincronizaEvento($venda);
            }
                
            DB::commit();
            //return back()->with('success', 'venda criada com sucesso!');
            return redirect()->to('admin/vendas/'.$venda->idnf.'/edit');
           
        } catch(Exception $e)
        {
            DB::rollBack();

            return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', $e->getMessage());
        }
    }

    public function edit($idNf, Pessoa $pessoaModel, Empresa $empresaModel, FormaPagamento $formaPagamentoModel,TipoProdutoServico $tipoProdutoServicoModel, NfItem $nfItemModel, ProdutoServico $prodservModel, Veiculo $veiculoModel,NfParcela $nfParcelaModel, Evento $eventoModel)
    {
        $venda = Model::find($idNf);
        $pessoas = $pessoaModel
                        ->orderBy('nome')
                        ->funcionarioOuCliente()
                        ->get();

        $veiculos = $veiculoModel
                        ->orderBy('placa')
                        ->get();

        $empresas = $empresaModel
                        ->orderBy('razaosocial')
                        ->get();

        $tiposProdutoServico = $tipoProdutoServicoModel
                        ->orderBy('prodservtipo')
                        ->get();

        $formaPagamento = $formaPagamentoModel
                        ->orderBy('formapagamento')
                        ->get();
  
        $nfItem = $nfItemModel
                        ->where('idnf', $idNf) // Filtra pelos registros com o idnf relacionado
                        ->orderBy('idprodserv')
                        ->get();

        $nfItem = $nfItemModel
            ->leftJoin('evento', 'nfitem.idnfitem', '=', 'evento.idnfitem') // Substitua 'eventos' pelo nome da tabela de eventos e 'nf_items' pelo nome da tabela de itens da nota fiscal
            ->leftJoin('eventotipo', 'eventotipo.ideventotipo', '=', 'evento.ideventotipo') 
            ->where('nfitem.idnf', $idNf) // Certifique-se de usar o nome da tabela correto antes do nome da coluna
            ->orderBy('evento.datafim')
            ->orderBy('evento.horafim')
            ->orderBy('nfitem.idprodserv')
            ->get([
                'nfitem.*', // Seleciona todas as colunas da tabela nf_items
                'evento.evento as evento',
                'evento.idevento as evento_idevento',
                'evento.data as evento_data',
                'evento.hora as evento_hora',
                'eventotipo.eventotipo'  // Substitua 'campo_desejado' pelo nome do campo que você deseja selecionar da tabela eventos
            ]);
        
        $produtoServico = $prodservModel
                        ->where('idprodservtipo', $venda->idprodservtipo) // Filtra pelos registros com o idprodservtipo correspondente
                        ->orderBy('prodserv')
                        ->get();

        $nfParcela = $nfParcelaModel
                        ->where('idnf', $idNf) // Filtra pelos registros com a nf
                        ->orderBy('datavencimento')
                        ->get();
        
        $eventos = $eventoModel
                        ->where('idcliente',$venda->idpessoa)
                        ->where('status', 'ATIVO')
                        ->where(function ($query) {
                            $query->whereNull('idnfitem')
                                  ->orWhere('idnfitem', '');
                        })
                        ->get();
                        

        if ($venda->status == 'CONCLUÍDO'){
            $bg_status = "bg-success";
        }elseif($venda->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];
        // Obter a quantidade de estoque atual para cada produto
        $estoqueAtualPorProduto = [];
        foreach ($produtoServico as $produto) {
            $estoqueAtualPorProduto[$produto->idprodserv] = $this->estoqueService->calcularEstoqueAtual($produto->idprodserv);
        }

        return view('admin.vendas.edit', [ 
            'venda' => $venda,
            'pessoas' => $pessoas,
            'empresas' => $empresas,
            'formaPagamento' => $formaPagamento,
            'tiposProdutoServico' => $tiposProdutoServico,
            'nfItem' => $nfItem,
            'produtoServico' => $produtoServico,
            'veiculos' => $veiculos,
            'estoqueAtualPorProduto' => $estoqueAtualPorProduto,
            'bg_status' => $bg_status,
            'nfParcela' => $nfParcela,
            'eventos' => $eventos,
            'statusOptions' => $statusOptions
        ]);
    }

    public function update($idNf, Request $request)
    {
      
        $venda = Model::find($idNf);
        $camposMonetarios = ['descontonf'];

        $vendaUpdate = $request->all();
         // Trata os campos monetários
         foreach ($camposMonetarios as $campo) {
            if (isset($vendaUpdate[$campo])) {
                $vendaUpdate[$campo] = str_replace(',', '.', preg_replace('/[^\d,\.]/', '', $vendaUpdate[$campo]));
                $vendaUpdate[$campo] = floatval($vendaUpdate[$campo]);
            }
        }
        try
        {
            $venda->update($vendaUpdate);
            if ($request->exists('idnfitem')){
                $input = $request->all();
                $condition = $input['idnfitem'];
                $camposMonetarios = ['qtd','valorun','valoritem' /* outros campos */];
                if (isset($condition)){
                    foreach ($condition as $key => $condition) {
                        $item = NfItem::find($input['idnfitem'][$key]);

                        // Trata os campos monetários
                        foreach ($camposMonetarios as $campo) {
                            if (isset($input[$campo][$key])) {
                                $input[$campo][$key] = str_replace(',', '.', preg_replace('/[^\d,\.]/', '', $input[$campo][$key]));
                                $input[$campo][$key] = floatval($input[$campo][$key]);
                            }
                        }

                        if (isset($input['idprodserv'][$key]))
                            $item->idprodserv = $input['idprodserv'][$key];
                        $item->qtd = $input['qtd'][$key];
                        $item->idempresa = $venda->idempresa;
                        $item->un = $input['un'][$key];
                        $item->valorun = $input['valorun'][$key];
                        $item->valoritem = $input['valoritem'][$key];
                        $item->save();
                    }
                }
            }
        
            if ($venda->status == 'ATIVO'){
                $this->sincronizaEvento($venda);
            }


            $vendaUpdate = $request->all();

            $status = $vendaUpdate['status'];
            $totalnf = $vendaUpdate['totalnf'];
            $qtdparcela = $vendaUpdate['qtdparcela'];
            $datavencimento = $vendaUpdate['datavencimento'];

            if ($status == 'CONCLUÍDO'){
                // Verificar se existem registros relacionados na tabela nfparcela
                $count = DB::table('nfparcela')
                ->where('idnf', $idNf)
                ->count();
               $vparcela = 0;
                if ($count <= 0) {
                    $vparcela = $totalnf / $qtdparcela;
                    
                    $parcelas = [];

                    $dataVencimento = DateTime::createFromFormat('Y-m-d', $datavencimento);
                    $dataVencimento->setTime(0, 0, 0);
                    $dataVencimentoparcela = clone $dataVencimento; // Inicializa a variável aqui

                     
                  
                       // Iterar para calcular as demais datas de vencimento
                       for ($i = 0; $i < $qtdparcela; $i++) {
                            
                            // Clonar a data de vencimento anterior para criar uma nova instância
                            $dataVencimentoparcela = clone $dataVencimento;

                            // Adicionar 1 mês à data de vencimento
                            $dataVencimento->modify('+1 month');
                                
                    
                            // Verificar se o dia é válido
                            if (!$this->checkDiaUtil($dataVencimentoparcela)) {
                                // Se não for um dia útil, ajustar para a próxima segunda-feira
                                $dataVencimentoparcela = $this->ajustarDiaUtil($dataVencimentoparcela);
                            }
                            
                            // Adicionar a data de vencimento ao array
                            $parcelas[$i] = [
                                'tipo' => $venda->tipo,
                                'parcela' => $i + 1,
                                'idempresa' => (int) $venda->idempresa,
                                'valor' => $vparcela,
                                'datavencimento' => $dataVencimentoparcela,
                                'status' => 'ATIVO'
                            ];
                            
                        }
                        //dd($parcelas);



                    // TODO: gerar parcelas
                    $venda->parcela()->createMany($parcelas);
                    
                }
                
            }

            return back()->with('success', 'Venda atualizado!');

        } catch(Exception $e)
        {
            return redirect()
                    ->route('admin.vendas.index')
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function destroy($idNf)
    {
        $empresa = Model::find($idNf);

        try
        {
            $empresa->delete();

            return redirect()
                    ->route('admin.vendas.index')
                    ->with('success', 'Empresa removida!');
        } catch(Exception $e)
        {
            return redirect()
                    ->back()
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function search(Request $request)
    {
        $intervaloDatas = explode(' - ', $request->date_range);
       // dump($intervaloDatas);

        // Inicializa as variáveis $dataInicio e $dataFim como null ou strings vazias
        $dataInicio = '';
        $dataFim = '';

        // Verifica se existem pelo menos dois elementos no array
        if (count($intervaloDatas) >= 2) {
            $dataInicio = $intervaloDatas[0];
            $dataFim = $intervaloDatas[1];

            $dataInicio = Carbon::createFromFormat('d/m/y', $intervaloDatas[0])->format('Y-m-d');
            $dataFim = Carbon::createFromFormat('d/m/y', $intervaloDatas[1])->format('Y-m-d');
        }

        $vendas =  Model::with(['nfitem.prodserv'])
                            ->join('empresa', 'empresa.idempresa', '=', 'nf.idempresa')
                            ->when($request->date_range, function ($query) use ($dataInicio, $dataFim) {
                                $query->whereBetween('datavencimento', [$dataInicio, $dataFim]);
                            })
                            ->where('nf.tipo', '=', 'C') 
                            ->where(function ($query) use ($request) {
                                $query->where('idnf', 'like', '%' . $request->search . '%')
                                ->orwhereHas('nfitem.prodserv', function ($q) use ($request) {
                                    $q->where('prodserv', 'like', '%' . $request->search . '%');
                                })
                                ->orWhereHas('empresa', function ($q) use ($request) {
                                    $q->where('nomefantasia', 'like', '%' . $request->search . '%');
                                })
                                ->orWhereHas('empresa', function ($q) use ($request) {
                                    $q->where('razaosocial', 'like', '%' . $request->search . '%');
                                })
                                ->orWhereHas('pessoa', function ($q) use ($request) {
                                    $q->where('nome', 'like', '%' . $request->search . '%');
                                })
                                ->orWhereHas('pessoa', function ($q) use ($request) {
                                    $q->where('razaosocial', 'like', '%' . $request->search . '%');
                                })
                                ->orWhereHas('prodservtipo', function ($q) use ($request) { // Adicionando condição para prodservtipo
                                    $q->where('prodservtipo', 'like', '%' . $request->search . '%');
                                });
                            })
                            ->orderBy('datavencimento','desc')
                            ->paginate(100);


        $size = $vendas->count();

        // Colocando mensagens na session
        session()->forget('error');
        session()->forget('success');
        if ($size && $size > 0 )
        {
            session()->flash('success', $size.' valores econtrados.');

        } else
        {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

         //Salvar Sessão DateRange e Busca
         $module = $request->segment(2); // 'vendas' ou 'pedidos'

         $key = $module . '_dateRange'; // chave será 'vendas_dateRange' ou 'pedidos_dateRange'
         
         if (session($key)) {
             $dateRange = session($key);
         }else{
             $dateRange = $request->date_range ? urldecode($request->date_range) : '';
         }
 
         // Armazenar dateRange na session com chave específica do módulo
         session([$key => $dateRange]);

        return view('admin.vendas.index')->with(compact('vendas','dateRange'));
    }
    

    public function removeitem($idNf){
        NfItem::find($idNf)->delete($idNf);
        return response()->json([
        'success' => 'Record deleted successfully!'
        ]);
    }

    public function additem($idNf){
        $nfitem = new NfItem();
        $venda = Model::find($idNf);
        $nfitem->idnf = $idNf;
        $nfitem->idempresa = $venda->idempresa;
        $nfitem->save();
        return response()->json([
        'success' => $nfitem->idnfitem
        ]);
    }


    
    public function checkDiaUtil($data) {
        $diaSemana = $data->format('N'); // 1 (segunda-feira) a 7 (domingo)
        return ($diaSemana >= 1 && $diaSemana <= 5); // Verificar se é um dia útil (segunda a sexta-feira)
    }
    
    public function ajustarDiaUtil($data) {
        // Verificar se o dia é sábado (6) ou domingo (7)
        if ($data->format('N') == 6) {
            // Adicionar 2 dias para ir para a próxima segunda-feira
            $data->modify('+2 days');
        } else {
            // Adicionar 1 dia para ir para a próxima segunda-feira
            $data->modify('+1 day');
        }
    
        return $data;
    }

    public function sincronizaEvento($venda)
    {
                // Verifica se existem eventos ativos
                $eventosAtivos = Evento::query()
                ->join('eventotipo', 'eventotipo.ideventotipo', '=', 'evento.ideventotipo')
                ->join('prodserv', 'eventotipo.idprodserv', '=', 'prodserv.idprodserv')
                ->where('evento.idcliente', $venda->idpessoa)
                ->where('evento.status', 'ATIVO')
                ->where(function ($query) {
                    $query->whereNull('evento.idnfitem')
                        ->orWhere('evento.idnfitem', '');
                })
                ->select('evento.*', 'prodserv.valorvenda as prodserv_valorvenda','prodserv.idprodserv')
                ->get();
                // dd($eventosAtivos); // ou use dump($eventosAtivos);

    
        
                
                if ($eventosAtivos->count() > 0) {

                    foreach ($eventosAtivos as $evento) {

                        $nfitem = new NfItem();
                        $nfitem->idnf = $venda->idnf;
                        $nfitem->idempresa = $venda->idempresa;
                        $nfitem->idprodserv = $evento->idprodserv;
                        
                        // Combine data e hora:
                        $startDateTime = Carbon::parse($evento->data . ' ' . $evento->hora);

                        // Combine datafim e horafim:
                        $endDateTime = Carbon::parse($evento->datafim . ' ' . $evento->horafim);

                        // Calcule a diferença:
                        $duration = $startDateTime->diff($endDateTime);

                        // Formate a diferença para mostrar apenas horas e minutos:
                        $durationString = $duration->format('%H:%I');

                        // Separar horas e minutos
                        list($hours, $minutes) = explode(':', $durationString);

                        // Converter horas e minutos para decimal
                        $hoursInDecimal = $hours + ($minutes / 60);
                        
                        if ($evento->qtd > 0){
                            $nfitem->qtd = $evento->qtd;
                            $nfitem->valorun = $evento->valorun;
                            $nfitem->valoritem = $evento->valoritem;
                        }else{
                            $nfitem->qtd = $hoursInDecimal;
                            $nfitem->valorun = $evento->prodserv_valorvenda;
                            $nfitem->valoritem = $evento->prodserv_valorvenda*$hoursInDecimal;
                        }
                        
                       
                        $nfitem->save();
                        $idnfitem = $nfitem->idnfitem;
                        // Atualizar o status do evento para "PROGRAMADO"
                        Evento::where('idevento', $evento->idevento)->update(['status' => 'PROGRAMADO', 'idnfitem' => $idnfitem]);
                    }
                }
    }

  

}
