<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Agenda;
use App\Models\AgendaEventoTipo;
use App\Models\Empresa;
use App\Models\Evento;
use App\Models\EventoTipo;
use App\Models\Pessoa;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AgendaController extends Controller
{
    public function index(Agenda $agendaModel)
    {
        $agendas = $agendaModel
            ->with(['pessoa', 'agendaEventosTipos'])
            ->orderBy('nome')
            ->get();

        return view('admin.agendas.index', compact('agendas'));
    }

    public function show(Pessoa $pessoa)
    {
        $agenda = Agenda::
            with('pessoa', 'agendaEventosTipos.eventoTipo.prodserv', 'empresa')
            ->where('idpessoa', $pessoa->idpessoa)
            ->first();

        return view('pages.agenda', compact('agenda'));
    }

    public function agendar(Request $request)
    {
        $response = $request->all();
        $tipoReserva = $request->input('tipodereserva');

        $cliente = Pessoa::firstOrCreate(
            ['email' => $response['email']],
            [
                'idempresa' => $response['idempresa'],
                'nome' => $response['nome'],
                'email' => $response['email'] ?? null,
                'telefone' => $response['telefone'] ?? null,
            ]
        );

        $eventoData = [
            'ideventotipo' => $response['ideventotipo'],
            'idempresa' => $response['idempresa'],
            'idpessoa' => $response['idpessoa'],
            'idcliente' => $cliente->idpessoa,
            'idprodserv' => $response['idprodserv'],
            'evento' => $response['evento'],
            'status' => 'PENDENTE',
            'data' => $response['data'],
            'qtd' => $response['qtd'],
            'item' => $response['item'],
            'valoritem' => $response['valoritem'],
            'valorun' => $response['valorun'],
            'nome' => $cliente->nome,
            'telefone' => $cliente->telefone ?? null,
            'email' => $cliente->email ?? null,
        ];

        if ($tipoReserva === 'hora') {
            $eventoData['hora'] = $response['hora'];
            $eventoData['horafim'] = $response['horafim'];
            $eventoData['datafim'] = $response['datafim'];
        } else if ($tipoReserva === 'data') {
            $eventoData['datafim'] = $response['datafim'];
            $eventoData['horafim'] = '23:59:59';
            $eventoData['hora'] = '00:00:00';
        }

        $evento = Evento::create($eventoData);

        return response()->json(['status' => $evento->status]);
    }

    public function getHorariosOcupados(Request $request)
    {
        $request->validate([
            'data' => 'required|date',
            'idpessoa' => 'required|integer'
        ]);

        $data = $request->input('data');
        $idpessoa = $request->input('idpessoa');

        $eventos = Evento::where('data', '<=', $data)
            ->where('datafim', '>=', $data)
            ->where('idpessoa', $idpessoa)
            ->get(['hora', 'horafim', 'data', 'datafim']);

        return response()->json($eventos);
    }

    public function create(Empresa $empresaModel, Pessoa $pessoaModel, EventoTipo $eventoTipoModel)
    {
        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

        $pessoas = $pessoaModel
            ->where('temagenda', true)
            ->funcionarioOuCliente()
            ->orderBy('nome')
            ->get();

        $eventoTipos = $eventoTipoModel
                    ->with('prodserv')
                    ->orderBy('eventotipo')
                    ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.agendas.create', compact('empresas', 'statusOptions', 'pessoas', 'eventoTipos'));
    }

    public function store(Request $request, Agenda $agendaModel, AgendaEventoTipo $agendaEventoTipoModel)
    {
        $agendaRequest = $request->all();

        // Tratamento para toggle coletiva
        if (isset($agendaRequest['coletiva']) && $agendaRequest['coletiva'] == 'on') {
            $agendaRequest['coletiva'] = true;
        } else {
            $agendaRequest['coletiva'] = false;
        }

        // Tratamento para toggle online
        if (isset($agendaRequest['online']) && $agendaRequest['online'] == 'on') {
            $agendaRequest['online'] = true;
        } else {
            $agendaRequest['online'] = false;
        }

        DB::beginTransaction();
        try {
            $agenda = $agendaModel->create($agendaRequest);

            if (isset($agendaRequest['tiposevento'])) {
                $eventoTipos = $agendaRequest['tiposevento'];
                // Vincula eventos tipos a agenda
                foreach ($eventoTipos as $key => $value) {
                    $agendaEventoTipoModel->create([
                        'idagenda' => $agenda->idagenda,
                        'ideventotipo' => $value
                    ]);
                }
            }

            DB::commit();
            return redirect()
                ->route('admin.agendas.edit', $agenda)
                ->with('success', 'Agenda criada com sucesso!');
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->withInput()
                ->with('error', $e->getMessage());
        }
    }

    public function edit(Agenda $agenda, Pessoa $pessoaModel, Empresa $empresaModel, EventoTipo $eventoTipoModel)
    {
        $agendaEventosTipos = $agenda->agendaEventosTipos()->with('eventoTipo.prodserv')->get();

        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

        $pessoas = $pessoaModel
            ->where('temagenda', true)
            ->orderBy('nome')
            ->funcionarioOuCliente()
            ->get();

        $eventoTipos = $eventoTipoModel
                ->with('prodserv')
                ->orderBy('eventotipo')
                ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
        ];

        if ($agenda->status == 'ATIVO'){
            $bg_status = "bg-success";
        }elseif($agenda->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        return view('admin.agendas.edit', compact('agenda', 'empresas', 'statusOptions', 'pessoas', 'bg_status', 'eventoTipos', 'agendaEventosTipos'));
    }

    public function update(Agenda $agenda, Request $request, AgendaEventoTipo $agendaEventoTipoModel)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $agendaRequest = $request->all();

        if ($redirect == true) {
            // Tratamento para toggle coletiva
            $agendaRequest['coletiva'] = isset($agendaRequest['coletiva']) && $agendaRequest['coletiva'] == 'on';

            // Tratamento para toggle online
            $agendaRequest['online'] = isset($agendaRequest['online']) && $agendaRequest['online'] == 'on';

            // Definir os dias da semana
            $days = ['dom', 'seg', 'ter', 'qua', 'qui', 'sex', 'sab'];

            foreach ($days as $day) {
                if (!isset($agendaRequest['check-' . $day]) || !$agendaRequest['check-' . $day]) {
                    // Se o checkbox não está marcado, defina os campos de horário como null
                    $agendaRequest[$day . '_horainicio1'] = null;
                    $agendaRequest[$day . '_horafim1'] = null;
                    $agendaRequest[$day . '_horainicio2'] = null;
                    $agendaRequest[$day . '_horafim2'] = null;
                }
            }
        }


        DB::beginTransaction();
        try {
            $agenda->update($agendaRequest);

            if ($redirect == false) {
                DB::commit();
                return response()->json($agenda->status);
            }

            if (isset($agendaRequest['tiposevento'])) {
                $agendaEventoTipoModel->where('idagenda', $agenda->idagenda)->delete();
                $eventoTipos = $agendaRequest['tiposevento'];
                // Vincula eventos tipos a agenda
                foreach ($eventoTipos as $key => $value) {
                    $agendaEventoTipoModel->create([
                        'idagenda' => $agenda->idagenda,
                        'ideventotipo' => $value
                    ]);
                }
            }

            DB::commit();
            return redirect()
                ->route('admin.agendas.edit', $agenda)
                ->with('success', 'Agenda atualizada com sucesso!');
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->withInput()
                ->with('error', $e->getMessage());
        }
    }

    public function changeStatus($id, $status)
    {
        // Chamando a função updateStatus da trait, passando o modelo e o request
        return $this->updateStatus(Agenda::class, $id, $status);
    }

    public function destroy(Agenda $agenda)
    {
        DB::beginTransaction();
        try {
            $agenda->delete();
            DB::commit();
            return redirect()
                ->route('admin.agendas.index')
                ->with('success', 'Agenda removida!');
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->with('error', $e->getMessage())
                ->withInput();
        }
    }

    public function search(Request $request)
    {
        $agendas =  Agenda::with('empresa')->where('agencia', 'like', '%'.$request->search.'%')
                        ->orderBy('nome')
                        ->paginate(15);

        $size = $agendas->count();

        // Colocando mensagens na session
        session()->forget('error');
        session()->forget('success');
        if ($size && $size > 0 )
        {
            session()->flash('success', $size.' valor(es) encontrado(s).');

        } else
        {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        return view('admin.agendas.index')
                ->with(compact('agendas'));
    }
}
