<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use Illuminate\Http\Request;
use App\Models\Contrato as Model;
use App\Models\Pessoa;
use App\Models\TipoProdutoServico;
use App\Models\Veiculo;
use App\Models\ContratoVeiculo;
use App\Models\ContratoItem;
use App\Models\ProdutoServico;
use Exception;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class ContratoController extends Controller
{
    public function index(Request $request)
    {
        $module = $request->segment(2); // 'vendas' ou 'pedidos'
        $key = $module . '_dateRange'; // chave será 'vendas_dateRange' ou 'pedidos_dateRange'

        if ($request->has('clear') || session($key) === "") {
            session()->forget($key);
        }

        if (session($key)) {
            // Redirecionar para search com o date_range da session específica do módulo
            return redirect()->route('admin.contratos.search', ['date_range' => session($key)]);
        }

        // Aplicando eager loading para evitar múltiplas consultas desnecessárias
        $contratos = Model::with(['empresa', 'pessoa', 'prodservtipo'])
            ->where('status', '!=', 'INATIVO')
            ->orderBy('datafim', 'desc')
            ->paginate(100);

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
            // Adicione mais status conforme você desejar
        ];

        return view('admin.contratos.index')->with(compact('contratos', 'statusOptions'));
    }


    public function create(Pessoa $pessoaModel, Empresa $empresaModel, TipoProdutoServico $tipoProdutoServicoModel, )
    {
        $pessoas = $pessoaModel
                    ->orderBy('nome')
                    ->get();

        $empresas = $empresaModel
                    ->orderBy('razaosocial')
                    ->get();

        $tiposProdutoServico = $tipoProdutoServicoModel
                    ->orderBy('prodservtipo')
                    ->get();

        $statusOptions = [
                    'ATIVO' => 'Ativo',
                    'INATIVO' => 'Inativo',
                    'CONCLUÍDO' => 'Concluído',
                        // Adicione mais status conforme necessário
                    ];

        return view('admin.contratos.create')->with(compact('pessoas', 'empresas','tiposProdutoServico', 'statusOptions'));
    }

    public function store(Request $request)
    {
        $contratoRequest = $request->validate([
            'idempresa' => 'required|string|exists:empresa,idempresa',
            'idpessoa' => 'required|string|exists:pessoa,idpessoa',
            'tipo' => 'required|string|in:D,C',
            'idprodservtipo' => 'required|string|exists:prodservtipo,idprodservtipo',
            'recorrencia' => 'required|string|in:mensal,trimestral,semestral,anual',
            'diavencimento' => 'required|string|min:1|max:31',
            'antecedencia' => 'nullable|string',
            'status' => 'required|string|in:ATIVO,INATIVO,CONCLUÍDO'
        ]);

        if ($request->datainiciorecorrencia > $request->datafimrecorrencia) {
            return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', 'Data de início da cobrança não pode ser maior que a data de fim. Verifique os dados e tente novamente.');
        }

        // verifica se as validações falharam
        if ($contratoRequest === false) {
            return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', 'Erro ao cadastrar o Contrato. Verifique os dados e tente novamente.');
        }

        $camposCheck = ['mostracontrato', 'mostraendereco'];
        // Trata os campos checked (on) para boolean
        foreach ($camposCheck as $campo) {
            if (isset($request[$campo]) && $request[$campo] === 'on') {
                $request[$campo] = true;
            } else {
                $request[$campo] = false;
            }
        }

        try {

            $contrato = Model::create($request->all());

            return redirect()->to('admin/contratos/'.$contrato->idcontrato.'/edit');

        } catch(Exception $e) {
            return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', $e->getMessage());
        }
    }

    public function edit($idContrato, Empresa $empresaModel, Pessoa $pessoaModel, TipoProdutoServico $tipoProdutoServicoModel, Veiculo $veiculoModel, ContratoVeiculo $contratoVeiculoModel, ContratoItem $contratoItemModel, ProdutoServico $prodservModel)
    {
        $contrato = Model::find($idContrato);
        $pessoas = $pessoaModel
                    ->orderBy('nome')
                    ->get();
        $empresas = $empresaModel
                    ->orderBy('razaosocial')
                    ->get();
        $tiposProdutoServico = $tipoProdutoServicoModel
                    ->orderBy('prodservtipo')
                    ->get();

        $contratosveiculos = $contratoVeiculoModel
                    ->where('idcontrato', $idContrato) // Filtra pelos registros com o idnf relacionado
                    ->get();

        $veiculos = $veiculoModel
                    ->where('idempresa', $contrato->idempresa) // Filtra pelos registros com o idempresa correspondente
                    ->orderBy('marca', 'asc')
                    ->orderBy('modelo', 'asc')
                    ->orderBy('placa', 'asc')
                    ->get();

        $contratoItem = $contratoItemModel
                    ->where('idcontrato', $idContrato) // Filtra pelos registros com o idnf relacionado
                    ->orderBy('idcontratoitem')
                    ->get();

        $produtoServico = $prodservModel->whereHas('tipos', function ($query) use ($contrato) {
                        // Filtra pelos registros na tabela intermediária com o idprodservtipo correspondente
                        $query->where('prodserv_prodservtipo.idprodservtipo', $contrato->idprodservtipo);
                    })
                    ->orderBy('prodserv')
                    ->get();

        if ($contrato->status == 'CONCLUÍDO'){
            $bg_status = "bg-success";
        }elseif($contrato->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
            'CONCLUÍDO' => 'Concluído',
                // Adicione mais status conforme necessário
        ];

        return view('admin.contratos.edit')->with(compact('contrato', 'empresas', 'pessoas', 'bg_status','tiposProdutoServico','contratosveiculos','veiculos', 'contratoItem', 'statusOptions','produtoServico'));
    }

    public function update($idContrato, Request $request)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $contrato = Model::find($idContrato);
        $contratoUpdate = $request->all();

        if ($redirect == true) {
            $camposMonetarios = ['descontocontrato'];

            // Trata os campos monetários
            foreach ($camposMonetarios as $campo) {
                if (isset($contratoUpdate[$campo])) {
                    $contratoUpdate[$campo] = str_replace(',', '.', preg_replace('/[^\d,\.]/', '', $contratoUpdate[$campo]));
                    $contratoUpdate[$campo] = floatval($contratoUpdate[$campo]);
                }
            }

            $camposCheck = ['mostracontrato', 'mostraendereco'];
            // Trata os campos checked (on) para boolean
            foreach ($camposCheck as $campo) {
                if (isset($contratoUpdate[$campo]) && $contratoUpdate[$campo] === 'on') {
                    $contratoUpdate[$campo] = true;
                } else {
                    $contratoUpdate[$campo] = false;
                }
            }

            if ($request->exists('idcontratoveiculo')){
                $input = $request->all();
                $condition = $input['idcontratoveiculo'];
                if (isset($condition)){
                    foreach ($condition as $key => $condition) {
                        $contratoveiculo = ContratoVeiculo::find($input['idcontratoveiculo'][$key]);
                        $contratoveiculo->idempresa = $contrato->idempresa;
                        $contratoveiculo->idveiculo = $input['idveiculo'][$key];
                        $contratoveiculo->save();
                    }
                }
            }else{
               //echo 'nok';
            }

            if ($request->exists('idcontratoitem')){
                $input = $request->all();
                $condition = $input['idcontratoitem'];
                $camposMonetarios = ['qtd','valorun','valoritem' /* outros campos */];
                if (isset($condition)){
                    foreach ($condition as $key => $condition) {
                        $item = ContratoItem::find($input['idcontratoitem'][$key]);

                        if (isset($item)){

                            // Trata os campos monetários
                            foreach ($camposMonetarios as $campo) {
                                if (isset($input[$campo][$key])) {
                                    $input[$campo][$key] = str_replace(',', '.', preg_replace('/[^\d,\.]/', '', $input[$campo][$key]));
                                    $input[$campo][$key] = floatval($input[$campo][$key]);
                                }
                            }
                            if (isset($input['idprodserv'][$key]))
                                $item->idprodserv = $input['idprodserv'][$key];
                            $item->qtd = $input['qtd'][$key];
                            $item->idempresa = $contrato->idempresa;
                            $item->un = $input['un'][$key];
                            $item->valorun = $input['valorun'][$key];
                            $item->valoritem = $input['valoritem'][$key];
                            $item->save();

                        }else{
                            $item = new ContratoItem();
                            // Trata os campos monetários
                            foreach ($camposMonetarios as $campo) {
                                if (isset($input[$campo][$key])) {
                                    $input[$campo][$key] = str_replace(',', '.', preg_replace('/[^\d,\.]/', '', $input[$campo][$key]));
                                    $input[$campo][$key] = floatval($input[$campo][$key]);
                                }
                            }

                            if (isset($input['idprodserv'][$key])) {
                                $item->idprodserv = $input['idprodserv'][$key];
                                $item->idnf = $idContrato;
                                $item->qtd = $input['qtd'][$key];
                                $item->idempresa = $contrato->idempresa;
                                $item->un = $input['un'][$key];
                                $item->valorun = $input['valorun'][$key];
                                $item->valoritem = $input['valoritem'][$key];
                                $item->save();
                            }
                        }
                    }
                }
            }
        }

        DB::beginTransaction();
        try {
            $contrato->update($contratoUpdate);

            if ($redirect == false) {
                DB::commit();
                return response()->json($contrato->status);
            }

            DB::commit();
            return back()->with('success', 'Contrato atualizado!');
        } catch(Exception $e) {
            DB::rollBack();
            return redirect()
                    ->route('admin.contratos.index')
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function destroy($idContrato)
    {
        $contrato = Model::find($idContrato);

        try
        {
            $contrato->delete();

            return redirect()
                    ->route('admin.contratos.index')
                    ->with('success', 'Contrato removida!');
        } catch(Exception $e)
        {
            return redirect()
                    ->back()
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function search(Request $request)
    {
        $intervaloDatas = explode(' - ', $request->date_range);

        // Inicializa as variáveis $dataInicio e $dataFim como null ou strings vazias
        $dataInicio = '';
        $dataFim = '';

        // Verifica se existem pelo menos dois elementos no array
        if (count($intervaloDatas) >= 2) {
            $dataInicio = Carbon::createFromFormat('d/m/y', $intervaloDatas[0])->format('Y-m-d');
            $dataFim = Carbon::createFromFormat('d/m/y', $intervaloDatas[1])->format('Y-m-d');
        }

        // Aplicando eager loading para evitar N+1 queries
        $contratos = Model::with(['empresa', 'pessoa', 'prodservtipo'])
            ->when($request->date_range, function ($query) use ($dataInicio, $dataFim) {
                $query->whereBetween('datafim', [$dataInicio, $dataFim]);
            })
            ->where(function ($query) use ($request) {
                $query->where('nome', 'like', '%' . $request->search . '%') // Busca pelo nome do contrato
                ->orWhereHas('prodservtipo', function ($q) use ($request) {
                    $q->where('prodservtipo', 'like', '%' . $request->search . '%');
                })
                ->orWhereHas('empresa', function ($q) use ($request) {
                    $q->where('nomefantasia', 'like', '%' . $request->search . '%')
                    ->orWhere('razaosocial', 'like', '%' . $request->search . '%');
                })
                ->orWhereHas('pessoa', function ($q) use ($request) {
                    $q->where('nome', 'like', '%' . $request->search . '%')
                    ->orWhere('razaosocial', 'like', '%' . $request->search . '%');
                });
            })
            ->where('status', 'like', '%' . $request->status . '%')
            ->orderBy('contrato.datafim', 'desc')
            ->paginate(100);

        // Contagem de resultados encontrados
        $size = $contratos->count();

        // Colocando mensagens na session
        session()->forget('error');
        session()->forget('success');
        if ($size && $size > 0) {
            session()->flash('success', $size . ' valores encontrado(s).');
        } else {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        // Salvar Sessão DateRange e Busca
        $module = $request->segment(2); // 'vendas' ou 'pedidos'
        $key = $module . '_dateRange'; // chave será 'vendas_dateRange' ou 'pedidos_dateRange'

        if (session($key)) {
            $dateRange = session($key);
        } else {
            $dateRange = $request->date_range ? urldecode($request->date_range) : '';
        }

        // Armazenar dateRange na session com chave específica do módulo
        session([$key => $dateRange]);

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.contratos.index')->with(compact('contratos', 'dateRange', 'statusOptions'));
    }


    public function removeveiculo($idContratoVeiculo){
        ContratoVeiculo::find($idContratoVeiculo)->delete($idContratoVeiculo);
        return response()->json([
        'success' => 'Record deleted successfully!'
        ]);
    }

    public function addveiculo($idContrato){
        $contratoVeiculo = new ContratoVeiculo();
        $contrato = Model::find($idContrato);
        $contratoVeiculo->idcontrato = $idContrato;
        $contratoVeiculo->idempresa = $contrato->idempresa;
        $contratoVeiculo->save();
        return response()->json([
        'success' => $contratoVeiculo->idcontratoveiculo
        ]);
    }


    public function removeitem($idContrato){
        ContratoItem::find($idContrato)->delete($idContrato);
        return response()->json([
        'success' => 'Record deleted successfully!'
        ]);
    }

    public function additem($idContrato){
        $contratoItem = new ContratoItem();
        $contrato = Model::find($idContrato);
        $contratoItem->idcontrato = $idContrato;
        $contratoItem->idempresa = $contrato->idempresa;
        $contratoItem->save();
        return response()->json([
        'success' => $contratoItem->idcontratoitem
        ]);
    }


}
