<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Agencia;
use Illuminate\Http\Request;
use App\Models\NfParcela as Model;
use App\Models\Nf;
use App\Models\Empresa;
use App\Models\Pessoa;
use App\Models\TipoProdutoServico;
use App\Models\Veiculo;
use App\Models\FormaPagamento;
use App\Services\NfParcelaService;

use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\DB;

class NfParcelaController extends Controller
{


    protected $nfParcelaService;

    public function __construct(NfParcelaService $nfParcelaService)
    {
        $this->nfParcelaService = $nfParcelaService;
    }

    public function index(Request $request)
    {

        $module = $request->segment(2); // 'vendas' ou 'pedidos'
        $key = $module . '_dateRange'; // chave será 'vendas_dateRange' ou 'pedidos_dateRange'

        if ($request->has('clear') || session($key) === "") {
            session()->forget($key);
        }

        if (session($key)) {
            // Redirecionar para search com o date_range da session específica do módulo
            return redirect()->route('admin.nfparcelas.search', ['date_range' => session($key)]);
        }

        $parcelas = Model::NaoPedido()->where('status', '!=', 'INATIVO')->orderBy('datavencimento','desc')
            ->paginate(1000);

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
            // Adicione mais status conforme você desejar
        ];

        return view('admin.nfparcelas.index')->with(compact('parcelas', 'statusOptions'));
    }

    public function create()
    {

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];
        return view('admin.nfparcelas.create');
    }

    public function store(Request $request)
    {
        $parcelaCreate = $request->all();

        try {
            Model::create($parcelaCreate);

            return redirect()
                ->route('admin.nfparcelas.index')
                ->with('success', 'Parcela criada com sucesso!');
        } catch (Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->with('error', $e->getMessage());
        }
    }

    public function edit($idNfParcela, Nf $NfModel, Empresa $empresaModel, Pessoa $pessoaModel, FormaPagamento $formaPagamentoModel, TipoProdutoServico $tipoProdutoServicoModel, Veiculo $veiculoModel, Agencia $agenciaModel)
    {
        $parcela = Model::find($idNfParcela);

        $nf = $NfModel
        ->where('idnf', $parcela->idnf) // Filtra pelos registros com o idnf relacionado
        ->first();

        $empresa = $empresaModel
        ->where('idempresa', $nf->idempresa)
        ->first();

        $pessoa = $pessoaModel
        ->where('idpessoa', $nf->idpessoa)
        ->first();

        $veiculo = $veiculoModel
        ->where('idveiculo', $nf->idveiculo)
        ->first();

        $tipoProdutoServico = $tipoProdutoServicoModel
        ->where('idprodservtipo', $nf->idprodservtipo)
        ->first();

        $formaPagamento = $formaPagamentoModel
        ->where('idformapagamento', $nf->idformapagamento)
        ->first();

        if ($parcela->status == 'CONCLUÍDO'){
            $bg_status = "bg-success";
        }elseif($parcela->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        $agencias = $agenciaModel
        ->orderBy('nome')
        ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.nfparcelas.edit')->with(compact('parcela', 'nf', 'empresa', 'pessoa', 'veiculo','tipoProdutoServico','formaPagamento','bg_status', 'statusOptions', 'agencias'));
    }

    public function update($idNfParcela, Request $request)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $parcela = Model::find($idNfParcela);

        if ($redirect == true) {
            $camposMonetarios = ['valor'];

            $nfParcelaUpdate = $request->all();
             // Trata os campos monetários
             foreach ($camposMonetarios as $campo) {
                if (isset($nfParcelaUpdate[$campo])) {
                    $nfParcelaUpdate[$campo] = str_replace('.', '', $nfParcelaUpdate[$campo]);
                    $nfParcelaUpdate[$campo] = str_replace(',', '.', $nfParcelaUpdate[$campo]);
                    $nfParcelaUpdate[$campo] = floatval($nfParcelaUpdate[$campo]);
                }
            }
        }

        DB::beginTransaction();
        try {
            $novoStatus = $nfParcelaUpdate['status'] ?? $parcela->status;
            $statusAtual = $parcela->status;
            $parcela->update($nfParcelaUpdate);

            if (isset($parcela->idagencia)) {
                $isCredito = $parcela->tipo === "C";
                $isDebito = $parcela->tipo === "D";

                // Verifica se houve uma transição de status para 'CONCLUÍDO'
                if ($statusAtual !== 'CONCLUÍDO' && $novoStatus === 'CONCLUÍDO') {
                    $this->nfParcelaService->concluirParcela($parcela, $isDebito, $isCredito, $parcela->idagencia);
                }

                // Verifica se houve uma transição de status saindo de 'CONCLUÍDO'
                if ($statusAtual === 'CONCLUÍDO' && $novoStatus !== 'CONCLUÍDO') {
                    $this->nfParcelaService->estornarParcela($parcela, $isDebito, $isCredito, $parcela->idagencia);
                }
            }

            if ($redirect == true) {
                DB::commit();
                return response()->json($parcela->status);
            }

            DB::commit();
            return redirect()->back()->with('success', 'Parcela atualizada!');

        } catch (Exception $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->with('error', $e->getMessage())
                ->withInput();
        }
    }

    public function destroy($idNfParcela)
    {
        $parcela = Model::find($idNfParcela);

        try {
            $parcela->delete();

            return redirect()
                ->route('admin.nfparcelas.index')
                ->with('success', 'Parcela removida!');
        } catch (Exception $e) {
            return redirect()
                ->back()
                ->with('error', $e->getMessage())
                ->withInput();
        }
    }

    public function search(Request $request)
    {
        $intervaloDatas = explode(' - ', $request->date_range);
       // dump($intervaloDatas);

        // Inicializa as variáveis $dataInicio e $dataFim como null ou strings vazias
        $dataInicio = '';
        $dataFim = '';

        // Verifica se existem pelo menos dois elementos no array
        if (count($intervaloDatas) >= 2) {
            $dataInicio = $intervaloDatas[0];
            $dataFim = $intervaloDatas[1];

            $dataInicio = Carbon::createFromFormat('d/m/y', $intervaloDatas[0])->format('Y-m-d');
            $dataFim = Carbon::createFromFormat('d/m/y', $intervaloDatas[1])->format('Y-m-d');
        }

        $query = Model::query()
        ->select('nfparcela.*', 'empresa.status as empresa_status')
        ->join('empresa', 'empresa.idempresa', '=', 'nfparcela.idempresa')
        ->when($request->date_range, function ($query) use ($dataInicio, $dataFim) {
            $query->whereBetween('datavencimento', [$dataInicio, $dataFim]);
        })
        ->where(function ($query) use ($request) {
            $query->where('idnf', 'like', '%' . $request->search . '%')
            ->orwhereHas('nf.prodservtipo', function ($q) use ($request) {
                $q->where('prodservtipo', 'like', '%' . $request->search . '%');
            })
            ->orwhereHas('empresa', function ($q) use ($request) {
                $q->where('nomefantasia', 'like', '%' . $request->search . '%');
            })
            ->orWhereHas('empresa', function ($q) use ($request) {
                $q->where('razaosocial', 'like', '%' . $request->search . '%');
            })
            ->orWhereHas('nf.pessoa', function ($q) use ($request) { // Note o 'nf.pessoa' aqui
                $q->where('nome', 'like', '%' . $request->search . '%');
            })
            ->orWhereHas('nf.pessoa', function ($q) use ($request) { // Novamente, note o 'nf.pessoa'
                $q->where('razaosocial', 'like', '%' . $request->search . '%');
            });

        })
        ->where('nfparcela.status', $request->status)
        ->orderBy('datavencimento','desc');


        // Exibir a consulta SQL
        $sql = $query->toSql();
        $bindings = $query->getBindings();
        foreach ($bindings as $binding) {
            $value = is_numeric($binding) ? $binding : "'".$binding."'";
            $sql = preg_replace('/\?/', $value, $sql, 1);
        }

       // dump($sql);

        // Aplicar a paginação
        $parcelas = $query->paginate(100);


        $size = $parcelas->count();

        // Colocando mensagens na session
        session()->forget('error');
        session()->forget('success');
        if ($size && $size > 0 )
        {
            session()->flash('success', $size.' valores encontrado(s).');

        } else
        {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        //Salvar Sessão DateRange e Busca
        $module = $request->segment(2); // 'vendas' ou 'pedidos'

        $key = $module . '_dateRange'; // chave será 'vendas_dateRange' ou 'pedidos_dateRange'

        if (session($key)) {
            $dateRange = session($key);
        }else{
            $dateRange = $request->date_range ? urldecode($request->date_range) : '';
        }

        // Armazenar dateRange na session com chave específica do módulo
        session([$key => $dateRange]);

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.nfparcelas.index')->with(compact('parcelas','dateRange', 'statusOptions'));
    }
}
