<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Agencia;
use Illuminate\Http\Request;
use App\Models\NfParcela as Model;
use App\Models\Nf;
use App\Models\Empresa;
use App\Models\Pessoa;
use App\Models\TipoProdutoServico;
use App\Models\Veiculo;
use App\Models\FormaPagamento;
use App\Services\NfParcelaService;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\DB;

class NfParcelaDebitoController extends Controller
{
    public function __construct(
        protected NfParcelaService $nfParcelaService
    ) {}

    public function index(Request $request, Empresa $empresaModel, Pessoa $pessoaModel, Nf $nfModel, TipoProdutoServico $tipoProdutoServicoModel)
    {
        $parcelas = Model::query()
            ->select('nfparcela.*') // Renomeando o status de nfparcela
            ->join('empresa', 'empresa.idempresa', '=', 'nfparcela.idempresa') // Join com a tabela empresa
            ->with(['empresa', 'nf.fornecedor', 'nf.prodservtipo']) // Eager loading das relações necessárias
            ->NaoPedido()
            ->debito()
            ->where('nfparcela.status', '!=', 'INATIVO')
            ->orderBy('datavencimento', 'desc')
            ->paginate(100);

        // Calcular os valores totais
        $totalValue = Model::NaoPedido()
            ->debito()
            ->whereIn('status', ['ativo', 'concluido']) // Mantemos status original para cálculo
            ->sum('valor');

        $totalValueDebitoAtivo = Model::NaoPedido()
            ->debito()
            ->where('status', 'ativo') // Mantemos status original para cálculo
            ->sum('valor');

        $totalValueDebitoConcluido = Model::NaoPedido()
            ->debito()
            ->where('status', 'concluido') // Mantemos status original para cálculo
            ->sum('valor');

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
        ];

        $empresas = $empresaModel
            ->orderBy('nomefantasia', 'asc')
            ->get();

        $pessoas = $pessoaModel
            ->Fornecedor()
            ->get();

        $nfs = $nfModel
            ->Venda()
            ->get();

        $tiposProdutoServico = $tipoProdutoServicoModel
            ->orderBy('prodservtipo')
            ->get();

        return view('admin.nfparcelasdebito.index',
            compact(
                'parcelas',
                'statusOptions',
                'totalValue',
                'totalValueDebitoAtivo',
                'totalValueDebitoConcluido',
                'empresas',
                'pessoas',
                'nfs',
                'tiposProdutoServico'
            ));
    }

    public function create()
    {

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];
        return view('admin.nfparcelasdebito.create');
    }

    public function store(Request $request)
    {
        $nfParcelaCreate = $request->all();

        try {
            Model::create($nfParcelaCreate);

            return redirect()
                ->route('admin.nfparcelasdebito.index')
                ->with('success', 'Parcela criada com sucesso!');
        } catch (Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->with('error', $e->getMessage());
        }
    }

    public function edit($idNfParcela, Nf $NfModel, Empresa $empresaModel, Pessoa $pessoaModel, FormaPagamento $formaPagamentoModel, TipoProdutoServico $tipoProdutoServicoModel, Veiculo $veiculoModel, Agencia $agenciaModel)
    {
        $parcela = Model::find($idNfParcela);

        $nf = $NfModel
        ->where('idnf', $parcela->idnf) // Filtra pelos registros com o idnf relacionado
        ->first();

        $empresa = $empresaModel
        ->where('idempresa', $nf->idempresa)
        ->first();

        $pessoa = $pessoaModel
        ->where('idpessoa', $nf->idpessoa)
        ->first();

        $veiculo = $veiculoModel
        ->where('idveiculo', $nf->idveiculo)
        ->first();

        $tipoProdutoServico = $tipoProdutoServicoModel
        ->where('idprodservtipo', $nf->idprodservtipo)
        ->first();

        $formaPagamento = $formaPagamentoModel
        ->where('idformapagamento', $nf->idformapagamento)
        ->first();

        $agencias = $agenciaModel
        ->orderBy('nome')
        ->get();

        if ($parcela->status == 'CONCLUÍDO'){
            $bg_status = "bg-success";
        }elseif($parcela->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.nfparcelasdebito.edit')->with(compact('parcela', 'nf', 'empresa', 'pessoa', 'veiculo','tipoProdutoServico','formaPagamento','bg_status', 'statusOptions', 'agencias'));
    }

    public function update($idNfParcela, Request $request)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $parcela = Model::find($idNfParcela);
        $nfParcelaUpdate = $request->all();

        if ($redirect == true) {
            $camposMonetarios = ['valor'];

            // Trata os campos monetários
            foreach ($camposMonetarios as $campo) {
                if (isset($nfParcelaUpdate[$campo])) {
                    $nfParcelaUpdate[$campo] = str_replace('.', '', $nfParcelaUpdate[$campo]);
                    $nfParcelaUpdate[$campo] = str_replace(',', '.', $nfParcelaUpdate[$campo]);
                    $nfParcelaUpdate[$campo] = floatval($nfParcelaUpdate[$campo]);
                }
            }
        }

        DB::beginTransaction();
        try {
            $novoStatus = $nfParcelaUpdate['status'] ?? $parcela->status;
            $statusAtual = $parcela->status;

            $parcela->update(
                $redirect == false && $novoStatus === 'CONCLUÍDO' && !$parcela->datatransacao
                    ? array_merge($nfParcelaUpdate, ['datatransacao' => now()])
                    : $nfParcelaUpdate
            );

            if (isset($parcela->idagencia)) {
                if ($statusAtual !== 'CONCLUÍDO' && $novoStatus === 'CONCLUÍDO') {
                    $this->nfParcelaService->concluirParcela($parcela, true, false, $parcela->idagencia);
                }

                if ($statusAtual === 'CONCLUÍDO' && $novoStatus === 'ATIVO' || $statusAtual === 'CONCLUÍDO' && $novoStatus === 'INATIVO') {
                    $this->nfParcelaService->estornarParcela($parcela, true, false, $parcela->idagencia);
                }
            }

            if ($redirect == false) {
                DB::commit();
                return response()->json($parcela->status);
            }

            DB::commit();
            return back()->with('success', 'Parcela atualizada!');

        } catch (Exception $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->with('error', $e->getMessage())
                ->withInput();
        }
    }

    public function destroy($idNfParcela)
    {
        $parcela = Model::find($idNfParcela);

        try {

            $parcela->delete();

            return redirect()
                ->route('admin.nfparcelasdebito.index')
                ->with('success', 'Parcela removida!');
        } catch (Exception $e) {
            return redirect()
                ->back()
                ->with('error', $e->getMessage())
                ->withInput();
        }
    }

    public function search(
        Request $request,
        Pessoa $pessoaModel,
        Nf $nfModel,
        TipoProdutoServico $tipoProdutoServicoModel,
        Empresa $empresaModel
    ) {
        // Cria a query inicial com joins e filtragem básica
        $debitos = Model::select('nfparcela.*', 'nf.idpessoa', 'nf.idprodservtipo')
            ->join('nf', 'nfparcela.idnf', '=', 'nf.idnf')
            ->where('nfparcela.tipo', 'D')
            ->with(['nf.pessoa', 'nf.prodservtipo', 'empresa']) // Usa `nf.pessoa` e `nf.prodservtipo`
            ->when($request->status, function ($query) use ($request) {
                return $query->where('nfparcela.status', $request->status);
            });

        // Aplica filtros de busca por termo se o campo `search` estiver preenchido
        if ($request->filled('search')) {
            $searchTerm = '%' . $request->search . '%';
            $debitos->where(function ($query) use ($searchTerm) {
                $query->whereHas('nf.pessoa', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('nome', 'like', $searchTerm);
                })->orWhereHas('nf.prodservtipo', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('prodservtipo', 'like', $searchTerm);
                })->orWhereHas('empresa', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('sigla', 'like', $searchTerm);
                });
            });
        }

        // Aplicação dinâmica de filtros com base nos campos
        filter($request, $debitos);

        // Calcula os valores totais para status "ativo" e "concluído"
        $totalValue = $debitos->sum('valor');
        $totalValueDebitoAtivo = $debitos->clone()->where('nfparcela.status', 'ATIVO')->sum('valor');
        $totalValueDebitoConcluido = $debitos->clone()->where('nfparcela.status', 'CONCLUÍDO')->sum('valor');

        // Obter os resultados paginados
        $parcelas = $debitos->paginate(50);

        // Verificação do tamanho do resultado e notificação na sessão
        session()->forget('error');
        session()->forget('success');
        if ($parcelas->count() > 0) {
            session()->flash('success', $parcelas->count() . ' valores encontrado(s).');
        } else {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        // Coleta as listas de opções necessárias para filtros no modal
        $tiposProdutoServico = $tipoProdutoServicoModel->orderBy('prodservtipo')->get();
        $empresas = $empresaModel->orderBy('nomefantasia', 'asc')->get();
        $pessoas = $pessoaModel->Fornecedor()->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.nfparcelasdebito.index', compact(
            'parcelas',
            'statusOptions',
            'totalValue',
            'totalValueDebitoAtivo',
            'totalValueDebitoConcluido',
            'tiposProdutoServico',
            'empresas',
            'pessoas',
        ));
    }
}
