<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use App\Models\FormaPagamento;
use Illuminate\Http\Request;
use App\Models\Nf as Model;
use App\Models\NfParcela;
use App\Models\NfItem;
use App\Models\ProdutoServico;
use App\Models\Pessoa;
use App\Models\TipoProdutoServico;
use App\Models\Veiculo;
use App\Models\Nf;
use App\Services\EstoqueService;
use Exception;
use Illuminate\Support\Facades\DB;
use DateTime;
use Carbon\Carbon;

class PedidoController extends Controller
{
    private $estoqueService;

    public function __construct(EstoqueService $estoqueService)
    {
        $this->estoqueService = $estoqueService;
    }

    public function index(Request $request, Veiculo $veiculoModel, TipoProdutoServico $tipoProdutoServicoModel, Empresa $empresaModel, Pessoa $pessoaModel)
    {
        // Usar eager loading para carregar as relações de pedidos
        $pedidos = Model::with(['empresa', 'fornecedor', 'veiculo', 'prodservtipo'])
            ->pedido()
            ->orderBy('dataentrada', 'desc')
            ->paginate(50);

        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

        $pessoas = $pessoaModel
            ->FuncionarioOuCliente()
            ->get();

        $veiculos = $veiculoModel->with('pessoa')
            ->ativo()
            ->orderBy('placa')
            ->get();

        $tiposProdutoServico = $tipoProdutoServicoModel
            ->active()
            ->orderBy('prodservtipo')
            ->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.pedidos.index',
            compact(
                'pedidos',
                'statusOptions',
                'veiculos',
                'tiposProdutoServico',
                'empresas',
                'pessoas'
            ));
    }


    public function create(Pessoa $pessoaModel, Empresa $empresaModel, FormaPagamento $formaPagamentoModel, TipoProdutoServico $tipoProdutoServicoModel, Veiculo $veiculoModel,  Nf $nfModel)
    {
        $pessoas = $pessoaModel
                    ->orderBy('nome')
                    ->funcionarioOuCliente()
                    ->get();

        $tiposProdutoServico = $tipoProdutoServicoModel
                    ->active()
                    ->orderBy('prodservtipo')
                    ->get();

        $empresas = $empresaModel
                    //->where('idtenant', auth()->user()->idtenant)
                    ->orderBy('razaosocial')
                    ->get();

        $formaPagamento = $formaPagamentoModel
                    ->orderBy('formapagamento')
                    ->get();

        $veiculos = $veiculoModel
        ->ativo()
        ->orderBy('placa')
        ->get();

        $veiculos = $veiculos->map(function ($veiculo) {
            $ultimoRegistro = DB::table('nf')
                ->join('nfitem', function($join) {
                    $join->on('nfitem.idnf', '=', 'nf.idnf')
                        ->whereIn('nfitem.idprodserv', [20, 31]);
                })
                ->where('nf.idveiculo', $veiculo->idveiculo)
                ->orderByDesc('nf.dataentrada')
                ->select('nf.*', 'nf.km')
                ->first();

            $veiculo->ultimo_km = $ultimoRegistro ? $ultimoRegistro->km : null;
            return $veiculo;
        });

    //    dd($veiculos);


        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        $compras = $nfModel::latest()
            ->compra()
            ->orderBy('datavencimento','desc')
            ->paginate(1000);

        return view('admin.pedidos.create', compact('pessoas', 'empresas', 'formaPagamento','tiposProdutoServico','veiculos', 'statusOptions','compras'));
    }

    public function store(Request $request, NfParcela $nfParcelaModel)
    {
        try
        {
            DB::beginTransaction();

            $pedidoCreate = $request->all();

            $pedido = Model::create($pedidoCreate);

            DB::commit();
            //return back()->with('success', 'pedido criada com sucesso!');
            return redirect()->to('admin/pedidos/'.$pedido->idnf.'/edit');

        } catch(Exception $e)
        {
            DB::rollBack();

            return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', $e->getMessage());
        }
    }

    public function edit($idNf, Pessoa $pessoaModel, Empresa $empresaModel, FormaPagamento $formaPagamentoModel,TipoProdutoServico $tipoProdutoServicoModel, NfItem $nfItemModel, ProdutoServico $prodservModel, Veiculo $veiculoModel,NfParcela $nfParcelaModel, Nf $nfModel)
    {
        $pedido = Model::find($idNf);
        $pessoas = $pessoaModel
                        ->orderBy('nome')
                        ->funcionarioOuCliente()
                        ->get();

        $veiculos = $veiculoModel
        ->orderBy('placa')
        ->get();

        $veiculos = $veiculos->map(function ($veiculo) {
            $ultimoRegistro = DB::table('nf')
                ->join('nfitem', function($join) {
                    $join->on('nfitem.idnf', '=', 'nf.idnf')
                        ->whereIn('nfitem.idprodserv', [20, 31]);
                })
                ->where('nf.idveiculo', $veiculo->idveiculo)
                ->orderByDesc('nf.dataentrada')
                ->select('nf.*', 'nf.km')
                ->first();

            $veiculo->ultimo_km = $ultimoRegistro ? $ultimoRegistro->km : null;
            return $veiculo;
        });




        $empresas = $empresaModel
                        ->orderBy('razaosocial')
                        ->get();

        $tiposProdutoServico = $tipoProdutoServicoModel
                        ->orderBy('prodservtipo')
                        ->get();

        $formaPagamento = $formaPagamentoModel
                        ->orderBy('formapagamento')
                        ->get();

        $nfItem = $nfItemModel
                        ->where('idnf', $idNf) // Filtra pelos registros com o idnf relacionado
                        ->orderBy('idprodserv')
                        ->get();

        $produtoServico = $prodservModel
                        ->whereHas('tipos', function($query) use ($pedido) {
                            $query->where('prodserv_prodservtipo.idprodservtipo', $pedido->idprodservtipo);
                        })
                        ->orderBy('prodserv')
                        ->get();


        $nfParcela = $nfParcelaModel
                        ->where('idnf', $idNf) // Filtra pelos registros com a nf
                        ->orderBy('datavencimento')
                        ->get();

        $compras = $nfModel::compra()
                        ->where('status', 'CONCLUÍDO')

                        ->where('idprodservtipo', $pedido->idprodservtipo)
                        ->orderBy('datavencimento','desc')
                        ->paginate(100);

        if ($pedido->status == 'CONCLUÍDO'){
            $bg_status = "bg-success";
        }elseif($pedido->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        // Obter a quantidade de estoque atual para cada produto
        $estoqueAtualPorProduto = [];
        foreach ($produtoServico as $produto) {
            $estoqueAtualPorProduto[$produto->idprodserv] = $this->estoqueService->calcularEstoqueAtual($produto->idprodserv);
        }

        return view('admin.pedidos.edit', [
            'pedido' => $pedido,
            'pessoas' => $pessoas,
            'empresas' => $empresas,
            'formaPagamento' => $formaPagamento,
            'tiposProdutoServico' => $tiposProdutoServico,
            'nfItem' => $nfItem,
            'produtoServico' => $produtoServico,
            'veiculos' => $veiculos,
            'estoqueAtualPorProduto' => $estoqueAtualPorProduto,
            'bg_status' => $bg_status,
            'nfParcela' => $nfParcela,
            'statusOptions' => $statusOptions,
            'compras' => $compras,
        ]);
    }

    public function update($idNf, Request $request)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $pedido = Model::find($idNf);
        $pedidoUpdate = $request->all();

        if ($redirect == true) {
            $camposMonetarios = ['descontonf'];
            // Trata os campos monetários
            foreach ($camposMonetarios as $campo) {
                if (isset($pedidoUpdate[$campo])) {
                    $pedidoUpdate[$campo] = str_replace(',', '.', preg_replace('/[^\d,\.]/', '', $pedidoUpdate[$campo]));
                    $pedidoUpdate[$campo] = floatval($pedidoUpdate[$campo]);
                }
            }
        }

        DB::beginTransaction();
        try {
            $pedido->update($pedidoUpdate);

            if ($request->exists('idnfitem')){
                $input = $request->all();
                $condition = $input['idnfitem'];
                $camposMonetarios = ['qtd','valorun','valoritem' /* outros campos */];
                if (isset($condition)){
                    foreach ($condition as $key => $condition) {
                        $item = NfItem::find($input['idnfitem'][$key]);

                        if (isset($item)) {
                            // Trata os campos monetários
                            foreach ($camposMonetarios as $campo) {
                                if (isset($input[$campo][$key])) {
                                    $input[$campo][$key] = str_replace(',', '.', preg_replace('/[^\d,\.]/', '', $input[$campo][$key]));
                                    $input[$campo][$key] = floatval($input[$campo][$key]);
                                }
                            }

                            if (isset($input['idprodserv'][$key])) {
                                $item->idprodserv = $input['idprodserv'][$key];
                                $item->qtd = $input['qtd'][$key];
                                $item->idempresa = $pedido->idempresa;
                                $item->un = $input['un'][$key];
                                $item->valorun = $input['valorun'][$key];
                                $item->valoritem = $input['valoritem'][$key];
                                $item->save();
                            }
                        } else {
                            $item = new NfItem();
                            // Trata os campos monetários
                            foreach ($camposMonetarios as $campo) {
                                if (isset($input[$campo][$key])) {
                                    $input[$campo][$key] = str_replace(',', '.', preg_replace('/[^\d,\.]/', '', $input[$campo][$key]));
                                    $input[$campo][$key] = floatval($input[$campo][$key]);
                                }
                            }

                            if (isset($input['idprodserv'][$key])) {
                                $item->idprodserv = $input['idprodserv'][$key];
                                $item->idnf = $idNf;
                                $item->qtd = $input['qtd'][$key];
                                $item->idempresa = $pedido->idempresa;
                                $item->un = $input['un'][$key];
                                $item->valorun = $input['valorun'][$key];
                                $item->valoritem = $input['valoritem'][$key];
                                $item->save();
                            }
                        }
                    }
                }
            }

            if ($pedido->status === 'CONCLUÍDO') {
                $this->geraNfParcelas($pedido);
            }

            if ($redirect == false) {
                DB::commit();
                return response()->json($pedido->status);
            }

            DB::commit();
            return back()->with('success', 'Pedido atualizado!');
        } catch(Exception $e) {
            DB::rollBack();
            return redirect()
                    ->route('admin.pedidos.index')
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function destroy($idNf)
    {
        $empresa = Model::find($idNf);

        try
        {
            $empresa->delete();

            return redirect()
                    ->route('admin.pedidos.index')
                    ->with('success', 'Empresa removida!');
        } catch(Exception $e)
        {
            return redirect()
                    ->back()
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function search(Request $request, Veiculo $veiculoModel, TipoProdutoServico $tipoProdutoServicoModel, Empresa $empresaModel, Pessoa $pessoaModel)
    {
        $pedidos = Model::select('nf.*')
            ->with(['pessoa', 'empresa', 'veiculo', 'prodservtipo'])
            ->pedido();

        // Aplicar busca textual (search) se presente
        if ($request->filled('search')) {
            $searchTerm = '%' . $request->search . '%';
            $pedidos->where(function ($query) use ($searchTerm) {
                $query->whereHas('pessoa', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('nome', 'like', $searchTerm);
                })->orWhereHas('veiculo', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('placa', 'like', $searchTerm);
                })->orWhereHas('prodservtipo', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('prodservtipo', 'like', $searchTerm);
                })->orWhereHas('empresa', function ($subQuery) use ($searchTerm) {
                    $subQuery->where('sigla', 'like', $searchTerm);
                });
            });
        }

        // Aplica filtros dinâmicos com base nos campos
        filter($request, $pedidos);

        // Paginação e ordenação
        $pedidos = $pedidos->orderBy('dataentrada', 'desc')->paginate(50);

        session()->forget(['error', 'success']);
        if ($pedidos->count() > 0) {
            session()->flash('success', $pedidos->count() . ' valores encontrado(s).');
        } else {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        $veiculos = $veiculoModel->ativo()->orderBy('placa')->get();
        $tiposProdutoServico = $tipoProdutoServicoModel->active()->orderBy('prodservtipo')->get();
        $empresas = $empresaModel->orderBy('razaosocial')->get();
        $pessoas = $pessoaModel->FuncionarioOuCliente()->get();

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'CONCLUÍDO' => 'Concluído',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.pedidos.index')->with(compact('pedidos', 'statusOptions', 'veiculos', 'tiposProdutoServico', 'empresas', 'pessoas'));
    }


    public function removeitem($idNf){
        NfItem::find($idNf)->delete($idNf);
        return response()->json([
        'success' => 'Record deleted successfully!'
        ]);
    }

    public function additem($idNf){
        $nfitem = new NfItem();
        $pedido = Model::find($idNf);
        $nfitem->idnf = $idNf;
        $nfitem->idempresa = $pedido->idempresa;
        $nfitem->save();
        return response()->json([
        'success' => $nfitem->idnfitem
        ]);
    }



    public function checkDiaUtil($data) {
        $diaSemana = $data->format('N'); // 1 (segunda-feira) a 7 (domingo)
        return ($diaSemana >= 1 && $diaSemana <= 5); // Verificar se é um dia útil (segunda a sexta-feira)
    }

    public function ajustarDiaUtil($data) {
        // Verificar se o dia é sábado (6) ou domingo (7)
        if ($data->format('N') == 6) {
            // Adicionar 2 dias para ir para a próxima segunda-feira
            $data->modify('+2 days');
        } else {
            // Adicionar 1 dia para ir para a próxima segunda-feira
            $data->modify('+1 day');
        }

        return $data;
    }

    public function geraNfParcelas($pedido)
    {
        // Verificar se existem registros relacionados na tabela nfparcela
        $count = DB::table('nfparcela')
            ->where('idnf', $pedido->id)
            ->count();

        if ($count <= 0 and $pedido['qtdparcela'] > 0) {
            $vparcela = $pedido['totalnf'] / $pedido['qtdparcela'];
            $parcelas = [];

            $dataVencimento = DateTime::createFromFormat('Y-m-d', $pedido['datavencimento']);
            $dataVencimento->setTime(0, 0, 0);
            $dataVencimentoparcela = clone $dataVencimento;

            for ($i = 0; $i < $pedido['qtdparcela']; $i++) {
                $dataVencimentoparcela = clone $dataVencimento;
                $dataVencimento->modify('+1 month');

                if (!$this->checkDiaUtil($dataVencimentoparcela)) {
                    $dataVencimentoparcela = $this->ajustarDiaUtil($dataVencimentoparcela);
                }

                $parcelas[$i] = [
                    'tipo' => $pedido->tipo,
                    'parcela' => $i + 1,
                    'idempresa' => (int) $pedido->idempresa,
                    'valor' => $vparcela,
                    'datavencimento' => $dataVencimentoparcela,
                    'status' => 'ATIVO'
                ];
            }

            $pedido->parcela()->createMany($parcelas);
        }
    }


}
