<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use Illuminate\Http\Request;
use App\Models\ProdutoServico as Model;
use App\Models\TipoProdutoServico;
use App\Services\EstoqueService;
use Exception;
use Illuminate\Support\Facades\DB;

class ProdutoServicoController extends Controller
{
    private $estoqueService;

    public function __construct(EstoqueService $estoqueService)
    {
        $this->estoqueService = $estoqueService;
    }
    protected $unidadesDeMedida = [
        'BOBINA' => 'BOBINA',
        'CX' => 'CAIXA',
        'EA' => 'EA',
        'EQUIP' => 'EQUIPAMENTO',
        'FL' => 'FOLHA',
        'FR' => 'FRASCO',
        'GL' => 'GALAO',
        'GR' => 'GRAMA',
        'KG' => 'KILOGRAMA',
        'LT' => 'LITRO',
        'MG' => 'MG',
        'mL' => 'ML',
        'MT' => 'METRO',
        'Pares' => 'PARES',
        'PC' => 'PEÇA',
        'PCT' => 'PACOTE',
        'ROLO' => 'ROLO',
        'SACO' => 'SACO',
        'TB' => 'TUBO',
        'UI' => 'UI',
        'UN' => 'UNIDADE',
        'DS' => 'DOSE',
        'POT10' => 'POTÊNCIA 10',
        'MIL' => 'MILHEIRO',
        'TESTES' => 'TESTES',
        'BLD' => 'BALDE',
        'FDO' => 'FARDO',
        'TA' => 'TAMBOR',
        'TO' => 'TONELADA',
        'CM' => 'CENTÍMETRO',
        'M2' => 'm²',
        'M3' => 'M³',
        'JOGO' => 'JG',
    ];

    public function index()
    {
        // Carregar os produtos e seus tipos com Eager Loading
        $produtosServicos = Model::with(['tipos', 'empresa']) // Carrega as relações 'tipos' e 'empresa'
            ->orderBy('prodserv')
            ->orderBy('tipo')
            ->paginate(100);

        // Obter a quantidade de estoque atual para cada produto
        $estoqueAtualPorProduto = [];
        foreach ($produtosServicos as $produtoServico) {
            $estoqueAtualPorProduto[$produtoServico->idprodserv] = $this->estoqueService->calcularEstoqueAtual($produtoServico->idprodserv);
        }

        return view('admin.produto-servico.index')->with(compact('produtosServicos', 'estoqueAtualPorProduto'));
    }


    public function create(Empresa $empresaModel, TipoProdutoServico $tipoProdutoServicoModel)
    {
        $empresas = $empresaModel
                        ->orderBy('razaosocial')
                        ->get();

        $prodservTipo = $tipoProdutoServicoModel
                                ->active()
                                ->orderBy('prodservtipo')
                                ->get();

        $unidadesDeMedida = $this->unidadesDeMedida;

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.produto-servico.create')->with(compact('empresas', 'prodservTipo', 'unidadesDeMedida', 'statusOptions'));
    }

    public function store(Request $request)
    {
        $produtoServicoCreate = $request->except('idprodservtipo');
        $tipoIds = $request->input('idprodservtipo', []);

        try
        {
            $prodserv = Model::create($produtoServicoCreate);
            $prodserv->tipos()->sync($tipoIds); // Sincroniza a relação com os tipos selecionados

            return redirect()->to('admin/produto-servico/'.$prodserv->idprodserv.'/edit');

        } catch(Exception $e)
        {
            return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', $e->getMessage());
        }
    }

    public function edit($idProdutoServico, Empresa $empresaModel, TipoProdutoServico $tipoProdutoServicoModel)
    {
        $produtoServico = Model::find($idProdutoServico);
        $empresas = $empresaModel
                    ->orderBy('razaosocial')
                    ->get();
        $prodservTipo = $tipoProdutoServicoModel
            ->active()
            ->orderBy('prodservtipo')
            ->get();

        if ($produtoServico->status == 'ATIVO'){
            $bg_status = "bg-success";
        }elseif($produtoServico->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        $unidadesDeMedida = $this->unidadesDeMedida;

        // Obter a quantidade de estoque atual para cada produto
        $estoqueAtualPorProduto = $this->estoqueService->calcularEstoqueAtual($idProdutoServico);

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.produto-servico.edit')->with(compact('produtoServico', 'prodservTipo', 'empresas', 'unidadesDeMedida','estoqueAtualPorProduto','bg_status', 'statusOptions'));
    }

    public function update($idProdutoServico, Request $request)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $produtoServico = Model::find($idProdutoServico);

        if ($redirect == true) {
            $tipoIds = $request->input('idprodservtipo', []);
        }

        DB::beginTransaction();
        try {
            if ($redirect == true) {
                $produtoServico->update($request->except('idprodservtipo')); // Atualize o ProdutoServico excluindo 'idprodservtipo'
                $produtoServico->tipos()->sync($tipoIds); // Sincroniza a relação com os tipos selecionados
                DB::commit();
            } else {
                $produtoServico->update($request->all());

                DB::commit();
                return response()->json($produtoServico->status);
            }

            return back()->with('success', 'Produto atualizado!');

        } catch(Exception $e) {
            DB::rollBack();
            return redirect()
                    ->route('admin.produto-servico.index')
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function destroy($idProdutoServico)
    {
        $produtoServico = Model::find($idProdutoServico);

        try
        {
            $produtoServico->delete();

            return redirect()
                    ->route('admin.produto-servico.index')
                    ->with('success', 'Produto/Serviço removido!');
        } catch(Exception $e)
        {
            return redirect()
                    ->back()
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function search(Request $request)
    {
        // Aplicando Eager Loading para 'empresa' e 'prodservtipo'
        $produtosServicos = Model::with(['empresa', 'tipos']) // Carrega as relações com Eager Loading
            ->where(function ($query) use ($request) {
                $query->where('prodserv', 'like', '%' . $request->search . '%') // Busca por 'prodserv'
                    ->orWhere('tipo', 'like', '%' . $request->search . '%')
                    ->orWhereHas('tipos', function ($q) use ($request) { // Busca por 'prodservtipo'
                        $q->where('prodservtipo', 'like', '%' . $request->search . '%');
                    })
                    ->orWhereHas('empresa', function ($q) use ($request) { // Busca por 'nomefantasia' ou 'razaosocial'
                        $q->where('nomefantasia', 'like', '%' . $request->search . '%')
                            ->orWhere('razaosocial', 'like', '%' . $request->search . '%');
                    });
            })
            ->orderBy('prodserv.prodserv')
            ->paginate(100);

        $size = $produtosServicos->count();

        // Colocando mensagens na session
        session()->forget('error');
        session()->forget('success');
        if ($size > 0) {
            session()->flash('success', $size . ' valor(es) encontrado(s).');
        } else {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        // Obter a quantidade de estoque atual para cada produto
        $estoqueAtualPorProduto = [];
        foreach ($produtosServicos as $produtoServico) {
            $estoqueAtualPorProduto[$produtoServico->idprodserv] = $this->estoqueService->calcularEstoqueAtual($produtoServico->idprodserv);
        }

        return view('admin.produto-servico.index')
            ->with(compact('produtosServicos', 'estoqueAtualPorProduto'));
    }



}
