<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use App\Models\Pessoa;
use App\Models\Tag;
use App\Models\TagRetira;
use App\Services\TagRetiraService;
use Exception;
use Illuminate\Console\View\Components\Alert;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TagRetiraController extends Controller
{
    public function __construct(
        protected TagRetiraService $tagRetiraService
    ) {}

    public function index(TagRetira $tagRetiraModel)
    {
        $tagRetiradas = $tagRetiraModel
            ->with('tag', 'empresa', 'pessoa')
            ->where('status', '!=', 'CANCELADO')
            ->orderBy('criadoem', 'desc')
            ->get();

        $statusOptions = [
            "PENDENTE" => "Pendente",
            "EM ANDAMENTO" => "Em Andamento",
            "CONCLUÍDO" => "Concluído",
            "RESERVADA" => "Reservada",
            "CANCELADO" => "Cancelado",
        ];

        return view('admin.tagsretira.index', compact('tagRetiradas', 'statusOptions'));
    }

    public function create(Pessoa $pessoaModel, Empresa $empresaModel, Tag $tagModel)
    {
        $tags = $tagModel
            ->orderBy('tag')
            ->get();

        $pessoas = $pessoaModel
            ->orderBy('nome')
            ->funcionarioOuCliente()
            ->get();

        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

        $statusOptions = [
            "PENDENTE" => "Pendente",
            "EM ANDAMENTO" => "Em Andamento",
            "CONCLUÍDO" => "Concluído",
            "RESERVADA" => "Reservada",
            "CANCELADO" => "Cancelado",
        ];

        return view('admin.tagsretira.create', compact('tags', 'pessoas', 'empresas', 'statusOptions'));
    }

    public function store(TagRetira $tagRetiraModel, Request $request)
    {
        $tagRetiraRequest = $request->all();

        try {
            // Verifica se o status é diferente de 'RESERVADA'
            if ($tagRetiraRequest['status'] !== 'RESERVADA') {
                $tagRetiraModel->create($tagRetiraRequest);

                return redirect()
                    ->back()
                    ->with('success', 'Registrado com sucesso!');
            }

            // Se o status for 'RESERVADA'
            if ($tagRetiraRequest['status'] === 'RESERVADA') {
                // Verifica se já existe uma tag reservada com o mesmo ID
                $tagRetira = $tagRetiraModel->where('idtag', $tagRetiraRequest['idtag'])
                    ->where('status', 'RESERVADA')
                    ->first();

                // Se a tag estiver reservada, retorna erro
                if ($tagRetira) {
                    return back()->with('error', 'Tag selecionada já está reservada!');
                }

                // Se não estiver reservada, cria o registro
                $tagRetiraModel->create($tagRetiraRequest);

                return redirect()
                    ->route('admin.tagsretira.edit', $tagRetira)
                    ->with('success', 'Registrado com sucesso!');
            }
        } catch (Exception $e) {
            return back()
                ->with('error', $e->getMessage());
        }
    }


    public function edit(string $idTagRetira, TagRetira $tagRetiraModel, Pessoa $pessoaModel, Empresa $empresaModel, Tag $tagModel)
    {
        if (! $tagRetira = $tagRetiraModel->find($idTagRetira)) {
            return redirect()
                ->route('admin.tagsretira.index')
                ->with('error', 'Registro não encontrado!');
        }

        $pessoas = $pessoaModel
            ->orderBy('nome')
            ->funcionarioOuCliente()
            ->get();

        $empresas = $empresaModel
            ->orderBy('razaosocial')
            ->get();

        $tags = $tagModel
            ->orderBy('tag')
            ->get();

        $statusOptions = [
            "PENDENTE" => "Pendente",
            "EM ANDAMENTO" => "Em Andamento",
            "CONCLUÍDO" => "Concluído",
            "RESERVADA" => "Reservada",
            "CANCELADO" => "Cancelado",
        ];

        if ($tagModel->status === 'ATIVO'){
            $bg_status = "success";
        }elseif($tagModel->status === 'INATIVO'){
            $bg_status = "danger";
        }else{
            $bg_status = "warning";
        }

        return view('admin.tagsretira.edit', compact('tagRetira', 'pessoas', 'empresas', 'tags', 'statusOptions', 'bg_status'));
    }

    public function update($idTagRetira, Request $request)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $tagRetiraRequest = $request->all();
        // Retorna a tag selecionada para verificar status atual
        $tagRetira = TagRetira::find($idTagRetira);

        DB::beginTransaction();
        try {
            // Verifica se a tag já está concluída
            if ($tagRetira->status === 'CONCLUÍDO') {
                // Permite a alteração apenas da observação se a tag estiver concluída
                if (isset($tagRetiraRequest['observacao'])) {
                    $tagRetira->observacao = $tagRetiraRequest['observacao'];
                    $tagRetira->status = $tagRetiraRequest['status'];
                    $tagRetira->save();

                    return redirect()
                        ->back()
                        ->with('success', 'A tag já está concluída, outras informações não podem ser alteradas.');
                }

                if ($redirect == false) {
                    return response()->json([
                        'error' => 'A tag já está concluída, apenas a observação pode ser alterada.',
                    ]);
                }

                return redirect()
                    ->back()
                    ->with('error', 'A tag já foi concluída, apenas a observação pode ser alterada.');
            }

            // Verifica se já existe outra tag com o mesmo 'idtag' e com status 'RESERVADA'
            $existingReservation = TagRetira::where('idtag', $tagRetiraRequest['idtag'] ?? $tagRetira->idtag)
                ->where('status', 'RESERVADA')
                ->where('idtagretira', '!=', $idTagRetira) // Exclui a tag atual
                ->first();


            // Se já houver uma tag reservada com o mesmo 'idtag' e não for a tag atual e o status for 'RESERVADA', retorna erro
            if ($existingReservation && $tagRetiraRequest['status'] === 'RESERVADA') {

                if ($redirect == false) {
                    return response()->json([
                        'error' => 'Esta tag já foi reservada por outro colaborador.',
                    ]);
                }

                return redirect()
                    ->back()
                    ->with('error', 'Esta tag já foi reservada por outro colaborador.');
            }

            // Caso contrário, permite a atualização
            $tagRetira->update($tagRetiraRequest);

            if ($redirect == false) {
                DB::commit();
                return response()->json($tagRetira->status);
            }

            DB::commit();
            return redirect()
                ->back()
                ->with('success', 'Atualizado com sucesso!');
        } catch (Exception $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->with('error', $e->getMessage());
        }
    }

    public function destroy($idTagRetira, TagRetira $tagRetiraModel)
    {
        if (! $tagRetira = $tagRetiraModel->find($idTagRetira)) {
            return redirect()
                ->back()
                ->with('error', 'Tag Retira não encontrada');
        }

        try {
            $tagRetira->delete();

            return redirect()
                ->back()
                ->with('success', 'Excluído com sucesso');
        } catch (Exception $e) {
            return redirect()
                ->back()
                ->with('error', $e->getMessage());
        }
    }

    public function search(Request $request, TagRetira $tagRetiraModel)
    {
        $tagRetiradas = $tagRetiraModel
            ->with(['pessoa', 'tag', 'empresa'])
            ->where(function ($query) use ($request) {
                $query->where('status', $request->status)
                    ->where(function ($query) use ($request) {
                        $query->whereHas('pessoa', function ($query) use ($request) {
                            $query->where('nome', 'like', '%' . $request->search . '%');
                        })
                            ->orWhereHas('empresa', function ($query) use ($request) {
                                $query->where('razaosocial', 'like', '%' . $request->search . '%');
                            })
                            ->orWhereHas('tag', function ($query) use ($request) {
                                $query->where('tag', 'like', '%' . $request->search . '%');
                            });
                    });
            })
            ->orderBy('tagretira.idempresa')
            ->orderBy('tagretira.idpessoa')
            ->orderBy('tagretira.idtag')
            ->paginate(100);

        $size = $tagRetiradas->count();

        // Limpa as mensagens anteriores da sessão
        session()->forget('error');
        session()->forget('success');

        // Define mensagens de sucesso ou erro
        if ($size && $size > 0) {
            session()->flash('success', $size . ' valores encontrado(s).');
        } else {
            session()->flash('error', 'Nenhum registro encontrado.');
        }

        $statusOptions = [
            "PENDENTE" => "Pendente",
            "EM ANDAMENTO" => "Em Andamento",
            "CONCLUÍDO" => "Concluído",
            "RESERVADA" => "Reservada",
            "CANCELADO" => "Cancelado",
        ];

        // Retorna a view com os resultados
        return view('admin.tagsretira.index', compact('tagRetiradas', 'statusOptions')); // Corrige o nome da variável
    }

    public function getTagReservas($idtag)
    {
        $result = $this->tagRetiraService->get($idtag);

        return $result;
    }
}
