<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Pessoa;
use App\Models\Tenant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TenantController extends Controller
{
    // Injeta middleware diretamente no controller
    public function __construct()
    {
        $this->middleware('admin.tenant');
    }

    public function index(Tenant $tenantModel)
    {
        $tenants = $tenantModel
            ->with('pessoa')
            ->orderBy('nome')
            ->get();

        return view('admin.tenants.index', compact('tenants'));
    }

    public function create(Pessoa $pessoaModel)
    {
        $pessoas = $pessoaModel
            ->orderBy('nome')
            ->funcionarioOuCliente()
            ->get();

        $tipos = ['VENDAS' => 'Vendas', 'RECEITAS' => 'Receitas'];


        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.tenants.create', compact('pessoas', 'statusOptions', 'tipos'));
    }

    public function store(Request $request, Tenant $tenantModel)
    {
        $tenantRequest = $request->validate([
            'nome' => 'required',
            'idpessoa' => 'nullable|exists:pessoa,idpessoa',
            'is_admin_tenant' => 'nullable|string',
            'status' => 'required|in:ATIVO,INATIVO',
        ]);

        try {

            $tenantModel->create([
                'nome' => $tenantRequest['nome'],
                'idpessoa' => $tenantRequest['idpessoa'],
                'is_admin_tenant' => isset($tenantRequest['is_admin_tenant']) && $tenantRequest['is_admin_tenant'] == 'on' ? 1 : 0,
                'status' => $tenantRequest['status'],
            ]);
            return redirect()->route('admin.tenants.index')->with('success', 'Tenant criado com sucesso');

        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->with('error', $e->getMessage());
        }
    }

    public function edit(Tenant $tenant, Pessoa $pessoaModel)
    {
        if (! $tenant) {
            return redirect()
                ->route('admin.tenants.index')
                ->with('error', 'Tenant não encontrado');
        }

        $pessoas = $pessoaModel
            ->orderBy('nome')
            ->funcionarioOuCliente()
            ->get();

        $tipos = ['VENDAS' => 'Vendas', 'RECEITAS' => 'Receitas'];

        $bg_status = $tenant->status == 'ATIVO' ? 'bg-success' : 'bg-danger';

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
        ];

        return view('admin.tenants.edit', compact('tenant', 'pessoas', 'bg_status', 'statusOptions', 'tipos'));
    }

    public function update(Tenant $tenant, Request $request)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;

        if ($redirect == true) {
            $tenantRequest = $request->validate([
                'nome' => 'required',
                'idpessoa' => 'nullable|exists:pessoa,idpessoa',
                'is_admin_tenant' => 'nullable|string',
                'status' => 'required|in:ATIVO,INATIVO',
            ]);

            // Tratamento para is_admin_tenant
            if (isset($tenantRequest['is_admin_tenant']) && $tenantRequest['is_admin_tenant'] == 'on') {
                $tenantRequest['is_admin_tenant'] = 1;
            } else {
                $tenantRequest['is_admin_tenant'] = 0;
            }
        } else {
            $tenantRequest = $request->all();
        }

        DB::beginTransaction();
        try {
            $tenant->update($tenantRequest);

            if ($redirect == false) {
                DB::commit();
                return response()->json($tenant->status);
            }

            DB::commit();
            return redirect()->route('admin.tenants.edit', $tenant)->with('success', 'Tenant atualizado');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->with('error', $e->getMessage());
        }
    }

    public function destroy(Tenant $tenant)
    {
        try {

            $tenant->delete();
            return redirect()->route('admin.tenants.index')->with('success', 'Tenant excluído');

        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->with('error', $e->getMessage());
        }
    }

    public function search(Request $request, Tenant $tenantModel)
    {
        $search = $request->input('search');
        $tenants = $tenantModel
            ->where('nome', 'like', '%' . $search . '%')
            ->orderBy('nome')
            ->paginate(100);

        return view('admin.tenants.index', compact('tenants'));
    }
}
