<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use App\Models\Pessoa;
use App\Models\Veiculo as Model;
use App\Models\ContratoVeiculo;
use Exception;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class VeiculoController extends Controller
{
    protected $marcas = [
		'audi' => "Audi",
		'bmw' => "BMW",
		'chevrolet' => "Chevrolet",
        'FACHHINI' => "FACHHINI",
		'FIAT' => "FIAT",
		'FORD' => "FORD",
        'GM' => "GM",
		'honda' => "Honda",
		'hyundai' => "Hyundai",
        'IDEROL' => "IDEROL",
        'IVECO' => "IVECO",
		'jaguar' => "Jaguar",
		'jeep' => "Jeep",
        'John Deere' => 'John Deere',
		'kia' => "Kia",
		'landrover' => "Land Rover",
        'LIU GONG' => "LIU GONG",
  		'mazda' => "Mazda",
        'MICHIGAN' => "MICHIGAN",
		'MB' => "MB",
        'New Holland' => 'New Holland',
		'nissan' => "Nissan",
		'porsche' => "Porsche",
        'RANDON' => "RANDON",
        'ROBERTO' => "ROBERTO",
		'renault' => "Renault",
		'subaru' => "Subaru",
		'tesla' => "Tesla",
		'toyota' => "Toyota",
        'VOLVO' => "VOLVO",
		'VW' => "VW",
        'outros' => "Outros"
    ];

	 protected $tipoveiculo = [
		'bicicleta' => "Bicicleta",
        'caminhao' => "Caminhão",
        'carro' => "Carro",
        'maquina' => "Máquina",
        'motocicleta' => "Motocicleta",
        'onibus' => "Ônibus"
    ];

	protected $cor = [
		'amarelo' => "Amarelo",
		'azul' => "Azul",
		'bege' => "Bege",
		'branco' => "Branco",
		'cinza' => "Cinza",
		'dourado' => "Dourado",
		'laranja' => "Laranja",
		'marrom' => "Marrom",
		'preto' => "Preto",
		'prata' => "Prata",
		'rosa' => "Rosa",
		'roxo' => "Roxo",
		'verde' => "Verde",
		'vermelho' => "Vermelho"
	];

    public function index(Request $request)
    {
        $module = $request->segment(2); // 'vendas' ou 'pedidos'
        $key = $module . '_dateRange'; // chave será 'vendas_dateRange' ou 'pedidos_dateRange'

        if ($request->has('clear') || session($key) === "") {
            session()->forget($key);
        }

        if (session($key)) {
            // Redirecionar para search com o date_range da session específica do módulo
            return redirect()->route('admin.veiculos.search', ['date_range' => session($key)]);
        }

        // Aplicando eager loading para empresa e pessoa, evitando N+1 queries
        $veiculos = Model::with(['empresa', 'pessoa'])
            ->select('veiculo.*', 'veiculo.status as status')
            ->join('empresa', 'empresa.idempresa', '=', 'veiculo.idempresa')
            ->orderBy('empresa.sigla')
            ->orderBy('veiculo.marca')
            ->orderBy('veiculo.placa')
            ->paginate(100);

        $marcas = $this->marcas;
        $tipoveiculo = $this->tipoveiculo;
        $cor = $this->cor;

        return view('admin.veiculos.index')->with(compact('veiculos', 'marcas', 'tipoveiculo', 'cor'));
    }


    public function create(Pessoa $pessoaModel, Empresa $empresaModel)
    {
        $pessoas = $pessoaModel
                    ->orderBy('nome')
                    ->get();

        $empresas = $empresaModel
                        ->orderBy('razaosocial')
                        ->get();

        $fornecedores = $pessoaModel
                        ->orderBy('nome')
                        ->fornecedor()
                        ->get();

        $marcas = $this->marcas;

		$tipoveiculo = $this->tipoveiculo;

		$cor = $this->cor;

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.veiculos.create')->with(compact('pessoas', 'empresas', 'marcas', 'tipoveiculo', 'cor','fornecedores', 'statusOptions'));
    }

    public function store(Request $request)
    {
        $veiculoCreate = $request->all();

        // Defina o valor do campo dashboard com base na presença do checkbox no formulário
        $veiculoCreate['dashboard'] = $request->has('dashboard') ? 1 : 0;

        DB::beginTransaction();
        try {
            $veiculo = Model::create($veiculoCreate);

            DB::commit();
            return redirect()->to('admin/veiculos/'.$veiculo->idveiculo.'/edit');

        } catch(Exception $e) {
            DB::rollBack();
            return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', $e->getMessage());
        }
    }

    public function edit($idVeiculo, Empresa $empresaModel, Pessoa $pessoaModel, ContratoVeiculo $contratoVeiculoModel)
    {
        $veiculo = Model::find($idVeiculo);
        $pessoas = $pessoaModel
                    ->orderBy('nome')
                    ->get();
        $empresas = $empresaModel
                    ->orderBy('razaosocial')
                    ->get();

        $fornecedores = $pessoaModel
                    ->orderBy('nome')
                    ->fornecedor()
                    ->get();

        $marcas = $this->marcas;

		$tipoveiculo = $this->tipoveiculo;

		$cor = $this->cor;

        $contratosveiculos = $contratoVeiculoModel
        ->where('idveiculo', $idVeiculo) // Filtra pelos registros com o idnf relacionado
        ->get();


        if ($veiculo->status == 'ATIVO'){
            $bg_status = "bg-success";
        }elseif($veiculo->status == 'INATIVO'){
            $bg_status = "bg-danger";
        }else{
            $bg_status = "bg-info";
        }

        $statusOptions = [
            'ATIVO' => 'Ativo',
            'INATIVO' => 'Inativo',
                // Adicione mais status conforme necessário
            ];

        return view('admin.veiculos.edit')->with(compact('veiculo', 'empresas', 'pessoas', 'marcas', 'tipoveiculo', 'cor','bg_status','contratosveiculos','fornecedores', 'statusOptions'));
    }

    public function update($idVeiculo, Request $request)
    {
        $redirect = $request->has('redirect') ? filter_var($request->get('redirect'), FILTER_VALIDATE_BOOLEAN) : true;
        $veiculo = Model::find($idVeiculo);
        $veiculoRequest = $request->all();

        DB::beginTransaction();
        try {
            if ($redirect == true) {
                // Atualiza o valor do campo 'dashboard' com base no estado do checkbox
                $veiculo->dashboard = $request->input('dashboard') ? 1 : 0;

                // Atualiza os outros campos do veículo com os dados recebidos do formulário
                $veiculo->update($request->except('dashboard'));
            }

            if ($redirect == false) {
                $veiculo->update($veiculoRequest);

                DB::commit();
                return response()->json($veiculo->status);
            }

            DB::commit();
            return back()->with('success', 'Veículo atualizado!');

        } catch(Exception $e) {
            DB::rollBack();
            return redirect()
            ->back()
            ->with('error', $e->getMessage())
            ->withInput();
        }
    }

    public function destroy($idVeiculo)
    {
        $veiculo = Model::find($idVeiculo);

        try
        {
            $veiculo->delete();

            return redirect()
                    ->route('admin.veiculos.index')
                    ->with('success', 'Veículo removido!');
        } catch(Exception $e)
        {
            return redirect()
                    ->back()
                    ->with('error', $e->getMessage())
                    ->withInput();
        }
    }

    public function search(Request $request)
    {
        // Separar a data em data inicial e data final
        $intervaloDatas = explode(' - ', $request->date_range);
        $dataInicio = null;
        $dataFim = null;

        // Verifica se o intervalo de datas é válido
        if (count($intervaloDatas) >= 2) {
            try {
                $dataInicio = Carbon::createFromFormat('d/m/y', $intervaloDatas[0])->format('Y-m-d');
                $dataFim = Carbon::createFromFormat('d/m/y', $intervaloDatas[1])->format('Y-m-d');
            } catch (\Exception $e) {
                // Tratar erro de data inválida, se necessário
            }
        }

        // Aplicando eager loading para evitar N+1 queries
        $veiculos = Model::query()
                        ->join('empresa', 'veiculo.idempresa', '=', 'empresa.idempresa') // Inclua um join para acessar 'empresa.sigla'
                        ->with(['empresa', 'pessoa'])
                        ->when($dataInicio && $dataFim, function ($query) use ($dataInicio, $dataFim) {
                            $query->whereBetween('datavencimento', [$dataInicio, $dataFim]);
                        })
                        ->where(function ($query) use ($request) {
                            $query->where('marca', 'like', '%' . $request->search . '%')
                                ->orWhere('tipoveiculo', 'like', '%' . $request->search . '%')
                                ->orWhere('modelo', 'like', '%' . $request->search . '%')
                                ->orWhere('placa', 'like', '%' . $request->search . '%')
                                ->orWhereHas('empresa', function ($q) use ($request) {
                                    $q->where('nomefantasia', 'like', '%' . $request->search . '%')
                                        ->orWhere('razaosocial', 'like', '%' . $request->search . '%');
                                })
                                ->orWhereHas('pessoa', function ($q) use ($request) {
                                    $q->where('nome', 'like', '%' . $request->search . '%')
                                        ->orWhere('razaosocial', 'like', '%' . $request->search . '%');
                                });
                        })
                        ->orderBy('empresa.sigla') // Ordena usando a coluna agora acessível
                        ->orderBy('veiculo.marca')
                        ->orderBy('veiculo.placa')
                        ->paginate(100);


        // Tamanho da consulta
        $size = $veiculos->count();

        // Definindo mensagens de sessão
        session()->forget(['error', 'success']);
        session()->flash($size > 0 ? 'success' : 'error', $size > 0 ? $size . ' valores encontrado(s).' : 'Nenhum registro encontrado.');

        // Recuperar as listas de marcas, tipos de veículos e cores
        $marcas = $this->marcas;
        $tipoveiculo = $this->tipoveiculo;
        $cor = $this->cor;

        // Armazenar dateRange na session
        $module = $request->segment(2); // 'vendas' ou 'pedidos'
        $key = $module . '_dateRange';
        $dateRange = session($key, $request->date_range ? urldecode($request->date_range) : '');
        session([$key => $dateRange]);

        return view('admin.veiculos.index', compact('veiculos', 'marcas', 'tipoveiculo', 'cor', 'dateRange'));
    }


}
