<?php

namespace App\Http\Controllers;

Use Str;
Use Hash;
use Illuminate\Auth\Events\PasswordReset;
use App\Models\User;
use App\Models\UserHist;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Password;

class SessionsController extends Controller
{
    public function create()
    {
        return view('sessions.create');
    }

    public function store()
    {
        $attributes = request()->validate([
            'login' => 'required|string',
            'password' => 'required'
        ]);

        $login = $attributes['login'];
        $field = filter_var($login, FILTER_VALIDATE_EMAIL) ? 'email' : 'username';

        if (!auth()->attempt([$field => $login, 'password' => $attributes['password'], 'status' => 'ATIVO'])) {
            $user = User::where($field, $login)->first();
            if ($user && $user->status !== 'ATIVO') {
                return back()->withErrors(['login' => 'Sua conta está inativa. Por favor, entre em contato com o administrador.']);
            }
            return back()->withErrors(['login' => 'Credenciais inválidas.']);
        }

        session()->regenerate();

        // Atualizar o session_id do usuário no banco de dados
        $user = User::find(auth()->user()->id);
        $user->idsession = session()->getId();
        $user->last_login_at = Carbon::now();
        $user->save();

        // Registra o acesso do usuario no sistema
        UserHist::create([
            'iduser' => auth()->user()->id,
        ]);

        return redirect('/dashboard/financeiro');

    }

    public function show(){
        request()->validate([
            'email' => 'required|email',
        ]);

        $status = Password::sendResetLink(
            request()->only('email')
        );

        return $status === Password::RESET_LINK_SENT
                    ? back()->with(['status' => __($status)])
                    : back()->withErrors(['email' => __($status)]);

    }

    public function update(){

        request()->validate([
            'token' => 'required',
            'email' => 'required|email',
            'password' => 'required|min:8|confirmed',
        ]);

        $status = Password::reset(
            request()->only('email', 'password', 'password_confirmation', 'token'),
            function ($user, $password) {
                $user->forceFill([
                    'password' => ($password)
                ])->setRememberToken(Str::random(60));

                $user->save();

                event(new PasswordReset($user));
            }
        );

        return $status === Password::PASSWORD_RESET
                    ? redirect()->route('login')->with('status', __($status))
                    : back()->withErrors(['email' => [__($status)]]);
    }

    public function destroy()
    {
        auth()->logout();

        return redirect('/sign-in');
    }

}
