<?php

namespace App\Http\Livewire;

use Livewire\Component;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Models\TagRetira;

class CalendarTagComponent extends Component
{
    public $tags;

    public function mount()
    {
        $visualizacao = request()->query('visualizacao', 'duracao');

        // Defina o intervalo de datas que deseja exibir no calendário
        $inicioMesAtual = Carbon::now()->startOfMonth();
        $finalMesAtual = Carbon::now()->endOfMonth();

        // Carregar as tags para exibir no calendário
        $this->tags = TagRetira::query()
            ->select('tagretira.*',)
            ->with('pessoa', 'tag', 'tag.tagTipo')
            ->whereIn('tagretira.status', ['PENDENTE', 'RESERVADA'])
            ->whereBetween('tagretira.datainicio', [$inicioMesAtual, $finalMesAtual])
            ->get()
            ->map(function ($tag) use ($visualizacao) {
                $start = $tag->datainicio;
                $end = $tag->datafim;

                if ($tag->status == 'RESERVADA') {
                    $backgroundColor = '#EF5350';
                }

                if ($tag->status == 'PENDENTE') {
                    $backgroundColor = '#FFA726';
                }

                $textColor = '#fffeee';

                // Condicionais para visualização
                switch ($visualizacao) {
                    case 'inicio':
                        $end = ''; // Ignora a data/hora de término
                        break;
                    case 'termino':
                        $start = $end;
                        $end = '';
                        break;
                }

                // Informações da tag e colaborador
                $taginfo = "{$tag->tag?->tagTipo?->tagtipo} / {$tag->tag?->tag}";
                $colaborador = Str::upper($tag->pessoa?->nome ?? 'Sem Colaborador');

                return [
                    'title' => "TAG {$tag->status}: ($taginfo) {$colaborador}",
                    'start' => $start,
                    'end' => $end,
                    'className' => 'event-color-' . Str::slug($tag->idtag . '-' . $tag->status),
                    'backgroundColor' => $backgroundColor,
                    'textColor' => $textColor,
                    'url' => '/admin/tagsretira/' . $tag->idtagretira . '/edit',
                ];
            });
    }

    public function render()
    {
        return view('livewire.calendar-tag-component', [
            'tags' => $this->tags
        ]);
    }

    public function textColorForBackground($backgroundColor) {
        // Remover '#' se estiver presente
        $color = ltrim($backgroundColor, '#');

        // Convertendo o HEX em componentes RGB
        $r = hexdec(substr($color, 0, 2));
        $g = hexdec(substr($color, 2, 2));
        $b = hexdec(substr($color, 4, 2));

        // Calculando a luminância
        $luminance = (0.299 * $r + 0.587 * $g + 0.114 * $b) / 255;

        // Se a luminância for menor que 0.5, a cor de fundo é escura, então usamos branco; senão, usamos #333
        return $luminance < 0.5 ? '#FFFFFF' : '#333333';
    }
}



