<?php

namespace App\Http\Livewire;

use Livewire\Component;
use App\Models\NfParcela;
use Carbon\Carbon;

class GraphLucroLiquidoComponent extends Component
{
    public $DadosLucroLiquidoMes;
    public $GraficoLucroLiquidoMes;

    public function mount()
    {
        $dadosMeses = collect();

        // Preenche os últimos 6 meses
        for ($i = 5; $i >= 0; $i--) {
            $mes = Carbon::now()->subMonths($i);
            $dadosMeses->push([
                'mes' => $mes->month,
                'ano' => $mes->year,
                'totalCredito' => 0,
                'totalDebito' => 0,
                'lucroLiquido' => 0
            ]);
        }

        // Consulta os dados de crédito e débito
        $dadosCredito = NfParcela::credito()->concluido()
            ->selectRaw('MONTH(datavencimento) as mes, YEAR(datavencimento) as ano, SUM(valor) as total')
            ->where('datavencimento', '>=', Carbon::now()->subMonths(6))
            ->groupBy('mes', 'ano')
            ->get();

        $dadosDebito = NfParcela::debito()->concluido()
            ->selectRaw('MONTH(datavencimento) as mes, YEAR(datavencimento) as ano, SUM(valor) as total')
            ->where('datavencimento', '>=', Carbon::now()->subMonths(6))
            ->groupBy('mes', 'ano')
            ->get();

        // Preenche os dados nos meses
        $dadosMeses->transform(function ($mes) use ($dadosCredito, $dadosDebito) {
            $credito = $dadosCredito->firstWhere('mes', $mes['mes'])->total ?? 0;
            $debito = $dadosDebito->firstWhere('mes', $mes['mes'])->total ?? 0;

            $mes['totalCredito'] = $credito;
            $mes['totalDebito'] = $debito;
            $mes['lucroLiquido'] = $credito - $debito;

            return $mes;
        });

        $this->DadosLucroLiquidoMes = $dadosMeses;

        $this->GraficoLucroLiquidoMes = [
            'labels' => $dadosMeses->pluck('mes')->map(function ($mes) {
                return Carbon::createFromFormat('m', $mes)->format('F');
            })->toArray(),
            'data' => $dadosMeses->pluck('lucroLiquido')->toArray()
        ];
    }

    public function render()
    {
        return view('livewire.graph-lucro-liquido-component', [
            'DadosLucroLiquidoMes' => $this->DadosLucroLiquidoMes,
            'GraficoLucroLiquidoMes' => $this->GraficoLucroLiquidoMes
        ]);
    }
}
