<?php

namespace App\Http\Livewire;

use Livewire\Component;
use App\Models\NfParcela;
use Carbon\Carbon;

class GraphLucroLiquidoJsComponent extends Component
{
    public $DadosLucroLiquidoMes;
    public $GraficoLucroLiquidoMes;

    public function mount()
    {
        $dadosMeses = collect();

        // Preenche os últimos 6 meses
        for ($i = 5; $i >= 0; $i--) {
            $mes = Carbon::now()->subMonths($i);
            $dadosMeses->push([
                'mes' => $mes->month,
                'ano' => $mes->year,
                'totalCredito' => 0,
                'totalDebito' => 0,
                'lucroLiquido' => 0
            ]);
        }

        // Consulta os dados de crédito e débito
        $dadosCredito = NfParcela::credito()->concluido()
            ->selectRaw('MONTH(datavencimento) as mes, YEAR(datavencimento) as ano, COALESCE(SUM(valor), 0) as total')
            ->where('datavencimento', '>=', Carbon::now()->subMonths(6))
            ->groupBy('mes', 'ano')
            ->get();

        $dadosDebito = NfParcela::debito()->concluido()
            ->selectRaw('MONTH(datavencimento) as mes, YEAR(datavencimento) as ano, COALESCE(SUM(valor), 0) as total')
            ->where('datavencimento', '>=', Carbon::now()->subMonths(6))
            ->groupBy('mes', 'ano')
            ->get();

        // Preenche os dados nos meses
        $dadosMeses->transform(function ($mes) use ($dadosCredito, $dadosDebito) {
            foreach ($dadosCredito as $credito) {
                if ($mes['mes'] == $credito->mes && $mes['ano'] == $credito->ano) {
                    $mes['totalCredito'] = $credito->total;
                    break;
                }
            }
            foreach ($dadosDebito as $debito) {
                if ($mes['mes'] == $debito->mes && $mes['ano'] == $debito->ano) {
                    $mes['totalDebito'] = $debito->total;
                    break;
                }
            }
            $mes['lucroLiquido'] = $mes['totalCredito'] - $mes['totalDebito'];
            return $mes;
        });

        $this->DadosLucroLiquidoMes = $dadosMeses;

        $this->GraficoLucroLiquidoMes = [
            'labels' => $dadosMeses->map(fn($mes) => "{$mes['mes']}/{$mes['ano']}")->toArray(),
            'data' => $dadosMeses->map(fn($mes) => $mes['lucroLiquido'])->toArray()
        ];
    }

    public function render()
    {
        return view('livewire.graph-lucro-liquido-js-component', [
            'DadosLucroLiquidoMes' => $this->DadosLucroLiquidoMes,
            'GraficoLucroLiquidoMes' => $this->GraficoLucroLiquidoMes
        ]);
    }
}
