<?php

namespace App\Http\Livewire;

use Livewire\Component;
use App\Models\Nf;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Services\DateRangeService;

class GraphMediaPedidoDieselJsComponent extends Component
{
    public $mediasConsumoMesAtual;

    public $datarange;
    public $idempresa;
    public $idcontrato;
    public $idveiculo;


   protected $listeners = ['filtersApplied' => 'onFilterUpdate'];

    public function mount()
    {
        $this->loadInitialData();
    }

    public function loadInitialData()
    {
        if (empty($this->datarange)) {
            $this->setInitialDateRange();
        }
        $this->loadData();
    }

    private function setInitialDateRange()
    {
        $dataInicio = Carbon::now()->startOfMonth()->format('d-m-Y');
        $dataFim = Carbon::now()->endOfMonth()->format('d-m-Y');
        $this->datarange = $dataInicio . ' até ' . $dataFim;
    }

    private function loadData()
    {
        $dates = DateRangeService::generateDateRange($this->datarange);
        $inicioMesAtual = '2024-01-01';
        $finalMesAtual = '2024-01-30';

         // Agora você pode chamar a função e obter as médias ordenadas
        $this->mediasConsumoMesAtual = $this->calcularMediaConsumoPorVeiculo($dates['dataInicio'], $dates['dataFim']);
        //$this->mediasConsumoMesAtual = $this->calcularMediaConsumoPorVeiculo($inicioMesAtual, $finalMesAtual);

        $this->dispatchBrowserEvent('MediaPedidoDieselUpdated', [
            'mediasConsumoMesAtual' => $this->mediasConsumoMesAtual
        ]);

        // Exemplo de como exibir as médias de consumo por veículo
    //    foreach ($mediasConsumoMesAtual as $idVeiculo => $mediaConsumo) {
      //      echo "Veículo $idVeiculo: " . number_format($mediaConsumo, 2, '.', '') . " km/l<br>";
      //  }

    }

    public function render()
    {
        return view('livewire.graph-media-pedido-diesel-js-component', [
            'mediasConsumoMesAtual' => $this->mediasConsumoMesAtual
        ]);
    }

    function calcularMediaConsumoPorVeiculo($inicio, $fim) {
        // Inicializa a consulta com condições básicas
        $sql = "
            SELECT
                n.idnf,
                ni.qtd,
                n.km,
                n.idveiculo,
                v.placa as rotulo
            FROM nf n
            JOIN nfitem ni ON n.idnf = ni.idnf
            JOIN veiculo v on v.idveiculo = n.idveiculo and v.dashboard = 1
            WHERE ni.idprodserv IN (20, 31)
            AND n.dataentrada BETWEEN ? AND ?
            AND n.saida = 'P'
            AND ni.qtd > 0";

        $bindings = [$inicio, $fim];

        // Adiciona condições de filtro se idempresa ou idveiculo forem fornecidos
        if (!empty($this->idempresa)) {
            $sql .= " AND n.idempresa = ?";
            $bindings[] = $this->idempresa;
        }

        if (!empty($this->idveiculo)) {
            $sql .= " AND n.idveiculo = ?";
            $bindings[] = $this->idveiculo;
        }

        // Adiciona a condição para verificar a existência do contrato se idcontrato for fornecido
        if (!empty($this->idcontrato)) {
            $sql .= " AND EXISTS (
                SELECT 1
                FROM contratoveiculo cv
                WHERE cv.idveiculo = n.idveiculo
                AND cv.idcontrato = ?
            )";
            $bindings[] = $this->idcontrato;
        }

        $sql .= " ORDER BY n.idveiculo, n.dataentrada";

        // Executa a consulta
        $resultados = DB::select(DB::raw($sql), $bindings);

        // Estruturas para armazenar as informações por veículo
        $informacoesPorVeiculo = [];
        foreach ($resultados as $registro) {
            $idVeiculo = $registro->idveiculo;
            $kmAtual = $registro->km;
            $litros = $registro->qtd;
            $rotulo = $registro->rotulo; // Placa do veículo

            if (!isset($informacoesPorVeiculo[$idVeiculo])) {
                $kmAnterior = $this->getUltimoKmAntesDe($idVeiculo, $inicio);
                $informacoesPorVeiculo[$idVeiculo] = [
                    'kmAnterior' => $kmAnterior,
                    'distanciaTotal' => 0,
                    'litrosTotal' => 0,
                    'rotulo' => $rotulo
                ];
            }

            $distancia = $kmAtual - $informacoesPorVeiculo[$idVeiculo]['kmAnterior'];
            $informacoesPorVeiculo[$idVeiculo]['distanciaTotal'] += max($distancia, 0);
            $informacoesPorVeiculo[$idVeiculo]['kmAnterior'] = $kmAtual;

            $informacoesPorVeiculo[$idVeiculo]['litrosTotal'] += $litros;
        }

        // Calcula a média de consumo por veículo
        $mediasConsumoPorVeiculo = [];
        foreach ($informacoesPorVeiculo as $idVeiculo => $info) {
            $mediaConsumo = $info['litrosTotal'] > 0 ? $info['distanciaTotal'] / $info['litrosTotal'] : 0;
            if ($mediaConsumo > 0) {
                $mediasConsumoPorVeiculo[] = [
                    'id' => $idVeiculo,
                    'rotulo' => $info['rotulo'],
                    'valor' => $mediaConsumo
                ];
            }
        }

        // Ordena o array pelo consumo médio
        usort($mediasConsumoPorVeiculo, function ($item1, $item2) {
            return $item2['valor'] <=> $item1['valor'];
        });

        return $mediasConsumoPorVeiculo;
    }

    function getUltimoKmAntesDe($idVeiculo, $dataInicio) {
        $ultimoRegistro = DB::table('nf')
            ->join('nfitem', 'nfitem.idnf', '=', 'nf.idnf')
            ->where('nf.idveiculo', $idVeiculo)
            ->where('nf.dataentrada', '<', $dataInicio)
            ->whereIn('nfitem.idprodserv', [20, 31])  // Correção aqui, usando whereIn corretamente
            ->orderByDesc('nf.dataentrada')
            ->first();

        return $ultimoRegistro ? $ultimoRegistro->km : 0;
    }


    public function onFilterUpdate($filters)
    {
        $this->datarange = $filters['datarange'] ?? null;
        $this->idempresa = $filters['idempresa'] ?? null;
        $this->idcontrato = $filters['idcontrato'] ?? null;
        $this->idveiculo = $filters['idveiculo'] ?? null;
        $this->loadData();
//dd($this->datarange);
        $this->dispatchBrowserEvent('MediaPedidoDieselUpdated', [
            'mediasConsumoMesAtual' => $this->mediasConsumoMesAtual
        ]);
    }
}
