<?php

namespace App\Http\Livewire;

use Livewire\Component;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Services\DateRangeService;
use Illuminate\Support\Facades\Log;

class GraphMovimentacaoComponent extends Component
{
    public $valorTotalGrupoItem;
    public $datarange;
    public $idempresa;
    public $idveiculo;
    public $idGrupoItem;
    public $idcontrato;
    public $idagencia;

    protected $listeners = ['filtersApplied' => 'onFilterUpdate'];

    public function mount()
    {
        $this->loadInitialData();
    }

    public function loadInitialData()
    {
        if (empty($this->datarange)) {
            $this->setInitialDateRange();
        }
        $this->loadData();
    }

    private function setInitialDateRange()
    {
        $dataInicio = Carbon::now()->startOfMonth()->format('d-m-Y');
        $dataFim = Carbon::now()->endOfMonth()->format('d-m-Y');
        $this->datarange = $dataInicio . ' até ' . $dataFim;
    }

    private function loadData()
    {
        //Log::info('Data: ' . $this->datarange);
        $dates = DateRangeService::generateDateRange($this->datarange);
        $this->valorTotalGrupoItem = $this->calcularValorTotalPorGrupoItem($dates['dataInicio'], $dates['dataFim']);

        $this->dispatchBrowserEvent('MovimentacaoUpdated', [
            'valorTotalGrupoItem' => $this->valorTotalGrupoItem
        ]);
    }

    public function render()
    {
        return view('livewire.graph-movimentacao-component', [
            'valorTotalGrupoItem' => $this->valorTotalGrupoItem
        ]);
    }

    function calcularValorTotalPorGrupoItem($inicio, $fim)
    {
        // Pegando o tenant atual do usuário
        $user = auth()->user();
        $selectedTenantId = session('selectedTenant') ? session('selectedTenant')->idtenant : $user->idtenant;

        // Consulta Eloquent
        $query = \App\Models\Nf::join('nfparcela', 'nf.idnf', '=', 'nfparcela.idnf')
            ->join('nfitem', 'nf.idnf', '=', 'nfitem.idnf')
            ->join('empresa', 'empresa.idempresa', '=','nf.idempresa')
            ->leftJoin('grupoitem', 'nfitem.idgrupoitem', '=', 'grupoitem.idgrupoitem')
            ->where('nf.saida', '<>', 'P')
            ->where('nfparcela.status', 'CONCLUIDO')
            ->whereBetween('nfparcela.datatransacao', [$inicio, $fim])
            ->where('empresa.idtenant', $selectedTenantId); // Aplicando manualmente o filtro de tenant

        // Adiciona condições de filtro se idempresa, idveiculo ou idcontrato forem fornecidos
        if (!empty($this->idempresa)) {
            $query->where('nf.idempresa', $this->idempresa);
        }

        if (!empty($this->idcontrato)) {
            $query->whereExists(function ($subquery) {
                $subquery->select(DB::raw(1))
                    ->from('contratoveiculo')
                    ->whereColumn('contratoveiculo.idveiculo', 'nf.idveiculo')
                    ->where('contratoveiculo.idcontrato', $this->idcontrato);
            });
        }

        if (!empty($this->idveiculo)) {
            $query->where('nf.idveiculo', $this->idveiculo);
        }

        if (!empty($this->idagencia)) {
            $query->where('nfparcela.idagencia', $this->idagencia);
        }

        // Executa a consulta Eloquent com agregação e agrupamento
        $resultados = $query->groupBy('grupoitem.idgrupoitem')
            ->selectRaw('grupoitem.idgrupoitem, COALESCE(grupoitem.grupoitem, "Indefinido") AS rotulo, SUM((nfitem.valoritem / nf.totalnf) * nfparcela.valor) AS valoritem')
            ->orderBy('valoritem', 'DESC')
            ->get();

        // Estrutura para armazenar os custos totais por grupo de item
        $custosTotaisPorGrupoItem = [];
        foreach ($resultados as $registro) {
            $custosTotaisPorGrupoItem[] = [
                'id' => $registro->idgrupoitem,
                'rotulo' => $registro->rotulo,
                'valor' => $registro->valoritem
            ];
        }

        return $custosTotaisPorGrupoItem;
    }


    public function onFilterUpdate($filters)
    {
        $this->datarange = $filters['datarange'] ?? null;
        $this->idempresa = $filters['idempresa'] ?? null;
        $this->idveiculo = $filters['idveiculo'] ?? null;
        $this->idcontrato = $filters['idcontrato'] ?? null;
        $this->idagencia = $filters['idagencia'] ?? null;
        $this->loadData();

        $this->dispatchBrowserEvent('MovimentacaoUpdated', [
            'valorTotalGrupoItem' => $this->valorTotalGrupoItem
        ]);
    }
}
