<?php

namespace App\Http\Livewire;

use Livewire\Component;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Services\DateRangeService;

class GraphMovimentacaoJsComponent extends Component
{
    public $valorTotalGrupoItem;
    public $datarange;
    public $idempresa;
    public $idveiculo;
    public $idGrupoItem;
    public $idcontrato;

    protected $listeners = ['filtersApplied' => 'onFilterUpdate'];

    public function mount()
    {
        $this->loadInitialData();
    }

    public function loadInitialData()
    {
        if (empty($this->datarange)) {
            $this->setInitialDateRange();
        }
        $this->loadData();
    }

    private function setInitialDateRange()
    {
        $dataInicio = Carbon::now()->startOfMonth()->format('d-m-Y');
        $dataFim = Carbon::now()->endOfMonth()->format('d-m-Y');
        $this->datarange = $dataInicio . ' até ' . $dataFim;
    }

    private function loadData()
    {
        $dates = DateRangeService::generateDateRange($this->datarange);
        $this->valorTotalGrupoItem = $this->calcularValorTotalPorGrupoItem($dates['dataInicio'], $dates['dataFim']);

        $this->dispatchBrowserEvent('MovimentacaoUpdated', [
            'valorTotalGrupoItem' => $this->valorTotalGrupoItem
        ]);
    }

    public function render()
    {
        return view('livewire.graph-movimentacao-js-component', [
            'valorTotalGrupoItem' => $this->valorTotalGrupoItem
        ]);
    }

    function calcularValorTotalPorGrupoItem($inicio, $fim)
    {
        // Usa Eloquent para aplicar automaticamente o TenantScope
        $query = \App\Models\Nf::join('nfparcela', 'nf.idnf', '=', 'nfparcela.idnf')
            ->join('nfitem', 'nf.idnf', '=', 'nfitem.idnf')
            ->leftJoin('grupoitem', 'nfitem.idgrupoitem', '=', 'grupoitem.idgrupoitem')
            ->where('nf.saida', '<>', 'P')
            ->where('nfparcela.status', 'CONCLUIDO')
            ->whereBetween('nfparcela.datatransacao', [$inicio, $fim]);
    
        // Adiciona condições de filtro se idempresa, idveiculo ou idcontrato forem fornecidos
        if (!empty($this->idempresa)) {
            $query->where('nf.idempresa', $this->idempresa);
        }
    
        if (!empty($this->idcontrato)) {
            $query->whereExists(function ($subquery) {
                $subquery->select(\DB::raw(1))
                    ->from('contratoveiculo')
                    ->whereColumn('contratoveiculo.idveiculo', 'nf.idveiculo')
                    ->where('contratoveiculo.idcontrato', $this->idcontrato);
            });
        }
    
        if (!empty($this->idveiculo)) {
            $query->where('nf.idveiculo', $this->idveiculo);
        }
    
        // Realiza o agrupamento e a seleção dos dados, substituindo "null" por "Indefinido"
        $resultados = $query->groupBy('grupoitem.idgrupoitem')
            ->selectRaw('grupoitem.idgrupoitem, COALESCE(grupoitem.grupoitem, "Indefinido") AS rotulo, SUM((nfitem.valoritem / nf.totalnf) * nfparcela.valor) AS valoritem')
            ->orderBy('valoritem', 'DESC')
            ->get();
    
        // Formata os resultados para exibição
        $custosTotaisPorGrupoItem = [];
        foreach ($resultados as $registro) {
            $custosTotaisPorGrupoItem[] = [
                'id' => $registro->idgrupoitem,
                'rotulo' => $registro->rotulo,
                'valor' => $registro->valoritem
            ];
        }
    
        return $custosTotaisPorGrupoItem;
    }
    


    public function onFilterUpdate($filters)
    {
        $this->datarange = $filters['datarange'] ?? null;
        $this->idempresa = $filters['idempresa'] ?? null;
        $this->idveiculo = $filters['idveiculo'] ?? null;
        $this->idcontrato = $filters['idcontrato'] ?? null;
        $this->loadData();

        $this->dispatchBrowserEvent('MovimentacaoUpdated', [
            'valorTotalGrupoItem' => $this->valorTotalGrupoItem
        ]);
    }
}
