<?php

namespace App\Http\Livewire;

use Livewire\Component;
use App\Models\Nf;
use Carbon\Carbon;
use App\Services\DateRangeService;
use Illuminate\Support\Facades\DB;

class GraphPedidoDieselJsComponent extends Component
{

    protected $listeners = ['filtersApplied' => 'onFilterUpdate'];

    public $datarange;
    public $idempresa;
    public $idveiculo;
    public $idcontrato;

    public $pedidosDieselPorMesJs;
    public $GraficoPedidosDieselPorMesJs;

    public function mount()
    {
        $this->loadInitialData();
    }

    public function loadInitialData()
    {
        if (empty($this->datarange)) {

            $this->setInitialDateRange();
        }
        $this->loadDadosPorProduto();
    }

    private function setInitialDateRange()
    {
        $dataInicio = Carbon::now()->startOfMonth()->format('d-m-Y');
        $dataFim = Carbon::now()->format('d-m-Y');
        $this->datarange = $dataInicio . ' até ' . $dataFim;
    }


    public function loadDadosPorProduto() {
        $dados = DateRangeService::generateDateRange($this->datarange);

        // Inicializa a coleção de meses
        $dadosMeses = collect();

        // Preenche os meses dentro do intervalo
        $dataInicioLoop = clone $dados['dataInicio'];
        while ($dataInicioLoop->lte($dados['dataFim'])) {
            $dadosMeses->push([
                'mes' => $dataInicioLoop->month,
                'ano' => $dataInicioLoop->year,
                'total' => 0
            ]);
            $dataInicioLoop->addMonth();
        }

        // Inicializa a consulta
        $query = Nf::pedido()
            ->selectRaw('MONTH(dataentrada) as mes, YEAR(dataentrada) as ano, COALESCE(SUM(valoritem), 0) as total')
            ->join('nfitem', 'nfitem.idnf', '=', 'nf.idnf')
            ->whereIn('idprodserv', [20, 31])
            ->whereBetween('dataentrada', [$dados['dataInicio'], $dados['dataFim']]);

        // Aplica filtros adicionais se idempresa ou idveiculo estiverem presentes
        if (!empty($this->idempresa)) {
            $query->where('nf.idempresa', $this->idempresa);
        }

        if (!empty($this->idcontrato)) {
            $query->whereExists(function ($subquery) {
                $subquery->select(DB::raw(1))
                    ->from('contratoveiculo as cv')
                    ->whereColumn('cv.idveiculo', 'nf.idveiculo')
                    ->where('cv.idcontrato', $this->idcontrato);
            });
        }

        if (!empty($this->idveiculo)) {
            $query->where('nf.idveiculo', $this->idveiculo);
        }

        $dadosConsulta = $query->groupBy('mes', 'ano')->get();

        // Preenche os dados nos meses
        $dadosMeses->transform(function ($mes) use ($dadosConsulta) {
            foreach ($dadosConsulta as $dado) {
                if ($mes['mes'] == $dado->mes && $mes['ano'] == $dado->ano) {
                    $mes['total'] = $dado->total;
                    break;
                }
            }
            return $mes;
        });

        $this->pedidosDieselPorMesJs = $dadosMeses;

        $this->GraficoPedidosDieselPorMesJs = [
            'labels' => $dadosMeses->map(fn($mes) => "{$mes['mes']}/{$mes['ano']}")->toArray(),
            'data' => $dadosMeses->map(fn($mes) => $mes['total'])->toArray()
        ];

        $this->dispatchBrowserEvent('GraficoPedidosDieselPorMesJsAtualizado', [
            'pedidosDieselPorMes' => $this->pedidosDieselPorMesJs,
            'GraficoPedidosDieselPorMesJs' => $this->GraficoPedidosDieselPorMesJs
        ]);
    }


    public function render()
    {
        return view('livewire.graph-pedido-diesel-js-component', [
            'pedidosDieselPorMes' => $this->pedidosDieselPorMesJs,
            'GraficoPedidosDieselPorMesJs' => $this->GraficoPedidosDieselPorMesJs
        ]);
    }

    public function onFilterUpdate($filters)
    {
       // dd($filters['datarange'] );
        $this->datarange = $filters['datarange'] ?? null;
        $this->idempresa = $filters['idempresa'] ?? null;
        $this->idveiculo = $filters['idveiculo'] ?? null;
        $this->idcontrato = $filters['idcontrato'] ?? null;
       // dd($this->datarange);
        $this->loadDadosPorProduto();
       // dd($this->GraficoCotacoesPorDiaJs);
        $this->dispatchBrowserEvent('GraficoPedidosDieselPorMesJsAtualizado', [
            'pedidosDieselPorMes' => $this->pedidosDieselPorMesJs,
            'GraficoPedidosDieselPorMesJs' => $this->GraficoPedidosDieselPorMesJs
        ]);


    }

}
