<?php

namespace App\Http\Livewire;

use Livewire\Component;
use App\Models\NfParcela;
use Carbon\Carbon;

class NfParcelaCreditoComponent extends Component
{
    public $creditoAtivoMesAtual;
    public $creditoAtivoMesPassado;
    public $totalAtivoCreditoMesAtual;
    public $totalAtivoCreditoMesPassado;
    public $percentualAtivoCreditoMudanca;

    public $creditoConcluidoMesAtual;
    public $creditoConcluidoMesPassado;
    public $totalConcluidoCreditoMesAtual;
    public $totalConcluidoCreditoMesPassado;
    public $percentualConcluidoCreditoMudanca;

    public function mount()
    {
        // Utilizando a biblioteca Carbon para definir a data de início do mês atual
        $inicioMesAtual = Carbon::now()->startOfMonth();

        // Utilizando a biblioteca Carbon para definir a data de final do mês atual
        $finalMesAtual = Carbon::now()->endOfMonth();

        // Utilizando Carbon para definir a data de início do mês passado
        // 'subMonthNoOverflow' evita que a data mude para o mês anterior se o mês atual tem menos dias
        $inicioMesPassado = Carbon::now()->subMonthNoOverflow()->startOfMonth();

        // Utilizando Carbon para definir a data de final do mês passado
        $finalMesPassado = Carbon::now()->subMonthNoOverflow()->endOfMonth();

        // Consultando registros de créditos ativos no intervalo do mês atual
        $this->creditoAtivoMesAtual = NfParcela::credito()->ativo()
                                        ->whereBetween('datavencimento', [$inicioMesAtual, $finalMesAtual])
                                        ->get();

        // Consultando registros de créditos ativos no intervalo do mês passado
        $this->creditoAtivoMesPassado = NfParcela::credito()->ativo()
                                            ->whereBetween('datavencimento', [$inicioMesPassado, $finalMesPassado])
                                            ->get();

        // Calculando o total de valores dos créditos ativos para o mês atual
        $this->totalAtivoCreditoMesAtual = $this->creditoAtivoMesAtual->sum('valor');

        // Calculando o total de valores dos créditos ativos para o mês passado
        $this->totalAtivoCreditoMesPassado = $this->creditoAtivoMesPassado->sum('valor');

        // Calculando a mudança percentual do total de créditos ativos do mês passado para o mês atual
        $this->percentualAtivoCreditoMudanca = $this->totalAtivoCreditoMesPassado > 0 
                                            ? (($this->totalAtivoCreditoMesAtual - $this->totalAtivoCreditoMesPassado) / $this->totalAtivoCreditoMesPassado) * 100 
                                            : 0;

        // Consultando créditos concluídos para o mês atual.
        $this->creditoConcluidoMesAtual = NfParcela::credito()->concluido()
                                            ->whereBetween('datavencimento', [$inicioMesAtual, $finalMesAtual])
                                            ->get();

        // Consultando créditos concluídos para o mês passado.
        $this->creditoConcluidoMesPassado = NfParcela::credito()->concluido()
                                            ->whereBetween('datavencimento', [$inicioMesPassado, $finalMesPassado])
                                            ->get();

        // Calculando o total de valores dos créditos concluídos para o mês atual.
        $this->totalConcluidoCreditoMesAtual = $this->creditoConcluidoMesAtual->sum('valor');

        // Calculando o total de valores dos créditos concluídos para o mês passado.
        $this->totalConcluidoCreditoMesPassado = $this->creditoConcluidoMesPassado->sum('valor');

        // Calculando a mudança percentual do total de créditos concluídos do mês passado para o mês atual.
        $this->percentualConcluidoCreditoMudanca = $this->totalConcluidoCreditoMesPassado > 0 
                    ? (($this->totalConcluidoCreditoMesAtual - $this->totalConcluidoCreditoMesPassado) / $this->totalConcluidoCreditoMesPassado) * 100 
                    : 0;
    }

    public function render()
    {
        return view('livewire.nf-parcela-credito-component', [
            'creditoAtivoMesAtual' => $this->creditoAtivoMesAtual,
            'creditoAtivoMesPassado' => $this->creditoAtivoMesPassado,
            'totalAtivoCreditoMesAtual' => $this->totalAtivoCreditoMesAtual,
            'totalAtivoCreditoMesPassado' => $this->totalAtivoCreditoMesPassado,
            'percentualAtivoCreditoMudanca' => $this->percentualAtivoCreditoMudanca,
            'creditoConcluidoMesAtual' => $this->creditoConcluidoMesAtual,
            'creditoConcluidoMesPassado' => $this->creditoConcluidoMesPassado,
            'totalConcluidoCreditoMesAtual' => $this->totalConcluidoCreditoMesAtual,
            'totalConcluidoCreditoMesPassado' => $this->totalConcluidoCreditoMesPassado,
            'percentualConcluidoCreditoMudanca' => $this->percentualConcluidoCreditoMudanca,
        ]);
    }

}


