<?php

namespace App\Http\Livewire;

use Livewire\Component;
use App\Models\NfParcela;
use Carbon\Carbon;
use App\Services\DateRangeService;
use Illuminate\Support\Facades\DB;

class NfParcelaMovimentacaoComponent extends Component
{
    public $creditoConcluidoMesAtual;
    public $totalConcluidoCreditoMesAtual;
    public $debitoConcluidoMesAtual;
    public $totalConcluidoDebitoMesAtual;
    public $totalConcluidoLucroMesAtual;

    public $saldoInicial;
    public $saldoAtual;


    public $datarange;
    public $idempresa;
    public $idveiculo;
    public $idcontrato;
    public $idagencia;

    protected $listeners = ['filtersApplied' => 'onFilterUpdate'];

    public function mount()
    {
        $this->loadInitialData();
    }

    public function loadInitialData()
    {
        $this->setInitialDateRange();
        $this->loadData();
    }

    private function setInitialDateRange()
    {
        if (empty($this->datarange)) {
            $dataInicio = Carbon::now()->startOfMonth()->format('d-m-Y');
            $dataFim = Carbon::now()->format('d-m-Y');
            $this->datarange = "$dataInicio até $dataFim";
        }
    }

    private function loadData()
    {
        $dados = DateRangeService::generateDateRange($this->datarange);

        // Obter parcelas dentro do período, independentemente de crédito ou débito
        $parcelas = NfParcela::whereBetween('datatransacao', [$dados['dataInicio'], $dados['dataFim']])
            ->orderBy('datavencimento', 'asc')
            ->get();

        // Definir saldos inicial e final baseados nas parcelas filtradas
        if ($parcelas->isNotEmpty()) {
            $this->saldoInicial = $parcelas->first()->saldoinicial;
            $this->saldoAtual = $parcelas->last()->saldoatual;
        } else {
            // Caso não haja parcelas no período, definir os saldos como 0 ou nulo
            $this->saldoInicial = 0;
            $this->saldoAtual = 0;
        }

        $this->creditoConcluidoMesAtual = $this->getConcluidoCredito($dados);
        $this->debitoConcluidoMesAtual = $this->getConcluidoDebito($dados);

        $this->totalConcluidoCreditoMesAtual = $this->creditoConcluidoMesAtual->sum('valor');
        $this->totalConcluidoDebitoMesAtual = $this->debitoConcluidoMesAtual->sum('valor');

        $this->totalConcluidoLucroMesAtual = $this->calculateLucro();
    }

    private function getConcluidoCredito($dados)
    {
        $query = NfParcela::credito()->concluido()
            ->whereBetween('datatransacao', [$dados['dataInicio'], $dados['dataFim']]);

        return $this->applyFilters($query)->get();
    }

    private function getConcluidoDebito($dados)
    {
        $query = NfParcela::debito()->concluido()
            ->whereBetween('datatransacao', [$dados['dataInicio'], $dados['dataFim']]);

        return $this->applyFilters($query)->get();
    }

    private function applyFilters($query)
    {
        if ($this->idempresa) {
            $query->where('nfparcela.idempresa', $this->idempresa);
        }

        if (!empty($this->idcontrato)) {
            $query->whereExists(function ($subquery) {
                $subquery->select(DB::raw(1))
                    ->from('contratoveiculo as cv')
                    ->whereColumn('cv.idveiculo', 'nf.idveiculo')
                    ->where('cv.idcontrato', $this->idcontrato);
            });
        }

        if ($this->idveiculo) {
            $query->where('nf.idveiculo', $this->idveiculo);
        }

        if ($this->idagencia) {
            $query->where('nfparcela.idagencia', $this->idagencia);
        }

        return $query;
    }

    private function calculateLucro()
    {
        return $this->totalConcluidoCreditoMesAtual - $this->totalConcluidoDebitoMesAtual;
    }

    public function render()
    {
        return view('livewire.nf-parcela-movimentacao-component', [
            'totalCredito' => $this->totalConcluidoCreditoMesAtual,
            'totalDebito' => $this->totalConcluidoDebitoMesAtual,
            'totalLucro' => $this->totalConcluidoLucroMesAtual,
            'saldoInicial' => $this->saldoInicial,
            'saldoAtual' => $this->saldoAtual
        ]);
    }

    public function onFilterUpdate($filters)
    {
        $this->datarange = $filters['datarange'] ?? null;
        $this->idempresa = $filters['idempresa'] ?? null;
        $this->idveiculo = $filters['idveiculo'] ?? null;
        $this->idcontrato = $filters['idcontrato'] ?? null;
        $this->idagencia = $filters['idagencia'] ?? null;

        $this->loadData();

        $this->dispatchBrowserEvent('livewire.nf-parcela-movimentacao-component', [
            'totalCredito' => $this->totalConcluidoCreditoMesAtual,
            'totalDebito' => $this->totalConcluidoDebitoMesAtual,
            'totalLucro' => $this->totalConcluidoLucroMesAtual,
            'saldoInicial' => $this->saldoInicial,
            'saldoAtual' => $this->saldoAtual
        ]);
    }
}
