<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Traits\HandlesFileUploads;
use App\Scopes\TenantScope;
use Carbon\Carbon;

class Contrato extends Model
{
    use HasFactory, HandlesFileUploads;

    const CREATED_AT = 'criadoem';
    const UPDATED_AT = 'alteradoem';

    protected $table = 'contrato';
    protected $primaryKey = 'idcontrato';
    protected $fillable = [
        'idempresa',
        'idpessoa',
        'idprodservtipo',
        'datainicio',
        'datafim',
        'dataconclusao',
        'tipo',
        'recorrencia',
        'valorrecorrencia',
        'datainiciorecorrencia',
        'datafimrecorrencia',
        'dataultimarecorrencia',
        'totalcontrato',
        'valorcontrato',
        'descontocontrato',
        'valoracordocontrato',
        'diavencimento',
        'antecedencia',
        'nome',
        'cep',
        'endereco',
        'numero',
        'complemento',
        'bairro',
        'cidade',
        'estado',
        'status',
        'criadopor',
        'alteradopor',
        'mostracontrato',
        'mostraendereco'
    ];

    protected static function booted()
    {
        static::addGlobalScope(new TenantScope);
    }

    // Método para verificar se a coluna existe na tabela
    public static function hasColumn($column)
    {
        return in_array($column, (new static)->getFillable());
    }

    public function empresa()
    {
        return $this->hasOne(Empresa::class, 'idempresa', 'idempresa');
    }

    public function pessoa()
    {
        return $this->belongsTo(Pessoa::class, 'idpessoa', 'idpessoa');
    }

    public function prodservtipo()
    {
        return $this->hasOne(TipoProdutoServico::class, 'idprodservtipo', 'idprodservtipo');
    }

    public function verificaRecorrencia()
    {
        return !is_null($this->recorrencia) && !is_null($this->datarecorrencia);
    }

    public function getProximaDataRecorrencia()
    {
        if ($this->recorrencia == 'mensal') {
            return Carbon::parse($this->datarecorrencia)->addMonth();
        } elseif ($this->recorrencia == 'anual') {
            return Carbon::parse($this->datarecorrencia)->addYear();
        }
        return null;
    }

    public function precisaGerarCobranca()
    {
        $hoje = now();

        $dataInicio = Carbon::parse($this->datainiciorecorrencia);
        $dataFim = Carbon::parse($this->datafimrecorrencia);

        if ($hoje->lt($dataInicio) || $hoje->gt($dataFim)) {
            return false;
        }

        $proxVencimento = $this->calcularProximoVencimento($hoje);

        $dataGeracao = $proxVencimento->copy()->subDays($this->antecedencia);

        return $hoje->gte($dataGeracao);
    }

    protected function calcularProximoVencimento($dataBase)
    {
        $proxVencimento = Carbon::create($dataBase->year, $dataBase->month, $this->diavencimento);

        switch ($this->recorrencia) {
            case 'mensal':
                if ($dataBase->gt($proxVencimento)) {
                    $proxVencimento->addMonth();
                }
                break;
            case 'trimestral':
                if ($dataBase->gt($proxVencimento)) {
                    $proxVencimento->addMonth(3);
                }
                break;
            case 'semestral':
                if ($dataBase->gt($proxVencimento)) {
                    $proxVencimento->addMonth(6);
                }
                break;
            case 'anual':
                if ($dataBase->gt($proxVencimento)) {
                    $proxVencimento->addYear();
                }
                break;
            default:
                throw new \Exception('Recorrencia inválida');
        }

        return $proxVencimento;
    }
}
