<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Scopes\TenantScope;

class TagRetira extends Model
{
    use HasFactory;

    // Definindo os campos personalizados de timestamps
    const CREATED_AT = 'criadoem';
    const UPDATED_AT = 'alteradoem';

    // Nome da tabela e chave primária
    protected $table = 'tagretira';
    protected $primaryKey = 'idtagretira';

    // Campos que podem ser preenchidos em massa
    protected $fillable = [
        // informações cadastrais
        'idpessoa',
        'idempresa',
        'idtag',

        // informações de periodo
        'datainicio',
        'datafim',
        'datadevolucao',

        // informarções de identificação
        'status',
        'observacao',
        'criadopor',
        'alteradopor',
    ];

    // Definindo as relações

    /**
     * Relacionamento com o model Empresa (uma TagRetira pertence a uma Empresa).
     */
    public function empresa() : BelongsTo
    {
        return $this->belongsTo(Empresa::class, 'idempresa', 'idempresa');
    }

    /**
     * Relacionamento com o model Tag (uma TagRetira pode ter um Tag associado).
     */
    public function tag() : BelongsTo
    {
        return $this->belongsTo(Tag::class, 'idtag', 'idtag');
    }

    /**
     * Relacionamento com o model Pessoa (uma TagRetira pertence a uma Pessoa).
     */
    public function pessoa() : BelongsTo
    {
        return $this->belongsTo(Pessoa::class, 'idpessoa', 'idpessoa');
    }

    // Escopo global
    protected static function booted()
    {
        static::addGlobalScope(new TenantScope);
    }

    // Método para verificar se a coluna existe na tabela
    public static function hasColumn($column)
    {
        return in_array($column, (new static)->getFillable());
    }

    public function filter($filters = false, $return = 'get', $paginate = '*')
    {
        if ($filters) {
            return $this->latest()
                ->with('pessoa', 'tag.tagTipo', 'empresa') // Carrega os relacionamentos necessários
                ->where(function ($query) use ($filters) {
                    // Aplica os filtros diretamente nas colunas da tabela
                    foreach ($filters['filter'] as $filter) {
                        if ($filter == 'idpessoa') {
                            // Filtro para o relacionamento pessoa
                            $query->orWhereHas('pessoa', function ($q) use ($filters) {
                                $q->where('nome', 'like', '%' . $filters['query'] . '%');
                            });
                        } elseif ($filter == 'idtag') {
                            // Filtro para o relacionamento tag e tagtipo
                            $query->orWhereHas('tag', function ($q) use ($filters) {
                                $q->where('tag', 'like', '%' . $filters['query'] . '%') // Busca pelo campo 'tag'
                                ->orWhereHas('tagtipo', function ($q2) use ($filters) {
                                    $q2->where('tagtipo', 'like', '%' . $filters['query'] . '%'); // Busca pelo campo 'descricao' de tagtipo
                                });
                            });
                        } else {
                            // Filtro direto nas colunas da tabela 'tagretira'
                            $query->orWhere($filter, 'like', '%' . $filters['query'] . '%');
                        }
                    }
                })
                ->$return($paginate);
        } else {
            return $this->getAll($return, $paginate);
        }
    }

}
