<?php

// App\Scopes\TenantScope.php
namespace App\Scopes;

use Illuminate\Database\Eloquent\Scope;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Log;

class TenantScope implements Scope
{
    private static $applyingScope = false;

    public function apply(Builder $builder, Model $model)
    {
        if (self::$applyingScope) {
            return;
        }

        self::$applyingScope = true;

        Log::info('Entrando no Scopo: ' . $model->getTable());

        $user = auth()->user();

        if ($user && $model->getTable() === 'users') {
            $userId = $user->id;
            $userTenantId = $user->idtenant;
            $selectedTenantId = session('selectedTenant') ? session('selectedTenant')->idtenant : $userTenantId;

            $builder->where(function ($query) use ($userId, $model, $selectedTenantId) {
                $query->where('id', $userId)
                      ->orWhere($model->getTable() . '.idtenant', $selectedTenantId);
            });
        } elseif ($model::hasColumn('idempresa')
                && !in_array($model->getTable(), ['upload', 'empresa', 'tenant', 'users', 'funcionalidades', 'planos', 'plano_funcionalidade'])) {
            Log::info('A tabela: ' . $model->getTable() . " possui a coluna idempresa.");

            if (auth()->check()) {
                Log::info('Usuário está autenticado no TenantScope.');

                $selectedTenantId = session('selectedTenant') ? session('selectedTenant')->idtenant : $user->idtenant;

                $builder->whereExists(function ($query) use ($model, $selectedTenantId) {
                    $query->select(\DB::raw(1))
                          ->from('empresa')
                          ->whereColumn('empresa.idempresa', $model->getTable() . '.idempresa')
                          ->where('empresa.idtenant', $selectedTenantId);
                });
            }
        } else {
            Log::info('A tabela: ' . $model->getTable() . " não possui a coluna idempresa.");

            if (auth()->check() && $user->idtenant && !in_array($model->getTable(), ['tenant', 'funcionalidades', 'planos', 'plano_funcionalidade'])) {
                $selectedTenantId = session('selectedTenant') ? session('selectedTenant')->idtenant : $user->idtenant;

                $builder->where($model->getTable() . '.idtenant', $selectedTenantId);

                $querySql = $builder->toSql();
                $bindings = $builder->getBindings();

                // Log da query
                Log::info('SQL Query: ' . $querySql);
                Log::info('Bindings: ' . implode(', ', $bindings));
            }
        }

        self::$applyingScope = false;
    }
}
