<?php

namespace App\Services;

use App\Models\ContratoItem;
use App\Models\Nf;
use App\Models\NfItem;
use Carbon\Carbon;
use DateTime;
use Exception;
use Illuminate\Support\Facades\DB;

class NfService
{
    public function store($contrato)
    {
        try {
            DB::beginTransaction();

            $nf = Nf::create([
                'idpessoa'       => $contrato->idpessoa,
                'idempresa'      => $contrato->idempresa,
                'idprodservtipo' => $contrato->idprodservtipo,
                'datainicio'     => $contrato->datainicio ?? null,
                'datafim'        => $contrato->datafim ?? null,
                'totalnf'        => $contrato->valoracordocontrato - $contrato->descontocontrato ?? $contrato->valor,
                'valornf'        => $contrato->valoracordocontrato ?? $contrato->valor,
                'descontonf'     => $contrato->descontocontrato,
                'dataemissao'    => now(),
                'qtdparcela'     => 1,
                'datavencimento' => $this->getVencimentoDate($contrato->diavencimento),
                'tipo'           => $contrato->tipo,
                'status'         => 'CONCLUÍDO',
                'saida'          => $contrato->tipo == 'C' ? 'V' : 'N',
            ]);


            DB::commit();

            return $nf;
        } catch (Exception $e) {
            DB::rollBack();

        }
    }

    public function storeNfItem($idContrato, $nf)
    {
        $itens = ContratoItem::where('idcontrato', $idContrato)->get();

        foreach ($itens as $item) {
            $nfItem = NfItem::create([
                'idnf'        => $nf->idnf,
                'idempresa'   => $nf->idempresa,
                'idprodserv'  => $item->idprodserv,
                'iditem'      => $item->iditem,
                'item'        => $item->item,
                'idgrupoitem' => $item->idgrupoitem,
                'idtipoitem'  => $item->idtipoitem,
                'qtd'         => $item->qtd,
                'un'          => $item->un,
                'valorun'     => $item->valorun,
                'valoritem'   => $item->valoritem,
            ]);

        }
    }

    public function handleParcels($nf)
    {
        $status = $nf->status;
        $totalnf = $nf->totalnf;
        $qtdparcela = $nf->qtdparcela;
        $datavencimento = $nf->datavencimento;

        if ($status == 'CONCLUÍDO') {
            $this->generateParcelsIfNoneExists($nf->idnf, $totalnf, $qtdparcela, $datavencimento, $nf);
        } elseif ($status == 'ATIVO') {
            $this->removeNonConcludedParcels($nf->idnf);
        }
    }

    private function generateParcelsIfNoneExists($idNf, $totalnf, $qtdparcela, $datavencimento, $compra)
    {
        $count = DB::table('nfparcela')->where('idnf', $idNf)->count();
        if ($count <= 0) {
            $parcelas = $this->generateParcelas($totalnf, $qtdparcela, $datavencimento, $compra);
            $compra->parcela()->createMany($parcelas);
        }
    }

    private function generateParcelas($totalnf, $qtdparcela, $datavencimento, $nf)
    {
        $vparcela = $totalnf / $qtdparcela;
        $parcelas = [];
        // se for apenas o numero do dia, convervter para data
        if ($datavencimento == date('d')) {
            $dataVencimento = DateTime::createFromFormat('Y-m-d', $datavencimento);
        } else {
            $dataVencimento = $datavencimento;
        }

        for ($i = 0; $i < $qtdparcela; $i++) {
            $dataVencimento->modify('+1 month');
            $dataVencimentoparcela = $this->adjustToValidDay($dataVencimento);

            $parcelas[$i] = [
                'tipo' => $nf->tipo,
                'idempresa' => $nf->idempresa,
                'parcela' => $i + 1,
                'valor' => $vparcela,
                'datavencimento' => $dataVencimentoparcela,
                'status' => 'ATIVO'
            ];
        }
        return $parcelas;
    }

    private function adjustToValidDay($dataVencimento)
    {
        if (!$this->checkDiaUtil($dataVencimento)) {
            $dataVencimento = $this->ajustarDiaUtil($dataVencimento);
        }
        return $dataVencimento;
    }

    private function removeNonConcludedParcels($idNf)
    {
        $countConcluido = DB::table('nfparcela')
            ->where('idnf', $idNf)
            ->where('status', 'CONCLUÍDO')
            ->count();

        if ($countConcluido == 0) {
            DB::table('nfparcela')
                ->where('idnf', $idNf)
                ->where('status', '<>', 'CONCLUÍDO')
                ->delete();
        }
    }

    public function convertValue($valueString)
    {
        $value = floatval(str_replace(['R$', ','], ['', '.'], $valueString));
        return $value;
    }

    public function checkDiaUtil($data) {
        $diaSemana = $data->format('N'); // 1 (segunda-feira) a 7 (domingo)
        return ($diaSemana >= 1 && $diaSemana <= 5); // Verificar se é um dia útil (segunda a sexta-feira)
    }

    public function ajustarDiaUtil($data) {
        // Verificar se o dia é sábado (6) ou domingo (7)
        if ($data->format('N') == 6) {
            // Adicionar 2 dias para ir para a próxima segunda-feira
            $data->modify('-1 days');
        } else {
            // Adicionar 1 dia para ir para a próxima segunda-feira
            $data->modify('-2 day');
        }

        return $data;
    }

    private function getVencimentoDate($diavencimento)
    {
        // Pega o mes
        $currentMonth = Carbon::now()->month;
        // Pega o ano
        $currentYear = Carbon::now()->year;

        try {
            // Gera uma data de vencimento completa
            $dataVencimento = Carbon::create($currentYear, $currentMonth, $diavencimento);

            // Verifica se a data foi criada corretamente
            if (!$dataVencimento) {
            }

            return $dataVencimento;
        } catch (Exception $e) {
            // Lida com o erro de criação da data
        }
    }
}
